"use strict";
/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
Object.defineProperty(exports, "__esModule", { value: true });
exports.derivedObservableWithWritableCache = exports.derivedObservableWithCache = exports.keepAlive = exports.wasEventTriggeredRecently = exports.debouncedObservable = exports.observableSignal = exports.observableSignalFromEvent = exports.FromEventObservable = exports.observableFromEvent = exports.waitForState = exports.observableFromPromise = exports.constObservable = void 0;
const lifecycle_1 = require("../lifecycle");
const autorun_1 = require("./autorun");
const base_1 = require("./base");
const derived_1 = require("./derived");
const logging_1 = require("./logging");
function constObservable(value) {
    return new ConstObservable(value);
}
exports.constObservable = constObservable;
class ConstObservable extends base_1.ConvenientObservable {
    constructor(value) {
        super();
        this.value = value;
    }
    get debugName() {
        return this.toString();
    }
    get() {
        return this.value;
    }
    addObserver(observer) {
        // NO OP
    }
    removeObserver(observer) {
        // NO OP
    }
    toString() {
        return `Const: ${this.value}`;
    }
}
function observableFromPromise(promise) {
    const observable = (0, base_1.observableValue)('promiseValue', {});
    promise.then((value) => {
        observable.set({ value }, undefined);
    });
    return observable;
}
exports.observableFromPromise = observableFromPromise;
function waitForState(observable, predicate) {
    return new Promise(resolve => {
        let didRun = false;
        let shouldDispose = false;
        const d = (0, autorun_1.autorun)('waitForState', reader => {
            const currentState = observable.read(reader);
            if (predicate(currentState)) {
                if (!didRun) {
                    shouldDispose = true;
                }
                else {
                    d.dispose();
                }
                resolve(currentState);
            }
        });
        didRun = true;
        if (shouldDispose) {
            d.dispose();
        }
    });
}
exports.waitForState = waitForState;
function observableFromEvent(event, getValue) {
    return new FromEventObservable(event, getValue);
}
exports.observableFromEvent = observableFromEvent;
class FromEventObservable extends base_1.BaseObservable {
    constructor(event, getValue) {
        super();
        this.event = event;
        this.getValue = getValue;
        this.hasValue = false;
        this.handleEvent = (args) => {
            var _a;
            const newValue = this.getValue(args);
            const didChange = this.value !== newValue;
            (_a = (0, logging_1.getLogger)()) === null || _a === void 0 ? void 0 : _a.handleFromEventObservableTriggered(this, { oldValue: this.value, newValue, change: undefined, didChange });
            if (didChange) {
                this.value = newValue;
                if (this.hasValue) {
                    (0, base_1.transaction)((tx) => {
                        for (const o of this.observers) {
                            tx.updateObserver(o, this);
                            o.handleChange(this, undefined);
                        }
                    }, () => {
                        const name = this.getDebugName();
                        return 'Event fired' + (name ? `: ${name}` : '');
                    });
                }
                this.hasValue = true;
            }
        };
    }
    getDebugName() {
        return (0, base_1.getFunctionName)(this.getValue);
    }
    get debugName() {
        const name = this.getDebugName();
        return 'From Event' + (name ? `: ${name}` : '');
    }
    onFirstObserverAdded() {
        this.subscription = this.event(this.handleEvent);
    }
    onLastObserverRemoved() {
        this.subscription.dispose();
        this.subscription = undefined;
        this.hasValue = false;
        this.value = undefined;
    }
    get() {
        if (this.subscription) {
            if (!this.hasValue) {
                this.handleEvent(undefined);
            }
            return this.value;
        }
        else {
            // no cache, as there are no subscribers to keep it updated
            return this.getValue(undefined);
        }
    }
}
exports.FromEventObservable = FromEventObservable;
(function (observableFromEvent) {
    observableFromEvent.Observer = FromEventObservable;
})(observableFromEvent = exports.observableFromEvent || (exports.observableFromEvent = {}));
function observableSignalFromEvent(debugName, event) {
    return new FromEventObservableSignal(debugName, event);
}
exports.observableSignalFromEvent = observableSignalFromEvent;
class FromEventObservableSignal extends base_1.BaseObservable {
    constructor(debugName, event) {
        super();
        this.debugName = debugName;
        this.event = event;
        this.handleEvent = () => {
            (0, base_1.transaction)((tx) => {
                for (const o of this.observers) {
                    tx.updateObserver(o, this);
                    o.handleChange(this, undefined);
                }
            }, () => this.debugName);
        };
    }
    onFirstObserverAdded() {
        this.subscription = this.event(this.handleEvent);
    }
    onLastObserverRemoved() {
        this.subscription.dispose();
        this.subscription = undefined;
    }
    get() {
        // NO OP
    }
}
function observableSignal(debugName) {
    return new ObservableSignal(debugName);
}
exports.observableSignal = observableSignal;
class ObservableSignal extends base_1.BaseObservable {
    constructor(debugName) {
        super();
        this.debugName = debugName;
    }
    trigger(tx) {
        if (!tx) {
            (0, base_1.transaction)(tx => {
                this.trigger(tx);
            }, () => `Trigger signal ${this.debugName}`);
            return;
        }
        for (const o of this.observers) {
            tx.updateObserver(o, this);
            o.handleChange(this, undefined);
        }
    }
    get() {
        // NO OP
    }
}
function debouncedObservable(observable, debounceMs, disposableStore) {
    const debouncedObservable = (0, base_1.observableValue)('debounced', undefined);
    let timeout = undefined;
    disposableStore.add((0, autorun_1.autorun)('debounce', reader => {
        const value = observable.read(reader);
        if (timeout) {
            clearTimeout(timeout);
        }
        timeout = setTimeout(() => {
            (0, base_1.transaction)(tx => {
                debouncedObservable.set(value, tx);
            });
        }, debounceMs);
    }));
    return debouncedObservable;
}
exports.debouncedObservable = debouncedObservable;
function wasEventTriggeredRecently(event, timeoutMs, disposableStore) {
    const observable = (0, base_1.observableValue)('triggeredRecently', false);
    let timeout = undefined;
    disposableStore.add(event(() => {
        observable.set(true, undefined);
        if (timeout) {
            clearTimeout(timeout);
        }
        timeout = setTimeout(() => {
            observable.set(false, undefined);
        }, timeoutMs);
    }));
    return observable;
}
exports.wasEventTriggeredRecently = wasEventTriggeredRecently;
/**
 * This ensures the observable is kept up-to-date.
 * This is useful when the observables `get` method is used.
*/
function keepAlive(observable) {
    const o = new KeepAliveObserver();
    observable.addObserver(o);
    return (0, lifecycle_1.toDisposable)(() => {
        observable.removeObserver(o);
    });
}
exports.keepAlive = keepAlive;
class KeepAliveObserver {
    beginUpdate(observable) {
        // NO OP
    }
    handleChange(observable, change) {
        // NO OP
    }
    endUpdate(observable) {
        // NO OP
    }
}
function derivedObservableWithCache(name, computeFn) {
    let lastValue = undefined;
    const observable = (0, derived_1.derived)(name, reader => {
        lastValue = computeFn(reader, lastValue);
        return lastValue;
    });
    return observable;
}
exports.derivedObservableWithCache = derivedObservableWithCache;
function derivedObservableWithWritableCache(name, computeFn) {
    let lastValue = undefined;
    const counter = (0, base_1.observableValue)('derivedObservableWithWritableCache.counter', 0);
    const observable = (0, derived_1.derived)(name, reader => {
        counter.read(reader);
        lastValue = computeFn(reader, lastValue);
        return lastValue;
    });
    return Object.assign(observable, {
        clearCache: (transaction) => {
            lastValue = undefined;
            counter.set(counter.get() + 1, transaction);
        },
    });
}
exports.derivedObservableWithWritableCache = derivedObservableWithWritableCache;
//# sourceMappingURL=utils.js.map