/*******************************************************************************
 * Copyright (c) 2000, 2017 IBM Corporation and others.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 *******************************************************************************/
package org.eclipse.dltk.internal.ui.text;

import java.text.BreakIterator;
import java.text.CharacterIterator;

import org.eclipse.core.runtime.Assert;

/**
 * Breaksscripttext into word starts, also stops at line start and end. No
 * direction dependency.
 *
 *
 */
public class DLTKWordIterator extends BreakIterator {

	/**
	 * The underlyingscriptbreak iterator. It returns all breaks, including before
	 * and after every whitespace.
	 */
	private ScriptBreakIterator fIterator;
	/** The current index for the stateful operations. */
	private int fIndex;

	/**
	 * Creates a new word iterator.
	 */
	public DLTKWordIterator() {
		fIterator = new ScriptBreakIterator();
		first();
	}

	@Override
	public int first() {
		fIndex = fIterator.first();
		return fIndex;
	}

	@Override
	public int last() {
		fIndex = fIterator.last();
		return fIndex;
	}

	@Override
	public int next(int n) {
		int next = 0;
		while (--n > 0 && next != DONE) {
			next = next();
		}
		return next;
	}

	@Override
	public int next() {
		fIndex = following(fIndex);
		return fIndex;
	}

	@Override
	public int previous() {
		fIndex = preceding(fIndex);
		return fIndex;
	}

	@Override
	public int preceding(int offset) {
		int first = fIterator.preceding(offset);
		if (isWhitespace(first, offset)) {
			int second = fIterator.preceding(first);
			if (second != DONE && !isDelimiter(second, first))
				return second;
		}
		return first;
	}

	@Override
	public int following(int offset) {
		int first = fIterator.following(offset);
		if (eatFollowingWhitespace(offset, first)) {
			int second = fIterator.following(first);
			if (isWhitespace(first, second))
				return second;
		}
		return first;
	}

	private boolean eatFollowingWhitespace(int offset, int exclusiveEnd) {
		if (exclusiveEnd == DONE || offset == DONE)
			return false;

		if (isWhitespace(offset, exclusiveEnd))
			return false;
		if (isDelimiter(offset, exclusiveEnd))
			return false;

		return true;
	}

	/**
	 * Returns <code>true</code> if the given sequence into the underlying text
	 * represents a delimiter, <code>false</code> otherwise.
	 *
	 * @param offset       the offset
	 * @param exclusiveEnd the end offset
	 * @return <code>true</code> if the given range is a delimiter
	 */
	private boolean isDelimiter(int offset, int exclusiveEnd) {
		if (exclusiveEnd == DONE || offset == DONE)
			return false;

		Assert.isTrue(offset >= 0);
		Assert.isTrue(exclusiveEnd <= getText().getEndIndex());
		Assert.isTrue(exclusiveEnd > offset);

		CharSequence seq = fIterator.fText;

		while (offset < exclusiveEnd) {
			char ch = seq.charAt(offset);
			if (ch != '\n' && ch != '\r')
				return false;
			offset++;
		}

		return true;
	}

	/**
	 * Returns <code>true</code> if the given sequence into the underlying text
	 * represents whitespace, but not a delimiter, <code>false</code> otherwise.
	 *
	 * @param offset       the offset
	 * @param exclusiveEnd the end offset
	 * @return <code>true</code> if the given range is whitespace
	 */
	private boolean isWhitespace(int offset, int exclusiveEnd) {
		if (exclusiveEnd == DONE || offset == DONE)
			return false;

		Assert.isTrue(offset >= 0);
		Assert.isTrue(exclusiveEnd <= getText().getEndIndex());
		Assert.isTrue(exclusiveEnd > offset);

		CharSequence seq = fIterator.fText;

		while (offset < exclusiveEnd) {
			char ch = seq.charAt(offset);
			if (!Character.isWhitespace(ch))
				return false;
			if (ch == '\n' || ch == '\r')
				return false;
			offset++;
		}

		return true;
	}

	@Override
	public int current() {
		return fIndex;
	}

	@Override
	public CharacterIterator getText() {
		return fIterator.getText();
	}

	/**
	 * Sets the text as <code>CharSequence</code>.
	 * 
	 * @param newText the new text
	 */
	public void setText(CharSequence newText) {
		fIterator.setText(newText);
		first();
	}

	@Override
	public void setText(CharacterIterator newText) {
		fIterator.setText(newText);
		first();
	}

	@Override
	public void setText(String newText) {
		setText((CharSequence) newText);
	}

}
