/**
 * Copyright (c) 2021 Contributors to the Eclipse Foundation
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 */
/*
 * generated by Xtext
 */
package org.eclipse.lsat.dispatching.teditor.validation

import activity.Activity
import activity.Move
import activity.util.EventSequenceValidator.ErrorRaiser
import dispatching.ActivityDispatching
import dispatching.Dispatch
import dispatching.DispatchGroup
import dispatching.DispatchingPackage
import dispatching.impl.AttributesMapEntryImpl
import java.math.BigDecimal
import java.util.List
import machine.IResource
import machine.Import
import machine.MachinePackage
import machine.SymbolicPosition
import machine.util.ResourcePeripheralKey
import org.eclipse.emf.common.util.URI
import org.eclipse.emf.ecore.util.EcoreUtil
import org.eclipse.lsat.common.graph.directed.editable.EdgQueries
import org.eclipse.lsat.common.queries.QueryableIterable
import org.eclipse.xtext.EcoreUtil2
import org.eclipse.xtext.validation.Check

import static extension activity.util.EventSequenceValidator.validate
import static extension dispatching.util.DispatchingUtil.getNumberRepeats

/**
 * Custom validation rules. 
 * 
 * see http://www.eclipse.org/Xtext/documentation.html#validation
 */
class DispatchingValidator extends AbstractDispatchingValidator {

    public static val GREATER_OFFSET = 'greaterOffset'
    public static val INVALID_IMPORT = 'invalidImport'
    public static val REMOVE_ITEM = 'removeItem'
    public static val SELECT_ITEM_FOR_RESOURCE = 'selectItemForResource'
    public static val REPLACE_ITEM_FOR_RESOURCE = 'replaceItemForResource'
    public static val CONVERT_YIELD = 'convertYield'

    @Check
    def checkObsoleteThroughput(ActivityDispatching ad) {
        if (ad.eIsSet(DispatchingPackage.Literals.ACTIVITY_DISPATCHING__NUMBER_OF_ITERATIONS)) {
            warning('''Througput is obsolete. Specify yield below in activities''', ad,
                DispatchingPackage.Literals.ACTIVITY_DISPATCHING__NUMBER_OF_ITERATIONS, CONVERT_YIELD)
        }
        for (var i = 0; i < ad.resourceIterations.size; i++) {
            warning('''Througput is obsolete. Specify yield below in activities''', ad,
                DispatchingPackage.Literals.ACTIVITY_DISPATCHING__RESOURCE_ITERATIONS, i, CONVERT_YIELD)
        }
    }

    @Check
    def checkDuplicateDispatchGroup(ActivityDispatching ad) {
        ad.dispatchGroups.filter[eIsSet(DispatchingPackage.Literals.DISPATCH_GROUP__NAME)].groupBy[name].values.filter [
            size > 1
        ].flatten.forEach [
            error('''Duplicate dispatch group name  '«name»' Rename one''', ad,
                DispatchingPackage.Literals.ACTIVITY_DISPATCHING__DISPATCH_GROUPS, ad.dispatchGroups.indexOf(it));

        ]
    }

    @Check
    def checkImportIsValid(Import imp) {
        try {
            val isImportUriValid = EcoreUtil2.isValidUri(imp, URI.createURI(imp.importURI))
            if (!isImportUriValid) {
                error('''The import «imp.importURI» cannot be resolved. Make sure that the name is spelled correctly.''',
                    imp, MachinePackage.Literals.IMPORT__IMPORT_URI, INVALID_IMPORT)
            }
            val isUnderstood = imp.importURI.matches(".*\\.(activity)")
            if (!isUnderstood) {
                error('''Importing «imp.importURI» is not allowed. Only 'activity' files are allowed''',
                    imp, MachinePackage.Literals.IMPORT__IMPORT_URI, INVALID_IMPORT)
            }
        } catch (IllegalArgumentException e) {
            error('''The import «imp.importURI» is not a valid URI.''', imp,
                MachinePackage.Literals.IMPORT__IMPORT_URI, INVALID_IMPORT)
        }
    }

    @Check
    def checkOffsetShouldBeGreaterThanPrevious(ActivityDispatching activityDispatching) {
        var dispatchGroups = activityDispatching.dispatchGroups
        for (var i = 1; i < dispatchGroups.size; i++) {
            if (dispatchGroups.get(i).offset < dispatchGroups.get(i - 1).offset) {
                error('Offset should be greater then previously defined offset', dispatchGroups.get(i),
                    DispatchingPackage.Literals.DISPATCH_GROUP__OFFSET, GREATER_OFFSET)
            }
        }
    }

    @Check
    def checkResourceItemsDuplicate(Dispatch dis) {
        dis.resourceItems.groupBy[resource].values.filter[size > 1].flatten.forEach [
            error('''Duplicate item specified for Resource '«resource.fqn»' Remove one''', dis,
                DispatchingPackage.Literals.DISPATCH__RESOURCE_ITEMS, dis.resourceItems.indexOf(it), REMOVE_ITEM);
        ]
    }

    @Check
    def checkMissingResourceItem(Dispatch dis) {
        val resourcesNeedingItem = dis.resourcesNeedingItem
        EcoreUtil.resolveAll(dis.eContainer);
        resourcesNeedingItem.removeAll(dis.resourceItems.map[resource])
        resourcesNeedingItem.forEach [ res |
            error('''No item specified for Resource '«res.fqn»' Press <ctrl> spacebar''', dis,
                DispatchingPackage.Literals.DISPATCH__RESOURCE_ITEMS);
        ]
    }

    @Check
    def checkResourceItemNeedsResource(Dispatch dis) {
        dis.resourceItems.forEach [ item, index |
            if (index >= dis.resourcesNeedingItem.size()) {
                error('''Unnecessary item specified  '«item.fqn»' ''', dis,
                    DispatchingPackage.Literals.DISPATCH__RESOURCE_ITEMS, index, REMOVE_ITEM);
            }

        ]
    }

    @Check
    def validateAttributesDuplicate(Dispatch dis) {
        dis.userAttributes.groupBy[key.name].values.filter[size > 1].flatten.forEach [
            error('''Duplicate item specified for Attribute '«key.name»' Remove one''', dis,
                DispatchingPackage.Literals.HAS_USER_ATTRIBUTES__USER_ATTRIBUTES, dis.userAttributes.indexOf(it),
                REMOVE_ITEM);
        ]
    }

    @Check
    def validateAttributeValue(AttributesMapEntryImpl entry) {
        if(entry.value === null) return;
        val illegalChars = entry.value.replaceAll("[\\w\\.\\-\\s]", "");
        if (!illegalChars.empty) {
            val parent = EcoreUtil2.getContainerOfType(entry, Dispatch);
            error('''Attribute value may not contain  '«illegalChars»'. only 'a-zA-Z0-9-_.<space><tab>' are allowed ''',
                parent, DispatchingPackage.Literals.HAS_USER_ATTRIBUTES__USER_ATTRIBUTES,
                parent.userAttributes.indexOf(entry), REMOVE_ITEM);

        }
    }

    @Check
    def validatePosition(ActivityDispatching activityDispatching) {
        val positionState = newLinkedHashMap
        activityDispatching.dispatchGroups.forEach [ dispatchGroup |
            dispatchGroup.dispatches.forEach [ disp, index |
                var Activity activity = disp.activity
                var locationPrerequisites = disp.indexPrerequisites(index)
                for (prerequisite : locationPrerequisites.entrySet()) {
                    var positionInfo = positionState.put(prerequisite.getKey(),
                        new PositionInfo(prerequisite.getValue(), activity))
                    if (null !== positionInfo && !positionInfo.position.equals(prerequisite.getValue())) {
                        warning(
                            '''Activity «activity.name» requires peripheral «prerequisite.key.fqn» at position «prerequisite.value.name», but activity «positionInfo.activity.name» leaves it at position «positionInfo.position.name»''',
                            disp,
                            DispatchingPackage.Literals.DISPATCH__ACTIVITY
                        )
                    }
                }
                var topologicalOrder = EdgQueries.topologicalOrdering(activity.nodes)
                if (null === topologicalOrder) {
                    error(
                        '''Location prerequisites cannot be validated for the next activity as activity «activity.name» contains a cycle''',
                        disp,
                        DispatchingPackage.Literals.DISPATCH__ACTIVITY
                    )
                    positionState.clear()
                } else {
                    for (move : QueryableIterable.from(topologicalOrder).objectsOfKind(Move).filter[positionMove]) {
                        positionState.put(new ResourcePeripheralKey(getActualResource(disp, move.getResource()),
                            move.getPeripheral()), new PositionInfo(move.getTargetPosition(), activity))
                    }
                }
            ]
        ]
    }

    /**
     * Checks the number if events raised and required match test rules:
     * <ul>
     *  <li>#receives <= #sends at any point in the sequence</li>
     *  <li>0 <= #sends - #receives <= 1, such that at most one event can be outstanding</li>
     *  <li>the number of sends and receives matches over the full sequence for the event</li>
     * </ul>
     */
    @Check
    def checkEventsRaisedAndRequired(ActivityDispatching activityDispatching) {
        val unfoldedDispatches = activityDispatching.unfoldedDispatches
        val errorRaiser = new ErrorRaiser(){
            override raise(int listIndex, String msg) {
                if(listIndex < 0){
                    error(msg, activityDispatching, DispatchingPackage.Literals.ACTIVITY_DISPATCHING__DISPATCH_GROUPS);
                }
                else {
                    val d = unfoldedDispatches.get(listIndex)
                    var dg = d.eContainer as DispatchGroup
                    error(msg, dg, DispatchingPackage.Literals.DISPATCH_GROUP__DISPATCHES, dg.dispatches.indexOf(d));
                }
            }
            
        } 
        unfoldedDispatches.map[activity].toList.validate(errorRaiser);
    }

    private static def List<Dispatch> unfoldedDispatches(ActivityDispatching activityDispatching) {
        val unfoldedDispatches = newArrayList
        // first make a sequence that mimics the eventual unfolded sequence
        activityDispatching.dispatchGroups.forEach [ dg |
            for (i : 1 .. dg.numberRepeats) {
                unfoldedDispatches.addAll(dg.dispatches)
            }
        ]
        return unfoldedDispatches
    }

    @Check
    def validateRepeatAndOffset(DispatchGroup group) {
        if (BigDecimal.ZERO.compareTo(group.offset) != 0 && group.numberRepeats > 1) {
            val msg = '''offset must be ommited or zero when repeats are specified'''
            error(msg, group, DispatchingPackage.Literals.DISPATCH_GROUP__OFFSET);
            error(msg, group, DispatchingPackage.Literals.DISPATCH_GROUP__REPEATS, 0);
        }
    }

    @Check
    def validatePhase(AttributesMapEntryImpl entry) {
        val dispatchGroup = EcoreUtil2.getContainerOfType(entry, DispatchGroup)
        if(dispatchGroup === null) return;
        if ('phase'.equalsIgnoreCase(entry.key?.name) && !dispatchGroup.name.equalsIgnoreCase(entry.value)) {
            val msg = '''phase is a reserved attribute and may not be used'''
            error(msg, entry, DispatchingPackage.Literals.ATTRIBUTES_MAP_ENTRY__KEY);
        }
    }

    @Check
    def validateIteratorName(AttributesMapEntryImpl entry) {
        val dispatchGroup = EcoreUtil2.getContainerOfType(entry, DispatchGroup)
        if(dispatchGroup === null) return;
        if (dispatchGroup.iteratorName.equalsIgnoreCase(entry.key?.name) && dispatchGroup.numberRepeats > 1) {
            val msg = '''attribute key may not be equal to activites iterator name. Change key or iterator name '''
            error(msg, entry, DispatchingPackage.Literals.ATTRIBUTES_MAP_ENTRY__KEY);
            if (dispatchGroup.eIsSet(DispatchingPackage.Literals.DISPATCH_GROUP__ITERATOR_NAME)) {
                error(msg, dispatchGroup, DispatchingPackage.Literals.DISPATCH_GROUP__ITERATOR_NAME);
            }
        }
    }

    def getResourcesNeedingItem(Dispatch dis) {
        dis.activity.resourcesNeedingItem
    }

    private final static class PositionInfo {
        final SymbolicPosition position
        final Activity activity

        package new(SymbolicPosition position, Activity activity) {
            this.position = position
            this.activity = activity
        }
    }

    def private indexPrerequisites(Dispatch disp, int index) {
        var prerequisites = disp.activity.prerequisites
        val result = newLinkedHashMap
        for (prerequisite : prerequisites) {
            var key = new ResourcePeripheralKey(getActualResource(disp, prerequisite.getResource()),
                prerequisite.getPeripheral())
            if (result.containsKey(key)) {
                error('''Only one location prerequisite per peripheral is allowed''', disp,
                    DispatchingPackage.Literals.DISPATCH__ACTIVITY);
            }
            result.put(key, prerequisite.getPosition())
        }
        return result
    }

    def private static getActualResource(Dispatch ^dispatch, IResource resource) {
        val result = ^dispatch.getResourceItems().filter(IResource).findFirst[it.resource === resource]
        return (result === null) ? resource : result;
    }

}
