﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/codebuild/CodeBuild_EXPORTS.h>
#include <aws/codebuild/model/LogsConfig.h>
#include <aws/codebuild/model/ProjectEnvironment.h>
#include <aws/codebuild/model/ProjectFileSystemLocation.h>
#include <aws/codebuild/model/ProjectSource.h>
#include <aws/codebuild/model/ProjectSourceVersion.h>
#include <aws/codebuild/model/SandboxSession.h>
#include <aws/codebuild/model/VpcConfig.h>
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace CodeBuild {
namespace Model {

/**
 * <p>Contains sandbox information.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/Sandbox">AWS
 * API Reference</a></p>
 */
class Sandbox {
 public:
  AWS_CODEBUILD_API Sandbox() = default;
  AWS_CODEBUILD_API Sandbox(Aws::Utils::Json::JsonView jsonValue);
  AWS_CODEBUILD_API Sandbox& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_CODEBUILD_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The ID of the sandbox.</p>
   */
  inline const Aws::String& GetId() const { return m_id; }
  inline bool IdHasBeenSet() const { return m_idHasBeenSet; }
  template <typename IdT = Aws::String>
  void SetId(IdT&& value) {
    m_idHasBeenSet = true;
    m_id = std::forward<IdT>(value);
  }
  template <typename IdT = Aws::String>
  Sandbox& WithId(IdT&& value) {
    SetId(std::forward<IdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the sandbox.</p>
   */
  inline const Aws::String& GetArn() const { return m_arn; }
  inline bool ArnHasBeenSet() const { return m_arnHasBeenSet; }
  template <typename ArnT = Aws::String>
  void SetArn(ArnT&& value) {
    m_arnHasBeenSet = true;
    m_arn = std::forward<ArnT>(value);
  }
  template <typename ArnT = Aws::String>
  Sandbox& WithArn(ArnT&& value) {
    SetArn(std::forward<ArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The CodeBuild project name.</p>
   */
  inline const Aws::String& GetProjectName() const { return m_projectName; }
  inline bool ProjectNameHasBeenSet() const { return m_projectNameHasBeenSet; }
  template <typename ProjectNameT = Aws::String>
  void SetProjectName(ProjectNameT&& value) {
    m_projectNameHasBeenSet = true;
    m_projectName = std::forward<ProjectNameT>(value);
  }
  template <typename ProjectNameT = Aws::String>
  Sandbox& WithProjectName(ProjectNameT&& value) {
    SetProjectName(std::forward<ProjectNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>When the sandbox process was initially requested, expressed in Unix time
   * format.</p>
   */
  inline const Aws::Utils::DateTime& GetRequestTime() const { return m_requestTime; }
  inline bool RequestTimeHasBeenSet() const { return m_requestTimeHasBeenSet; }
  template <typename RequestTimeT = Aws::Utils::DateTime>
  void SetRequestTime(RequestTimeT&& value) {
    m_requestTimeHasBeenSet = true;
    m_requestTime = std::forward<RequestTimeT>(value);
  }
  template <typename RequestTimeT = Aws::Utils::DateTime>
  Sandbox& WithRequestTime(RequestTimeT&& value) {
    SetRequestTime(std::forward<RequestTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>When the sandbox process started, expressed in Unix time format.</p>
   */
  inline const Aws::Utils::DateTime& GetStartTime() const { return m_startTime; }
  inline bool StartTimeHasBeenSet() const { return m_startTimeHasBeenSet; }
  template <typename StartTimeT = Aws::Utils::DateTime>
  void SetStartTime(StartTimeT&& value) {
    m_startTimeHasBeenSet = true;
    m_startTime = std::forward<StartTimeT>(value);
  }
  template <typename StartTimeT = Aws::Utils::DateTime>
  Sandbox& WithStartTime(StartTimeT&& value) {
    SetStartTime(std::forward<StartTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>When the sandbox process ended, expressed in Unix time format.</p>
   */
  inline const Aws::Utils::DateTime& GetEndTime() const { return m_endTime; }
  inline bool EndTimeHasBeenSet() const { return m_endTimeHasBeenSet; }
  template <typename EndTimeT = Aws::Utils::DateTime>
  void SetEndTime(EndTimeT&& value) {
    m_endTimeHasBeenSet = true;
    m_endTime = std::forward<EndTimeT>(value);
  }
  template <typename EndTimeT = Aws::Utils::DateTime>
  Sandbox& WithEndTime(EndTimeT&& value) {
    SetEndTime(std::forward<EndTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The status of the sandbox.</p>
   */
  inline const Aws::String& GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  template <typename StatusT = Aws::String>
  void SetStatus(StatusT&& value) {
    m_statusHasBeenSet = true;
    m_status = std::forward<StatusT>(value);
  }
  template <typename StatusT = Aws::String>
  Sandbox& WithStatus(StatusT&& value) {
    SetStatus(std::forward<StatusT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const ProjectSource& GetSource() const { return m_source; }
  inline bool SourceHasBeenSet() const { return m_sourceHasBeenSet; }
  template <typename SourceT = ProjectSource>
  void SetSource(SourceT&& value) {
    m_sourceHasBeenSet = true;
    m_source = std::forward<SourceT>(value);
  }
  template <typename SourceT = ProjectSource>
  Sandbox& WithSource(SourceT&& value) {
    SetSource(std::forward<SourceT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Any version identifier for the version of the sandbox to be built.</p>
   */
  inline const Aws::String& GetSourceVersion() const { return m_sourceVersion; }
  inline bool SourceVersionHasBeenSet() const { return m_sourceVersionHasBeenSet; }
  template <typename SourceVersionT = Aws::String>
  void SetSourceVersion(SourceVersionT&& value) {
    m_sourceVersionHasBeenSet = true;
    m_sourceVersion = std::forward<SourceVersionT>(value);
  }
  template <typename SourceVersionT = Aws::String>
  Sandbox& WithSourceVersion(SourceVersionT&& value) {
    SetSourceVersion(std::forward<SourceVersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> An array of <code>ProjectSource</code> objects. </p>
   */
  inline const Aws::Vector<ProjectSource>& GetSecondarySources() const { return m_secondarySources; }
  inline bool SecondarySourcesHasBeenSet() const { return m_secondarySourcesHasBeenSet; }
  template <typename SecondarySourcesT = Aws::Vector<ProjectSource>>
  void SetSecondarySources(SecondarySourcesT&& value) {
    m_secondarySourcesHasBeenSet = true;
    m_secondarySources = std::forward<SecondarySourcesT>(value);
  }
  template <typename SecondarySourcesT = Aws::Vector<ProjectSource>>
  Sandbox& WithSecondarySources(SecondarySourcesT&& value) {
    SetSecondarySources(std::forward<SecondarySourcesT>(value));
    return *this;
  }
  template <typename SecondarySourcesT = ProjectSource>
  Sandbox& AddSecondarySources(SecondarySourcesT&& value) {
    m_secondarySourcesHasBeenSet = true;
    m_secondarySources.emplace_back(std::forward<SecondarySourcesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> An array of <code>ProjectSourceVersion</code> objects.</p>
   */
  inline const Aws::Vector<ProjectSourceVersion>& GetSecondarySourceVersions() const { return m_secondarySourceVersions; }
  inline bool SecondarySourceVersionsHasBeenSet() const { return m_secondarySourceVersionsHasBeenSet; }
  template <typename SecondarySourceVersionsT = Aws::Vector<ProjectSourceVersion>>
  void SetSecondarySourceVersions(SecondarySourceVersionsT&& value) {
    m_secondarySourceVersionsHasBeenSet = true;
    m_secondarySourceVersions = std::forward<SecondarySourceVersionsT>(value);
  }
  template <typename SecondarySourceVersionsT = Aws::Vector<ProjectSourceVersion>>
  Sandbox& WithSecondarySourceVersions(SecondarySourceVersionsT&& value) {
    SetSecondarySourceVersions(std::forward<SecondarySourceVersionsT>(value));
    return *this;
  }
  template <typename SecondarySourceVersionsT = ProjectSourceVersion>
  Sandbox& AddSecondarySourceVersions(SecondarySourceVersionsT&& value) {
    m_secondarySourceVersionsHasBeenSet = true;
    m_secondarySourceVersions.emplace_back(std::forward<SecondarySourceVersionsT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const ProjectEnvironment& GetEnvironment() const { return m_environment; }
  inline bool EnvironmentHasBeenSet() const { return m_environmentHasBeenSet; }
  template <typename EnvironmentT = ProjectEnvironment>
  void SetEnvironment(EnvironmentT&& value) {
    m_environmentHasBeenSet = true;
    m_environment = std::forward<EnvironmentT>(value);
  }
  template <typename EnvironmentT = ProjectEnvironment>
  Sandbox& WithEnvironment(EnvironmentT&& value) {
    SetEnvironment(std::forward<EnvironmentT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> An array of <code>ProjectFileSystemLocation</code> objects for a CodeBuild
   * build project. A <code>ProjectFileSystemLocation</code> object specifies the
   * <code>identifier</code>, <code>location</code>, <code>mountOptions</code>,
   * <code>mountPoint</code>, and <code>type</code> of a file system created using
   * Amazon Elastic File System. </p>
   */
  inline const Aws::Vector<ProjectFileSystemLocation>& GetFileSystemLocations() const { return m_fileSystemLocations; }
  inline bool FileSystemLocationsHasBeenSet() const { return m_fileSystemLocationsHasBeenSet; }
  template <typename FileSystemLocationsT = Aws::Vector<ProjectFileSystemLocation>>
  void SetFileSystemLocations(FileSystemLocationsT&& value) {
    m_fileSystemLocationsHasBeenSet = true;
    m_fileSystemLocations = std::forward<FileSystemLocationsT>(value);
  }
  template <typename FileSystemLocationsT = Aws::Vector<ProjectFileSystemLocation>>
  Sandbox& WithFileSystemLocations(FileSystemLocationsT&& value) {
    SetFileSystemLocations(std::forward<FileSystemLocationsT>(value));
    return *this;
  }
  template <typename FileSystemLocationsT = ProjectFileSystemLocation>
  Sandbox& AddFileSystemLocations(FileSystemLocationsT&& value) {
    m_fileSystemLocationsHasBeenSet = true;
    m_fileSystemLocations.emplace_back(std::forward<FileSystemLocationsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>How long, in minutes, from 5 to 2160 (36 hours), for CodeBuild to wait before
   * timing out this sandbox if it does not get marked as completed.</p>
   */
  inline int GetTimeoutInMinutes() const { return m_timeoutInMinutes; }
  inline bool TimeoutInMinutesHasBeenSet() const { return m_timeoutInMinutesHasBeenSet; }
  inline void SetTimeoutInMinutes(int value) {
    m_timeoutInMinutesHasBeenSet = true;
    m_timeoutInMinutes = value;
  }
  inline Sandbox& WithTimeoutInMinutes(int value) {
    SetTimeoutInMinutes(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The number of minutes a sandbox is allowed to be queued before it times out.
   * </p>
   */
  inline int GetQueuedTimeoutInMinutes() const { return m_queuedTimeoutInMinutes; }
  inline bool QueuedTimeoutInMinutesHasBeenSet() const { return m_queuedTimeoutInMinutesHasBeenSet; }
  inline void SetQueuedTimeoutInMinutes(int value) {
    m_queuedTimeoutInMinutesHasBeenSet = true;
    m_queuedTimeoutInMinutes = value;
  }
  inline Sandbox& WithQueuedTimeoutInMinutes(int value) {
    SetQueuedTimeoutInMinutes(value);
    return *this;
  }
  ///@}

  ///@{

  inline const VpcConfig& GetVpcConfig() const { return m_vpcConfig; }
  inline bool VpcConfigHasBeenSet() const { return m_vpcConfigHasBeenSet; }
  template <typename VpcConfigT = VpcConfig>
  void SetVpcConfig(VpcConfigT&& value) {
    m_vpcConfigHasBeenSet = true;
    m_vpcConfig = std::forward<VpcConfigT>(value);
  }
  template <typename VpcConfigT = VpcConfig>
  Sandbox& WithVpcConfig(VpcConfigT&& value) {
    SetVpcConfig(std::forward<VpcConfigT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const LogsConfig& GetLogConfig() const { return m_logConfig; }
  inline bool LogConfigHasBeenSet() const { return m_logConfigHasBeenSet; }
  template <typename LogConfigT = LogsConfig>
  void SetLogConfig(LogConfigT&& value) {
    m_logConfigHasBeenSet = true;
    m_logConfig = std::forward<LogConfigT>(value);
  }
  template <typename LogConfigT = LogsConfig>
  Sandbox& WithLogConfig(LogConfigT&& value) {
    SetLogConfig(std::forward<LogConfigT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Key Management Service customer master key (CMK) to be used for
   * encrypting the sandbox output artifacts.</p>
   */
  inline const Aws::String& GetEncryptionKey() const { return m_encryptionKey; }
  inline bool EncryptionKeyHasBeenSet() const { return m_encryptionKeyHasBeenSet; }
  template <typename EncryptionKeyT = Aws::String>
  void SetEncryptionKey(EncryptionKeyT&& value) {
    m_encryptionKeyHasBeenSet = true;
    m_encryptionKey = std::forward<EncryptionKeyT>(value);
  }
  template <typename EncryptionKeyT = Aws::String>
  Sandbox& WithEncryptionKey(EncryptionKeyT&& value) {
    SetEncryptionKey(std::forward<EncryptionKeyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of a service role used for this sandbox.</p>
   */
  inline const Aws::String& GetServiceRole() const { return m_serviceRole; }
  inline bool ServiceRoleHasBeenSet() const { return m_serviceRoleHasBeenSet; }
  template <typename ServiceRoleT = Aws::String>
  void SetServiceRole(ServiceRoleT&& value) {
    m_serviceRoleHasBeenSet = true;
    m_serviceRole = std::forward<ServiceRoleT>(value);
  }
  template <typename ServiceRoleT = Aws::String>
  Sandbox& WithServiceRole(ServiceRoleT&& value) {
    SetServiceRole(std::forward<ServiceRoleT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The current session for the sandbox.</p>
   */
  inline const SandboxSession& GetCurrentSession() const { return m_currentSession; }
  inline bool CurrentSessionHasBeenSet() const { return m_currentSessionHasBeenSet; }
  template <typename CurrentSessionT = SandboxSession>
  void SetCurrentSession(CurrentSessionT&& value) {
    m_currentSessionHasBeenSet = true;
    m_currentSession = std::forward<CurrentSessionT>(value);
  }
  template <typename CurrentSessionT = SandboxSession>
  Sandbox& WithCurrentSession(CurrentSessionT&& value) {
    SetCurrentSession(std::forward<CurrentSessionT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_id;

  Aws::String m_arn;

  Aws::String m_projectName;

  Aws::Utils::DateTime m_requestTime{};

  Aws::Utils::DateTime m_startTime{};

  Aws::Utils::DateTime m_endTime{};

  Aws::String m_status;

  ProjectSource m_source;

  Aws::String m_sourceVersion;

  Aws::Vector<ProjectSource> m_secondarySources;

  Aws::Vector<ProjectSourceVersion> m_secondarySourceVersions;

  ProjectEnvironment m_environment;

  Aws::Vector<ProjectFileSystemLocation> m_fileSystemLocations;

  int m_timeoutInMinutes{0};

  int m_queuedTimeoutInMinutes{0};

  VpcConfig m_vpcConfig;

  LogsConfig m_logConfig;

  Aws::String m_encryptionKey;

  Aws::String m_serviceRole;

  SandboxSession m_currentSession;
  bool m_idHasBeenSet = false;
  bool m_arnHasBeenSet = false;
  bool m_projectNameHasBeenSet = false;
  bool m_requestTimeHasBeenSet = false;
  bool m_startTimeHasBeenSet = false;
  bool m_endTimeHasBeenSet = false;
  bool m_statusHasBeenSet = false;
  bool m_sourceHasBeenSet = false;
  bool m_sourceVersionHasBeenSet = false;
  bool m_secondarySourcesHasBeenSet = false;
  bool m_secondarySourceVersionsHasBeenSet = false;
  bool m_environmentHasBeenSet = false;
  bool m_fileSystemLocationsHasBeenSet = false;
  bool m_timeoutInMinutesHasBeenSet = false;
  bool m_queuedTimeoutInMinutesHasBeenSet = false;
  bool m_vpcConfigHasBeenSet = false;
  bool m_logConfigHasBeenSet = false;
  bool m_encryptionKeyHasBeenSet = false;
  bool m_serviceRoleHasBeenSet = false;
  bool m_currentSessionHasBeenSet = false;
};

}  // namespace Model
}  // namespace CodeBuild
}  // namespace Aws
