# (c) Copyright 2021-2022. CodeWeavers, Inc.

from gi.repository import GdkPixbuf
from gi.repository import Gio
from gi.repository import GLib
from gi.repository import GObject
from gi.repository import Gtk
from gi.repository import Pango

import multiprocessing
import threading
import os

import c4profilesmanager
import cxguitools
import cxproduct
import cxutils
import iconutils
import installerview
import pyop

from cxutils import cxgettext as _


class PackageViewController(GObject.Object):

    def __init__(self, parent_window):
        GObject.Object.__init__(self)

        self.xml = Gtk.Builder()
        self.xml.set_translation_domain('crossover')
        self.xml.add_from_file(cxguitools.get_ui_path('packageview'))
        self.xml.connect_signals(self)

        self.installer = None
        self.installer_selected = False

        self.parent_window = parent_window
        self.profile_list = {}
        self.profile_list_store = Gio.ListStore.new(PackageViewItem)
        self.profile_tree = {}
        self.unknown_profile = None

        self.icon_utils = iconutils.IconUtils()
        self.icon_queue = pyop.PythonOperationQueue(1)

        self.package_view_items = {}

        self._bottle_to_select = None
        self._installer_source_to_select = None
        self._profile_id_to_select = None
        self._c4pfile_to_parse = None

        self._loading = True
        self._finished_loading = False

        self.search_keys = {}
        self.search_string = ''

        self.xml.get_object('PackageFlowBox').bind_model(self.profile_list_store, self.create_package_view_item)

        self.update_profiles()

    def update_profiles(self):
        pyop.sharedOperationQueue.enqueue(UpdateProfilesOperation(self))
        pyop.sharedOperationQueue.enqueue(UpdateIconListOperation(self.icon_utils))

    def update_search_keys(self):
        self.search_keys = {}

        for appid, profile in self.profile_list.items():
            names = []
            for lang in cxutils.get_preferred_languages():
                if lang in profile.localized_names:
                    name = profile.localized_names[lang].lower()
                    if name not in names:
                        names.append(name)
            self.search_keys[appid] = '\n'.join(names)

    def filter_profile_list(self):
        content = set()
        if self.search_string:
            for appid, names in self.search_keys.items():
                if self.search_string in names:
                    content.add(appid)
        else:
            content = set(appid for (appid, profile) in self.profile_list.items() if profile.is_highlighted)

        content = sorted(content, key=lambda appid: self.profile_list[appid].name)

        # Limit the amount of items that we show to prevent
        # excessive redrawing and long waits while searching
        content = content[:128]

        return content

    def update_package_list(self):
        if not self.profile_list:
            return

        content = self.filter_profile_list()

        self.package_view_items = {}
        for appid in content:
            self.package_view_items[appid] = PackageViewItem(self.profile_list[appid], self)

        self.profile_list_store.remove_all()
        self.profile_list_store.splice(0, 0, list(self.package_view_items.values()))

        if len(content) == 0:
            self.xml.get_object('PackageScrollView').hide()
            self.xml.get_object('Loading').hide()
            self.xml.get_object('NoResults').show()
        else:
            self.xml.get_object('Loading').hide()
            self.xml.get_object('NoResults').hide()
            self.xml.get_object('PackageScrollView').show()

        if self.search_string:
            self.xml.get_object('PopularApplications').hide()
        else:
            self.xml.get_object('PopularApplications').show()

        self.update_icons(content)

    def update_icons(self, appids):
        self.icon_queue.enqueue(UpdateIconsOperation(self.icon_utils, appids, self))

    def finish_loading(self):
        if self._loading:
            self._loading = False

            if self._bottle_to_select:
                self.set_bottle(self._bottle_to_select)

            if self._installer_source_to_select:
                self.set_installer_source(self._installer_source_to_select)

            if self._profile_id_to_select:
                self.set_profile_from_id(self._profile_id_to_select)

            if self._c4pfile_to_parse:
                self.parse_c4pfile(self._c4pfile_to_parse)

        self._finished_loading = True

    def set_bottle(self, bottle_name):
        if self._finished_loading:
            self.back()

        if self._loading:
            self._bottle_to_select = bottle_name
            return

        if not self.installer:
            self.installer = installerview.InstallerViewController(None, self.parent_window)

        if bottle_name:
            self.xml.get_object('PackageViewHeader').set_label(_('Install a Windows Application into %s') % bottle_name)

        self.installer.set_bottle(bottle_name)

    def set_installer_source(self, path):
        if self._finished_loading:
            self.back()

        if self._loading:
            self._installer_source_to_select = path
            return

        if not self.installer:
            self.installer = installerview.InstallerViewController(None, self.parent_window)

        self.installer.set_custom_installer_source(path)

        if self.unknown_profile:
            self.installer.set_unknown_profile(self.unknown_profile)

        self.show_installer()

    def set_profile_from_id(self, profile_id):
        if self._finished_loading:
            self.back()

        if self._loading:
            self._profile_id_to_select = profile_id
            return

        if not self.installer:
            self.installer = installerview.InstallerViewController(None, self.parent_window)

        self.installer.set_profile(self.profile_list.get(profile_id))

        self.show_installer()

    def parse_c4pfile(self, filename):
        if self._finished_loading:
            self.back()

        if self._loading:
            self._c4pfile_to_parse = filename
            return

        if not self.installer:
            self.installer = installerview.InstallerViewController(None, self.parent_window)

        self.installer.parse_c4pfile(filename)
        self.show_installer()

    def update_back_button_callback(self, widget, visible):
        if widget.get_name() == 'BackButton':
            widget.set_visible(visible)

        if isinstance(widget, Gtk.Container):
            self.update_back_button(widget)

    def update_back_button(self, widget=None):
        if not widget:
            widget = self.parent_window

        widget.foreach(self.update_back_button_callback, self.installer_selected)

    def back(self):
        self.xml.get_object('PackageView').show()
        self.xml.get_object('InstallerView').hide()
        self.xml.get_object('PackageFlowBox').unselect_all()
        self.xml.get_object('PackageViewHeader').set_label(_('Install a Windows Application'))

        self.installer_selected = False
        self.update_back_button()

        if self.installer:
            self.xml.get_object('InstallerView').remove(self.installer.get_view())
            self.installer.destroy()
            self.installer = None

    def grab_focus(self):
        self.xml.get_object('SearchBox').grab_focus()

    def show_installer(self):
        if self.xml.get_object('PackageView').get_visible():
            self.xml.get_object('InstallerView').pack_start(self.installer.get_view(), True, True, 0)
            self.xml.get_object('InstallerView').show()
            self.xml.get_object('PackageView').hide()

            self.installer_selected = True
            self.update_back_button()

    def show_installer_for_profile(self, profile):
        if not self.installer:
            self.installer = installerview.InstallerViewController(profile, self.parent_window)
        else:
            self.installer.set_profile(profile)

        self.installer.set_icon(self.icon_utils.get_icon_path(profile.appid))
        self.show_installer()

    def load_icon(self, appid):
        if self.installer and self.installer.profile and self.installer.profile.appid == appid:
            self.installer.set_icon(self.icon_utils.get_icon_path(appid))

        item = self.package_view_items.get(appid)
        if item:
            item.load_icon(self.icon_utils.get_icon_path(appid))

    def create_package_view_item(self, item):
        widget = item.get_widget()
        self.load_icon(item.appid)
        return widget

    def get_view(self):
        return self.xml.get_object('ContentView')

    def on_PackageFlowBox_child_activated(self, _widget, child):
        profile = self.profile_list_store.get_item(child.get_index()).profile
        self.show_installer_for_profile(profile)

    def on_SearchBox_map(self, _widget):
        GLib.idle_add(self.grab_focus)

    def on_SearchBox_search_changed(self, widget):
        self.search_string = widget.get_text().lower()
        self.update_package_list()

    def on_UnlistedApplicationButton_clicked(self, _widget):
        if self.unknown_profile:
            self.show_installer_for_profile(self.unknown_profile)

        # Clear the name for the new bottle to force the user to select one
        self.installer.set_bottle('')


class PackageViewItem(GObject.Object):

    cxexe_icon = cxguitools.get_std_icon('cxexe', ('48x48', ))

    def __init__(self, profile, parent):
        GObject.Object.__init__(self)

        self.appid = profile.appid
        self.profile = profile
        self.parent = parent

        self.name = profile.name
        self.rating = profile.app_profile.medal_rating

        self.icon = None
        self.widget = None

    def get_widget(self):
        if self.widget:
            return self.widget

        if not self.rating:
            icon_names = [cxguitools.get_icon_name(('non-starred', ), Gtk.IconSize.LARGE_TOOLBAR, symbolic=True)] * 5
        elif 1 <= self.rating <= 5:
            icon_names = [cxguitools.get_icon_name(('starred', ), Gtk.IconSize.LARGE_TOOLBAR, symbolic=True)] * self.rating \
                + [cxguitools.get_icon_name(('non-starred', ), Gtk.IconSize.LARGE_TOOLBAR, symbolic=True)] * (5 - self.rating)
        else:
            icon_names = [cxguitools.get_icon_name(('dialog-warning', 'gtk-dialog-warning'), Gtk.IconSize.LARGE_TOOLBAR)]

        self.icon = Gtk.Image()

        box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=16)

        hbox = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=10)
        hbox.pack_start(self.icon, False, False, 0)

        vbox = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=6)

        name = Gtk.Label()
        name.set_markup('<b><big>' + GLib.markup_escape_text(self.name) + '</big></b>')
        name.set_halign(Gtk.Align.START)
        name.set_max_width_chars(34)
        name.set_ellipsize(Pango.EllipsizeMode.END)
        vbox.pack_start(name, False, False, 0)

        rating_box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL)
        for icon in icon_names:
            icon_widget = Gtk.Image.new_from_icon_name(icon, Gtk.IconSize.LARGE_TOOLBAR)
            rating_box.pack_start(icon_widget, False, False, 0)

        rating_description = Gtk.Label(label=self.profile.app_profile.rating_description)
        rating_description.set_property('margin-start', 6)
        rating_box.pack_start(rating_description, False, False, 0)

        vbox.pack_start(rating_box, False, False, 0)

        hbox.pack_start(vbox, True, True, 0)
        box.pack_start(hbox, True, False, 0)
        box.set_tooltip_text(self.name)

        sep = Gtk.Separator()
        box.pack_start(sep, True, False, 0)

        self.widget = Gtk.FlowBoxChild()
        self.widget.set_size_request(340, -1)
        self.widget.add(box)
        self.widget.show_all()

        self.icon.hide()

        return self.widget

    def load_icon(self, path):
        icon = self.cxexe_icon
        if path and os.path.exists(path):
            size = 48
            icon = GdkPixbuf.Pixbuf.new_from_file_at_size(path, size, size)

        self.icon.set_from_pixbuf(icon)
        self.icon.show()


class UpdateProfilesOperation(pyop.PythonOperation):

    _lock = threading.Lock()

    def __init__(self, parent):
        pyop.PythonOperation.__init__(self)
        self.parent = parent
        self.profile_list = None
        self.profile_tree = None
        self.unknown_profile = None

    def __unicode__(self):
        return '%s' % self.__class__.__name__

    def enqueued(self):
        pyop.PythonOperation.enqueued(self)

    def main(self):
        self._lock.acquire()
        profiles = c4profilesmanager.C4ProfilesSet.all_profiles()

        self.profile_list = {}
        self.profile_tree = {}
        categories = {}

        global_config = cxproduct.get_config()
        show_untested_apps = (global_config['OfficeSetup'].get('ShowUntestedApps', '1') != '0')

        for appid, profile in profiles.items():
            if not profile.is_for_current_product:
                continue

            if not show_untested_apps and not profile.is_ranked and not profile.is_unknown:
                continue

            category = self.profile_tree
            category_path = profile.app_profile.category

            if profile.is_unknown:
                self.unknown_profile = profile
                continue
            if category_path in categories:
                category = categories[category_path]
            else:
                category = self.profile_tree
                for category_name in category_path.split('/'):
                    if category_name not in category:
                        category[category_name] = {}

                    category = category[category_name]

                categories[category_path] = category

            category[appid] = profile

            self.profile_list[appid] = profile

    def finish(self):
        self._lock.release()

        self.parent.profile_list = self.profile_list
        self.parent.profile_tree = self.profile_tree
        self.parent.unknown_profile = self.unknown_profile

        self.parent.update_search_keys()
        self.parent.update_package_list()
        self.parent.finish_loading()

        pyop.PythonOperation.finish(self)


class UpdateIconListOperation(pyop.PythonOperation):

    _lock = threading.Lock()

    def __init__(self, icon_utils):
        pyop.PythonOperation.__init__(self)
        self.icon_utils = icon_utils

    def __unicode__(self):
        return '%s' % self.__class__.__name__

    def enqueued(self):
        pyop.PythonOperation.enqueued(self)

    def main(self):
        with self._lock:
            self.icon_utils.fetch_icon_list()

    def finish(self):
        pyop.PythonOperation.finish(self)


class UpdateIconsOperation(pyop.PythonOperation):

    def __init__(self, icon_utils, appids, parent):
        pyop.PythonOperation.__init__(self)
        self.icon_utils = icon_utils
        self.appids = appids
        self.parent = parent
        self.failed = False

    def __unicode__(self):
        return '%s' % self.__class__.__name__

    def enqueued(self):
        pyop.PythonOperation.enqueued(self)

    def load_icon(self, appid):
        GLib.idle_add(self.parent.load_icon, appid)

    def fetch_icon(self, appid, ret):
        ret.value = self.icon_utils.fetch_icon(appid)

    def main(self):
        for appid in self.appids:
            ret = multiprocessing.Value('b', False)
            worker = multiprocessing.Process(target=self.fetch_icon, args=(appid, ret), daemon=True)
            worker.start()
            worker.join()

            if ret.value:
                self.load_icon(appid)

    def finish(self):
        pyop.PythonOperation.finish(self)
