﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/clouddirectory/CloudDirectoryRequest.h>
#include <aws/clouddirectory/CloudDirectory_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace CloudDirectory {
namespace Model {

/**
 */
class UpgradeAppliedSchemaRequest : public CloudDirectoryRequest {
 public:
  AWS_CLOUDDIRECTORY_API UpgradeAppliedSchemaRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpgradeAppliedSchema"; }

  AWS_CLOUDDIRECTORY_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The revision of the published schema to upgrade the directory to.</p>
   */
  inline const Aws::String& GetPublishedSchemaArn() const { return m_publishedSchemaArn; }
  inline bool PublishedSchemaArnHasBeenSet() const { return m_publishedSchemaArnHasBeenSet; }
  template <typename PublishedSchemaArnT = Aws::String>
  void SetPublishedSchemaArn(PublishedSchemaArnT&& value) {
    m_publishedSchemaArnHasBeenSet = true;
    m_publishedSchemaArn = std::forward<PublishedSchemaArnT>(value);
  }
  template <typename PublishedSchemaArnT = Aws::String>
  UpgradeAppliedSchemaRequest& WithPublishedSchemaArn(PublishedSchemaArnT&& value) {
    SetPublishedSchemaArn(std::forward<PublishedSchemaArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN for the directory to which the upgraded schema will be applied.</p>
   */
  inline const Aws::String& GetDirectoryArn() const { return m_directoryArn; }
  inline bool DirectoryArnHasBeenSet() const { return m_directoryArnHasBeenSet; }
  template <typename DirectoryArnT = Aws::String>
  void SetDirectoryArn(DirectoryArnT&& value) {
    m_directoryArnHasBeenSet = true;
    m_directoryArn = std::forward<DirectoryArnT>(value);
  }
  template <typename DirectoryArnT = Aws::String>
  UpgradeAppliedSchemaRequest& WithDirectoryArn(DirectoryArnT&& value) {
    SetDirectoryArn(std::forward<DirectoryArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Used for testing whether the major version schemas are backward compatible or
   * not. If schema compatibility fails, an exception would be thrown else the call
   * would succeed but no changes will be saved. This parameter is optional.</p>
   */
  inline bool GetDryRun() const { return m_dryRun; }
  inline bool DryRunHasBeenSet() const { return m_dryRunHasBeenSet; }
  inline void SetDryRun(bool value) {
    m_dryRunHasBeenSet = true;
    m_dryRun = value;
  }
  inline UpgradeAppliedSchemaRequest& WithDryRun(bool value) {
    SetDryRun(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_publishedSchemaArn;
  bool m_publishedSchemaArnHasBeenSet = false;

  Aws::String m_directoryArn;
  bool m_directoryArnHasBeenSet = false;

  bool m_dryRun{false};
  bool m_dryRunHasBeenSet = false;
};

}  // namespace Model
}  // namespace CloudDirectory
}  // namespace Aws
