/**
 * The object holds HTTP Client session meta data, e.g. list of global variables.
 */
declare const client: HttpClient;

/**
 * The object holds information about current request.
 */
declare const request: HttpClientRequest;

// @ts-ignore
declare const console: Console;

/**
 * HTTP Client session meta data, e.g. list of global variables.
 *
 * HTTP Client session is started on IDE start and ends on IDE close,
 * values are not preserved after IDE restart.
 */
interface HttpClient {
    /**
     * Global variables defined in response handler scripts,
     * can be used as variables in HTTP Requests,
     *
     * Example:
     * ### Authorization request, receives token as an attribute of json body
     * GET https://example.com/auth
     *
     * > {% client.global.set("auth_token", response.body.token) %}
     *
     * ### Request executed with received auth_token
     * GET http://example.com/get
     * Authorization: Bearer {{auth_token}}
     */
    global: Variables;

    /**
     * Prints an array of `args` to the response handler or test stdout with separator and then terminates the line.
     * If an element of `args` is not `string`, the function converts it to string.
     * Also, it prints JS objects and arrays as their `JSON.stringify` presentation.
     */
    log(...args: any[]): void;
}

/**
 * Variables storage, can be used to define, undefine or retrieve variables.
 */
interface Variables {
    /**
     * Saves variable with name 'varName' and sets its value to 'varValue'.
     */
    set(varName: string, varValue: any): void;

    /**
     * Returns value of variable 'varName'.
     */
    get(varName: string): any;

    /**
     * Checks no variables are defined.
     */
    isEmpty(): boolean;

    /**
     * Removes variable 'varName'.
     * @param varName {string}
     */
    clear(varName: string): void;

    /**
     * Removes all variables.
     */
    clearAll(): void;

    /**
     * Global Http headers.
     */
    headers: {
        set(name: string, value: string | null): void;
    };
}

/**
 * Information about request, including variables, URL and e.t.c.
 */
interface HttpClientRequest {
    /**
     * Environment used for sending this request
     */
    environment: RequestEnvironment

    /**
     * Current request variables, which can be updated in Pre-request handler script.
     * Those variables are not shared between requests.
     */
    variables: RequestVariables

    /**
     * Header of the current request.
     */
    headers: RequestHeaders

    /**
     * Method of the current request
     */
    method: string
    
    iteration(): number

    templateValue(expressionNumber: number): string | boolean | number
}

/**
 * Object for accessing headers of the current request.
 */
interface RequestHeaders {
    /**
     * Array of all headers
     */
    all(): [RequestHeader]

    /**
     * Searches header by its name, returns null is there is not such header.
     * @param name header name for searching
     */
    findByName(name: string): RequestHeader | null
}

/**
 * Environment used for sending request.
 * Contains environment variables from http-client.env.json and http-client.private.env.json files.
 */
interface RequestEnvironment {
    /**
     * Retrieves variable value by its name. Returns null if there is no such variable.
     * @param name variable name.
     */
    get(name: string): string | null
}

/**
 * Variables for constructing current request. Can be updated in Pre-request handler script.
 */
interface RequestVariables {
    /**
     * Retrieves request variable value by its name. Returns null if there is no such variable
     * @param name request variable name
     */
    get(name: string): object | null
}

/**
 * Information about request header
 */
interface RequestHeader {
    /**
     * Header name
     */
    name: string
}

/**
 * Retrieves a value from a JSON object using a JSONPath expression.
 *
 * @param {any} obj - The JSON object to search in.
 * @param {string} expression - The JSONPath expression to use for searching.
 * @return {any} - The value found in the JSON object using the JSONPath expression.
 */
declare function jsonPath(obj: any, expression: string): any;

/**
 * Retrieves a value from a XML object using a XPath expression.
 *
 * @param {any} obj - The obj to search in with XPath expression:
 *                    - if obj is Node, Element or  Document returns result of search with expression.
 *                    - null in all other cases.
 * @param {string} expression - The XPath expression to use for searching.
 * @return {any} - The value found in the XML object using the XPath expression.
 */
declare function xpath(obj: any, expression: string): any;

/**
 * Utility for working with query parameters of the URL.
 */
// @ts-ignore
declare class URLSearchParams implements Iterable<[string, string]> {
    constructor(options?: string | object | [any])

    size: number

    append(name: string, value: string): void

    delete(name: string, value?: string): void

    get(name: string): string | null

    getAll(name: string): string[]

    has(name: string, value?: string): boolean

    set(name: string, value: string): void

    sort(): void

    toString(): string

    entries(): [[string, string]]

    forEach(callback: (key: string, value: string, searchParams?: string) => void): void

    keys(): [string]

    values(): [string]
}

interface Console {
    /**
     * Same as `log` from `HttpClient` interface
     */
    log(...args: any[]): void
}

/**
 * Stub from es2015
 */
interface String {
    /**
     * Returns a nonnegative integer Number less than 1114112 (0x110000) that is the code point
     * value of the UTF-16 encoded code point starting at the string element at position pos in
     * the String resulting from converting this object to a String.
     * If there is no element at that position, the result is undefined.
     * If a valid UTF-16 surrogate pair does not begin at pos, the result is the code unit at pos.
     */
    codePointAt(pos: number): number | undefined;

    /**
     * Returns true if searchString appears as a substring of the result of converting this
     * object to a String, at one or more positions that are
     * greater than or equal to position; otherwise, returns false.
     * @param searchString search string
     * @param position If position is undefined, 0 is assumed, so as to search all of the String.
     */
    includes(searchString: string, position?: number): boolean;

    /**
     * Returns true if the sequence of elements of searchString converted to a String is the
     * same as the corresponding elements of this object (converted to a String) starting at
     * endPosition – length(this). Otherwise returns false.
     */
    endsWith(searchString: string, endPosition?: number): boolean;

    /**
     * Returns the String value result of normalizing the string into the normalization form
     * named by form as specified in Unicode Standard Annex #15, Unicode Normalization Forms.
     * @param form Applicable values: "NFC", "NFD", "NFKC", or "NFKD", If not specified default
     * is "NFC"
     */
    normalize(form: "NFC" | "NFD" | "NFKC" | "NFKD"): string;

    /**
     * Returns the String value result of normalizing the string into the normalization form
     * named by form as specified in Unicode Standard Annex #15, Unicode Normalization Forms.
     * @param form Applicable values: "NFC", "NFD", "NFKC", or "NFKD", If not specified default
     * is "NFC"
     */
    normalize(form?: string): string;

    /**
     * Returns a String value that is made from count copies appended together. If count is 0,
     * the empty string is returned.
     * @param count number of copies to append
     */
    repeat(count: number): string;

    /**
     * Returns true if the sequence of elements of searchString converted to a String is the
     * same as the corresponding elements of this object (converted to a String) starting at
     * position. Otherwise returns false.
     */
    startsWith(searchString: string, position?: number): boolean;

    /**
     * Returns an `<a>` HTML anchor element and sets the name attribute to the text value
     * @deprecated A legacy feature for browser compatibility
     * @param name
     */
    anchor(name: string): string;

    /**
     * Returns a `<big>` HTML element
     * @deprecated A legacy feature for browser compatibility
     */
    big(): string;

    /**
     * Returns a `<blink>` HTML element
     * @deprecated A legacy feature for browser compatibility
     */
    blink(): string;

    /**
     * Returns a `<b>` HTML element
     * @deprecated A legacy feature for browser compatibility
     */
    bold(): string;

    /**
     * Returns a `<tt>` HTML element
     * @deprecated A legacy feature for browser compatibility
     */
    fixed(): string;

    /**
     * Returns a `<font>` HTML element and sets the color attribute value
     * @deprecated A legacy feature for browser compatibility
     */
    fontcolor(color: string): string;

    /**
     * Returns a `<font>` HTML element and sets the size attribute value
     * @deprecated A legacy feature for browser compatibility
     */
    fontsize(size: number): string;

    /**
     * Returns a `<font>` HTML element and sets the size attribute value
     * @deprecated A legacy feature for browser compatibility
     */
    fontsize(size: string): string;

    /**
     * Returns an `<i>` HTML element
     * @deprecated A legacy feature for browser compatibility
     */
    italics(): string;

    /**
     * Returns an `<a>` HTML element and sets the href attribute value
     * @deprecated A legacy feature for browser compatibility
     */
    link(url: string): string;

    /**
     * Returns a `<small>` HTML element
     * @deprecated A legacy feature for browser compatibility
     */
    small(): string;

    /**
     * Returns a `<strike>` HTML element
     * @deprecated A legacy feature for browser compatibility
     */
    strike(): string;

    /**
     * Returns a `<sub>` HTML element
     * @deprecated A legacy feature for browser compatibility
     */
    sub(): string;

    /**
     * Returns a `<sup>` HTML element
     * @deprecated A legacy feature for browser compatibility
     */
    sup(): string;
}

/**
 * WEB-API interface.
 */

// @ts-ignore
declare class Window {
    static atob(str: string): string;
    static btoa(bytes: string): string;
}

/**
 * Provides the ability to parse XML or HTML source code from a string into a DOM Document.
 *
 * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMParser)
 */
interface DOMParser {
    /**
     * Parses string using either the HTML or XML parser, according to type, and returns the resulting Document. type can be "text/html" (which will invoke the HTML parser), or any of "text/xml", "application/xml", "application/xhtml+xml", or "image/svg+xml" (which will invoke the XML parser).
     *
     * For the XML parser, if string cannot be parsed, then the returned Document will contain elements describing the resulting error.
     *
     * Note that script elements are not evaluated during parsing, and the resulting document's encoding will always be UTF-8.
     *
     * Values other than the above for type will cause a TypeError exception to be thrown.
     *
     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMParser/parseFromString)
     */
    parseFromString(string: string, type: DOMParserSupportedType): Document;
}

declare var DOMParser: {
    prototype: DOMParser;
    new(): DOMParser;
};

interface ChildNode extends Node {
}

declare var ChildNode: {
    prototype: ChildNode;
    new(): ChildNode;
};

/**
 * Node is an interface from which a number of DOM API object types inherit. It allows those types to be treated similarly; for example, inheriting the same set of methods, or being tested in the same way.
 *
 * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node)
 */
interface Node {
    /**
     * Returns node's node document's document base URL.
     *
     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/baseURI)
     */
    readonly baseURI: string;
    /**
     * Returns the children.
     *
     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/childNodes)
     */
    readonly childNodes: NodeListOf<ChildNode>;
    /**
     * Returns the first child.
     *
     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/firstChild)
     */
    readonly firstChild: ChildNode | null;
    /**
     * Returns true if node is connected and false otherwise.
     *
     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/isConnected)
     */
    readonly isConnected: boolean;
    /**
     * Returns the last child.
     *
     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/lastChild)
     */
    readonly lastChild: ChildNode | null;
    /**
     * Returns the next sibling.
     *
     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/nextSibling)
     */
    readonly nextSibling: ChildNode | null;
    /**
     * Returns a string appropriate for the type of node.
     *
     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/nodeName)
     */
    readonly nodeName: string;
    /**
     * Returns the type of node.
     *
     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/nodeType)
     */
    readonly nodeType: number;
    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/nodeValue) */
    nodeValue: string | null;
    /**
     * Returns the node document. Returns null for documents.
     *
     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/ownerDocument)
     */
    readonly ownerDocument: Document | null;
    /**
     * Returns the parent.
     *
     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/parentNode)
     */
    readonly parentNode: ParentNode | null;
    /**
     * Returns the previous sibling.
     *
     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/previousSibling)
     */
    readonly previousSibling: ChildNode | null;
    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/textContent) */
    textContent: string | null;

    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/appendChild) */
    appendChild<T extends Node>(node: T): T;

    /**
     * Returns a copy of node. If deep is true, the copy also includes the node's descendants.
     *
     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/cloneNode)
     */
    cloneNode(deep?: boolean): Node;

    /**
     * Returns a bitmask indicating the position of other relative to node.
     *
     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/compareDocumentPosition)
     */
    compareDocumentPosition(other: Node): number;

    /**
     * Returns true if other is an inclusive descendant of node, and false otherwise.
     *
     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/contains)
     */
    contains(other: Node | null): boolean;

    /**
     * Returns node's root.
     *
     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/getRootNode)
     */
    getRootNode(options?: GetRootNodeOptions): Node;

    /**
     * Returns whether node has children.
     *
     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/hasChildNodes)
     */
    hasChildNodes(): boolean;

    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/insertBefore) */
    insertBefore<T extends Node>(node: T, child: Node | null): T;

    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/isDefaultNamespace) */
    isDefaultNamespace(namespace: string | null): boolean;

    /**
     * Returns whether node and otherNode have the same properties.
     *
     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/isEqualNode)
     */
    isEqualNode(otherNode: Node | null): boolean;

    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/isSameNode) */
    isSameNode(otherNode: Node | null): boolean;

    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/lookupNamespaceURI) */
    lookupNamespaceURI(prefix: string | null): string | null;

    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/lookupPrefix) */
    lookupPrefix(namespace: string | null): string | null;

    /**
     * Removes empty exclusive Text nodes and concatenates the data of remaining contiguous exclusive Text nodes into the first of their nodes.
     *
     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/normalize)
     */
    normalize(): void;

    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/removeChild) */
    removeChild<T extends Node>(child: T): T;

    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/replaceChild) */
    replaceChild<T extends Node>(node: Node, child: T): T;

    /** node is an element. */
    readonly ELEMENT_NODE: 1;
    readonly ATTRIBUTE_NODE: 2;
    /** node is a Text node. */
    readonly TEXT_NODE: 3;
    /** node is a CDATASection node. */
    readonly CDATA_SECTION_NODE: 4;
    readonly ENTITY_REFERENCE_NODE: 5;
    readonly ENTITY_NODE: 6;
    /** node is a ProcessingInstruction node. */
    readonly PROCESSING_INSTRUCTION_NODE: 7;
    /** node is a Comment node. */
    readonly COMMENT_NODE: 8;
    /** node is a document. */
    readonly DOCUMENT_NODE: 9;
    /** node is a doctype. */
    readonly DOCUMENT_TYPE_NODE: 10;
    /** node is a DocumentFragment node. */
    readonly DOCUMENT_FRAGMENT_NODE: 11;
    readonly NOTATION_NODE: 12;
    /** Set when node and other are not in the same tree. */
    readonly DOCUMENT_POSITION_DISCONNECTED: 0x01;
    /** Set when other is preceding node. */
    readonly DOCUMENT_POSITION_PRECEDING: 0x02;
    /** Set when other is following node. */
    readonly DOCUMENT_POSITION_FOLLOWING: 0x04;
    /** Set when other is an ancestor of node. */
    readonly DOCUMENT_POSITION_CONTAINS: 0x08;
    /** Set when other is a descendant of node. */
    readonly DOCUMENT_POSITION_CONTAINED_BY: 0x10;
    readonly DOCUMENT_POSITION_IMPLEMENTATION_SPECIFIC: 0x20;
}

declare var Node: {
    prototype: Node;
    new(): Node;
    /** node is an element. */
    readonly ELEMENT_NODE: 1;
    readonly ATTRIBUTE_NODE: 2;
    /** node is a Text node. */
    readonly TEXT_NODE: 3;
    /** node is a CDATASection node. */
    readonly CDATA_SECTION_NODE: 4;
    readonly ENTITY_REFERENCE_NODE: 5;
    readonly ENTITY_NODE: 6;
    /** node is a ProcessingInstruction node. */
    readonly PROCESSING_INSTRUCTION_NODE: 7;
    /** node is a Comment node. */
    readonly COMMENT_NODE: 8;
    /** node is a document. */
    readonly DOCUMENT_NODE: 9;
    /** node is a doctype. */
    readonly DOCUMENT_TYPE_NODE: 10;
    /** node is a DocumentFragment node. */
    readonly DOCUMENT_FRAGMENT_NODE: 11;
    readonly NOTATION_NODE: 12;
    /** Set when node and other are not in the same tree. */
    readonly DOCUMENT_POSITION_DISCONNECTED: 0x01;
    /** Set when other is preceding node. */
    readonly DOCUMENT_POSITION_PRECEDING: 0x02;
    /** Set when other is following node. */
    readonly DOCUMENT_POSITION_FOLLOWING: 0x04;
    /** Set when other is an ancestor of node. */
    readonly DOCUMENT_POSITION_CONTAINS: 0x08;
    /** Set when other is a descendant of node. */
    readonly DOCUMENT_POSITION_CONTAINED_BY: 0x10;
    readonly DOCUMENT_POSITION_IMPLEMENTATION_SPECIFIC: 0x20;
};

/**
 * The CharacterData abstract interface represents a Node object that contains characters. This is an abstract interface, meaning there aren't any object of type CharacterData: it is implemented by other interfaces, like Text, Comment, or ProcessingInstruction which aren't abstract.
 *
 * [MDN Reference](https://developer.mozilla.org/docs/Web/API/CharacterData)
 */
interface CharacterData extends Node, ChildNode {
    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/CharacterData/data) */
    data: string;
    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/CharacterData/length) */
    readonly length: number;
    readonly ownerDocument: Document;

    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/CharacterData/appendData) */
    appendData(data: string): void;

    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/CharacterData/deleteData) */
    deleteData(offset: number, count: number): void;

    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/CharacterData/insertData) */
    insertData(offset: number, data: string): void;

    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/CharacterData/replaceData) */
    replaceData(offset: number, count: number, data: string): void;

    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/CharacterData/substringData) */
    substringData(offset: number, count: number): string;
}

declare var CharacterData: {
    prototype: CharacterData;
    new(): CharacterData;
};

/**
 * The textual content of Element or Attr. If an element has no markup within its content, it has a single child implementing Text that contains the element's text. However, if the element contains markup, it is parsed into information items and Text nodes that form its children.
 *
 * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Text)
 */
interface Text extends CharacterData {
    /**
     * Returns the combined data of all direct Text node siblings.
     *
     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Text/wholeText)
     */
    readonly wholeText: string;

    /**
     * Splits data at the given offset and returns the remainder as Text node.
     *
     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Text/splitText)
     */
    splitText(offset: number): Text;
}

// @ts-ignore
declare var Text: {
    prototype: Text;
    new(data: string): Text;
};

/**
 * A CDATA section that can be used within XML to include extended portions of unescaped text. The symbols < and & don’t need escaping as they normally do when inside a CDATA section.
 *
 * [MDN Reference](https://developer.mozilla.org/docs/Web/API/CDATASection)
 */
interface CDATASection extends Text {
}

declare var CDATASection: {
    prototype: CDATASection;
    new(): CDATASection;
};

/**
 * Element is the most general base class from which all objects in a Document inherit. It only has methods and properties common to all kinds of elements. More specific classes inherit from Element.
 *
 * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element)
 */
interface Element extends Node {
    /**
     * Returns the value of element's id content attribute. Can be set to change it.
     *
     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/id)
     */
    id: string;
    /**
     * Returns the HTML-uppercased qualified name.
     *
     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/tagName)
     */
    readonly tagName: string;
    /**
     * Returns the value of element's class content attribute. Can be set to change it.
     *
     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/className)
     */
    className: string;

    /**
     * Returns element's first attribute whose qualified name is qualifiedName, and null if there is no such attribute otherwise.
     *
     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/getAttribute)
     */
    getAttribute(qualifiedName: string): string | null;

    /**
     * Returns the qualified names of all element's attributes. Can contain duplicates.
     *
     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/getAttributeNames)
     */

    //getAttributeNames(): string[];
    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/getAttributeNode) */
    getAttributeNode(qualifiedName: string): Attr | null;

    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/getAttributeNodeNS) */
    //getAttributeNodeNS(namespace: string | null, localName: string): Attr | null;
    /**
     * Sets the value of element's attribute whose namespace is namespace and local name is localName to value.
     *
     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/setAttributeNS)
     */

    //setAttributeNS(namespace: string | null, qualifiedName: string, value: string): void;
    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/setAttributeNode) */
    setAttributeNode(attr: Attr): Attr | null;

    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/setAttributeNodeNS) */
    //setAttributeNodeNS(attr: Attr): Attr | null;
}

declare var Element: {
    prototype: Element;
    new(): Element;
};

interface ElementCreationOptions {
    is?: string;
}


/**
 * Any web page loaded in the browser and serves as an entry point into the web page's content, which is the DOM tree.
 *
 * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document)
 */
interface Document extends Element, Node {
    /**
     * Gets a reference to the root node of the document.
     *
     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/documentElement)
     */
    // @ts-ignore
    readonly documentElement: Element;

    /**
     * Creates an attribute object with a specified name.
     * @param name String that sets the attribute object's name.
     *
     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/createAttribute)
     */
    createAttribute(localName: string): Attr;

    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/createAttributeNS) */

    //createAttributeNS(namespace: string | null, qualifiedName: string): Attr;

    /**
     * Returns a CDATASection node whose data is data.
     *
     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/createCDATASection)
     */
    createCDATASection(data: string): CDATASection;

    /**
     * Creates a comment object with the specified data.
     * @param data Sets the comment object's data.
     *
     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/createComment)
     */

    //createComment(data: string): Comment;

    /**
     * Creates an instance of the element for the specified tag.
     * @param tagName The name of an element.
     *
     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/createElement)
     */
    createElement(tagName: string, options?: ElementCreationOptions): Element;

    /**
     * Returns an element with namespace namespace. Its namespace prefix will be everything before ":" (U+003E) in qualifiedName or null. Its local name will be everything after ":" (U+003E) in qualifiedName or qualifiedName.
     *
     * If localName does not match the Name production an "InvalidCharacterError" DOMException will be thrown.
     *
     * If one of the following conditions is true a "NamespaceError" DOMException will be thrown:
     *
     * localName does not match the QName production.
     * Namespace prefix is not null and namespace is the empty string.
     * Namespace prefix is "xml" and namespace is not the XML namespace.
     * qualifiedName or namespace prefix is "xmlns" and namespace is not the XMLNS namespace.
     * namespace is the XMLNS namespace and neither qualifiedName nor namespace prefix is "xmlns".
     *
     * When supplied, options's is can be used to create a customized built-in element.
     *
     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/createElementNS)
     */
    //createElementNS(namespaceURI: string | null, qualifiedName: string, options?: ElementCreationOptions): Element;
    //createElementNS(namespace: string | null, qualifiedName: string, options?: string | ElementCreationOptions): Element;
    /**
     * Creates a text string from the specified value.
     * @param data String that specifies the nodeValue property of the text node.
     *
     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/createTextNode)
     */
    createTextNode(data: string): Text;

    /**
     * Returns a reference to the first object with the specified value of the ID attribute.
     * @param elementId String that specifies the ID value.
     */
    getElementById(elementId: string): Element | null;

    /**
     * Returns a HTMLCollection of the elements in the object on which the method was invoked (a document or an element) that have all the classes given by classNames. The classNames argument is interpreted as a space-separated list of classes.
     *
     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/getElementsByClassName)
     */
    getElementsByClassName(classNames: string): NodeListOf<Element>;

    /**
     * Gets a collection of objects based on the value of the NAME or ID attribute.
     * @param elementName Gets a collection of objects based on the value of the NAME or ID attribute.
     *
     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/getElementsByName)
     */
    getElementsByName(elementName: string): NodeListOf<Element>;

    /**
     * Retrieves a collection of objects based on the specified element name.
     * @param name Specifies the name of an element.
     *
     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/getElementsByTagName)
     */
    getElementsByTagName(qualifiedName: string): NodeListOf<Element>;

    /**
     * If namespace and localName are "*" returns a HTMLCollection of all descendant elements.
     *
     * If only namespace is "*" returns a HTMLCollection of all descendant elements whose local name is localName.
     *
     * If only localName is "*" returns a HTMLCollection of all descendant elements whose namespace is namespace.
     *
     * Otherwise, returns a HTMLCollection of all descendant elements whose namespace is namespace and local name is localName.
     *
     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/getElementsByTagNameNS)
     */
    //getElementsByTagNameNS(namespace: string | null, localName: string): NodeListOf<Element>;
    /**
     * Returns the first element that is a descendant of node that matches selectors.
     */
    //querySelector<E extends Element = Element>(selectors: string): E | null;
}

declare var Document: {
    prototype: Document;
    new(): Document;
};

/**
 * Provides the serializeToString() method to construct an XML string representing a DOM tree.
 *
 * [MDN Reference](https://developer.mozilla.org/docs/Web/API/XMLSerializer)
 */
interface XMLSerializer {
    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/XMLSerializer/serializeToString) */
    serializeToString(root: Node): string;
}

declare var XMLSerializer: {
    prototype: XMLSerializer;
    new(): XMLSerializer;
};

declare function exec(command: string, options?: Record<string, any>, callback?: (error: number, stdout : string, stderr: string) => void) : object;

declare function execFile(file: string, args?: string[], options?: Record<string, any>, callback?: (error: number, stdout : string, stderr: string) => void) : object;

declare function spawn(file: string, args?: string[], options?: Record<string, any>) : object;

declare function execSync(command: string, options?: Record<string, any>) : string;

declare function execFileSync(file: string, args?: string[], options?: Record<string, any> ) : string;

declare function spawnSync(command: string, args?: string[], options?: Record<string, any>) : object;
