# Licensed to the Software Freedom Conservancy (SFC) under one
# or more contributor license agreements.  See the NOTICE file
# distributed with this work for additional information
# regarding copyright ownership.  The SFC licenses this file
# to you under the Apache License, Version 2.0 (the
# "License"); you may not use this file except in compliance
# with the License.  You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing,
# software distributed under the License is distributed on an
# "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
# KIND, either express or implied.  See the License for the
# specific language governing permissions and limitations
# under the License.

import os
import subprocess
import time
from unittest.mock import patch

import pytest

from selenium.common.exceptions import SessionNotCreatedException
from selenium.webdriver.chrome.service import Service


@pytest.mark.no_driver_after_test
def test_uses_chromedriver_logging(clean_driver, clean_options, driver_executable) -> None:
    log_file = "chromedriver.log"
    service_args = ["--append-log"]

    service1 = Service(
        log_output=log_file,
        service_args=service_args,
        executable_path=driver_executable,
    )

    service2 = Service(
        log_output=log_file,
        service_args=service_args,
        executable_path=driver_executable,
    )

    driver1 = None
    driver2 = None
    try:
        driver1 = clean_driver(options=clean_options, service=service1)
        with open(log_file) as fp:
            lines = len(fp.readlines())
        driver2 = clean_driver(options=clean_options, service=service2)
        with open(log_file) as fp:
            assert len(fp.readlines()) >= 2 * lines
    finally:
        if driver1:
            driver1.quit()
        if driver2:
            driver2.quit()
        os.remove(log_file)


@pytest.mark.no_driver_after_test
def test_log_output_as_filename(clean_driver, clean_options, driver_executable) -> None:
    log_file = "chromedriver.log"
    service = Service(log_output=log_file, executable_path=driver_executable)
    try:
        assert "--log-path=chromedriver.log" in service.service_args
        driver = clean_driver(options=clean_options, service=service)
        with open(log_file) as fp:
            assert "Starting ChromeDriver" in fp.readline()
    finally:
        driver.quit()
        os.remove(log_file)


@pytest.mark.no_driver_after_test
def test_log_output_as_file(clean_driver, clean_options, driver_executable) -> None:
    log_name = "chromedriver.log"
    log_file = open(log_name, "w", encoding="utf-8")
    service = Service(log_output=log_file, executable_path=driver_executable)
    try:
        driver = clean_driver(options=clean_options, service=service)
        time.sleep(1)
        with open(log_name) as fp:
            assert "Starting ChromeDriver" in fp.readline()
    finally:
        driver.quit()
        log_file.close()
        os.remove(log_name)


@pytest.mark.no_driver_after_test
def test_log_output_as_stdout(clean_driver, clean_options, capfd, driver_executable) -> None:
    service = Service(log_output=subprocess.STDOUT, executable_path=driver_executable)
    driver = clean_driver(options=clean_options, service=service)

    out, err = capfd.readouterr()
    assert "Starting ChromeDriver" in out
    driver.quit()


@pytest.mark.no_driver_after_test
def test_log_output_null_default(driver, capfd) -> None:
    out, err = capfd.readouterr()
    assert "Starting ChromeDriver" not in out
    driver.quit()


@pytest.mark.no_driver_after_test
def test_driver_is_stopped_if_browser_cant_start(clean_driver, clean_options, driver_executable) -> None:
    clean_options.add_argument("--user-data-dir=/no/such/location")
    service = Service(executable_path=driver_executable)
    with pytest.raises(SessionNotCreatedException):
        clean_driver(options=clean_options, service=service)
    assert not service.is_connectable()
    assert service.process.poll() is not None


@pytest.fixture
def service():
    return Service()


@pytest.mark.usefixtures("service")
class TestChromeDriverService:
    service_path = "/path/to/chromedriver"

    @pytest.fixture(autouse=True)
    def setup_and_teardown(self):
        os.environ["SE_CHROMEDRIVER"] = self.service_path
        yield
        os.environ.pop("SE_CHROMEDRIVER", None)

    def test_uses_path_from_env_variable(self, service):
        assert "chromedriver" in service.path

    def test_updates_path_after_setting_env_variable(self, service):
        service.executable_path = self.service_path  # Simulating the update
        with patch.dict("os.environ", {"SE_CHROMEDRIVER": "/foo/bar"}):
            assert "chromedriver" in service.executable_path
