from __future__ import annotations

import sys
from textwrap import dedent
from typing import TYPE_CHECKING

import pytest

from tox.config.loader.replacer import MatchError

if TYPE_CHECKING:
    from pathlib import Path

    from tox.pytest import ToxProjectCreator


def test_config_in_toml_core(tox_project: ToxProjectCreator) -> None:
    project = tox_project({
        "pyproject.toml": """
    [tool.tox]
    env_list = [ "A", "B"]

    [tool.tox.env_run_base]
    description = "Do magical things"
    commands = [
        ["python", "--version"],
        ["python", "-c", "import sys; print(sys.executable)"]
    ]
    """
    })

    outcome = project.run("c", "--core")
    outcome.assert_success()
    assert "# Exception: " not in outcome.out, outcome.out
    assert "# !!! unused: " not in outcome.out, outcome.out


def test_config_in_toml_non_default(tox_project: ToxProjectCreator) -> None:
    project = tox_project({
        "pyproject.toml": """
    [tool.tox.env.C]
    description = "Do magical things in C"
    commands = [
        ["python", "--version"]
    ]
    """
    })

    outcome = project.run("c", "-e", "C", "--core")
    outcome.assert_success()
    assert "# Exception: " not in outcome.out, outcome.out
    assert "# !!! unused: " not in outcome.out, outcome.out


def test_config_in_toml_extra(tox_project: ToxProjectCreator) -> None:
    project = tox_project({
        "pyproject.toml": """
    [tool.tox.env_run_base]
    description = "Do magical things"
    commands = [
        ["python", "--version"]
    ]
    """
    })

    outcome = project.run("c", "-e", ".".join(str(i) for i in sys.version_info[0:2]))
    outcome.assert_success()
    assert "# Exception: " not in outcome.out, outcome.out
    assert "# !!! unused: " not in outcome.out, outcome.out


def test_config_in_toml_explicit_mentioned(tox_project: ToxProjectCreator) -> None:
    project = tox_project({
        "pyproject.toml": """
    [tool.tox.env_run_base]
    description = "Do magical things"
    commands = [
        ["python", "--version"]
    ]
    """
    })

    outcome = project.run("l", "-c", "pyproject.toml")
    outcome.assert_success()
    assert "could not recognize config file pyproject.toml" not in outcome.out, outcome.out


def test_config_in_toml_replace_default(tox_project: ToxProjectCreator) -> None:
    project = tox_project({"pyproject.toml": '[tool.tox.env_run_base]\ndescription = "{missing:miss}"'})
    outcome = project.run("c", "-k", "description")
    outcome.assert_success()
    outcome.assert_out_err("[testenv:py]\ndescription = miss\n", "")


def test_config_in_toml_replace_env_name_via_env(tox_project: ToxProjectCreator) -> None:
    project = tox_project({
        "pyproject.toml": '[tool.tox.env_run_base]\ndescription = "Magic in {env:MAGICAL:{env_name}}"'
    })
    outcome = project.run("c", "-k", "description")
    outcome.assert_success()
    outcome.assert_out_err("[testenv:py]\ndescription = Magic in py\n", "")


def test_config_in_toml_replace_env_name_via_env_set(
    tox_project: ToxProjectCreator, monkeypatch: pytest.MonkeyPatch
) -> None:
    monkeypatch.setenv("MAGICAL", "YEAH")
    project = tox_project({
        "pyproject.toml": '[tool.tox.env_run_base]\ndescription = "Magic in {env:MAGICAL:{env_name}}"'
    })
    outcome = project.run("c", "-k", "description")
    outcome.assert_success()
    outcome.assert_out_err("[testenv:py]\ndescription = Magic in YEAH\n", "")


def test_config_in_toml_replace_from_env_section_absolute(tox_project: ToxProjectCreator) -> None:
    project = tox_project({
        "pyproject.toml": """
        [tool.tox.env.A]
        description = "a"
        [tool.tox.env.B]
        description = "{[tool.tox.env.A]env_name}"
        """
    })
    outcome = project.run("c", "-e", "B", "-k", "description")
    outcome.assert_success()
    outcome.assert_out_err("[testenv:B]\ndescription = A\n", "")


def test_config_in_toml_replace_from_section_absolute(tox_project: ToxProjectCreator) -> None:
    project = tox_project({
        "pyproject.toml": """
        [tool.tox.extra]
        ok = "o"
        [tool.tox.env.B]
        description = "{[tool.tox.extra]ok}"
        """
    })
    outcome = project.run("c", "-e", "B", "-k", "description")
    outcome.assert_success()
    outcome.assert_out_err("[testenv:B]\ndescription = o\n", "")


def test_config_in_toml_replace_from_section_absolute_nok(tox_project: ToxProjectCreator) -> None:
    project = tox_project({
        "pyproject.toml": """
        [tool.tox]
        extra = []
        [tool.tox.env.B]
        description = "{[tool.tox.extra.more]ok:failed}"
        """
    })
    outcome = project.run("c", "-e", "B", "-k", "description")
    outcome.assert_success()
    outcome.assert_out_err(
        "[testenv:B]\nROOT: Failed to load key more as not dictionary []\ndescription = failed\n", ""
    )


def test_config_in_toml_replace_posargs_default(tox_project: ToxProjectCreator) -> None:
    project = tox_project({
        "pyproject.toml": """
        [tool.tox.env.A]
        commands = [["python", { replace = "posargs", default = ["a", "b"], extend = true } ]]
        """
    })
    outcome = project.run("c", "-e", "A", "-k", "commands")
    outcome.assert_success()
    outcome.assert_out_err("[testenv:A]\ncommands = python a b\n", "")


def test_config_in_toml_replace_posargs_empty(tox_project: ToxProjectCreator) -> None:
    project = tox_project({
        "pyproject.toml": """
        [tool.tox.env.A]
        commands = [["python", { replace = "posargs", default = ["a", "b"], extend = true } ]]
        """
    })
    outcome = project.run("c", "-e", "A", "-k", "commands", "--")
    outcome.assert_success()
    outcome.assert_out_err("[testenv:A]\ncommands = python\n", "")


def test_config_in_toml_replace_posargs_empty_optional(tox_project: ToxProjectCreator) -> None:
    project = tox_project({
        "pyproject.toml": """
        [tool.tox.env.A]
        commands = [{ replace = "posargs", default = ["a", "b"] }, ["python"]]
        """
    })
    outcome = project.run("c", "-e", "A", "-k", "commands", "--")
    outcome.assert_success()
    outcome.assert_out_err("[testenv:A]\ncommands = python\n", "")


def test_config_in_toml_replace_posargs_set(tox_project: ToxProjectCreator) -> None:
    project = tox_project({
        "pyproject.toml": """
        [tool.tox.env.A]
        commands = [["python", { replace = "posargs", default = ["a", "b"], extend = true } ]]
        """
    })
    outcome = project.run("c", "-e", "A", "-k", "commands", "--", "c", "d")
    outcome.assert_success()
    outcome.assert_out_err("[testenv:A]\ncommands = python c d\n", "")


def test_config_in_toml_replace_env_default(tox_project: ToxProjectCreator, monkeypatch: pytest.MonkeyPatch) -> None:
    project = tox_project({
        "pyproject.toml": """
        [tool.tox.env.A]
        description = { replace = "env", name = "NAME", default = "OK" }
        """
    })
    monkeypatch.delenv("NAME", raising=False)

    outcome = project.run("c", "-e", "A", "-k", "description")
    outcome.assert_success()
    outcome.assert_out_err("[testenv:A]\ndescription = OK\n", "")


def test_config_in_toml_replace_env_set(tox_project: ToxProjectCreator, monkeypatch: pytest.MonkeyPatch) -> None:
    project = tox_project({
        "pyproject.toml": """
        [tool.tox.env.A]
        description = { replace = "env", name = "NAME", default = "OK" }
        """
    })
    monkeypatch.setenv("NAME", "OK2")

    outcome = project.run("c", "-e", "A", "-k", "description")
    outcome.assert_success()
    outcome.assert_out_err("[testenv:A]\ndescription = OK2\n", "")


def test_config_in_toml_replace_ref_of(tox_project: ToxProjectCreator, monkeypatch: pytest.MonkeyPatch) -> None:
    project = tox_project({
        "pyproject.toml": """
        [tool.tox.env_run_base]
        extras = ["A", "{env_name}"]
        [tool.tox.env.c]
        extras = [{ replace = "ref", of = ["tool", "tox", "env_run_base", "extras"], extend = true}, "B"]
        """
    })
    monkeypatch.setenv("NAME", "OK2")

    outcome = project.run("c", "-e", "c", "-k", "extras")
    outcome.assert_success()
    outcome.assert_out_err("[testenv:c]\nextras =\n  a\n  b\n  c\n", "")


def test_config_in_toml_replace_ref_env(tox_project: ToxProjectCreator, monkeypatch: pytest.MonkeyPatch) -> None:
    project = tox_project({
        "pyproject.toml": """
        [tool.tox.env.b]
        extras = ["{env_name}"]
        [tool.tox.env.a]
        extras = [{ replace = "ref", env = "b", "key" = "extras", extend = true }, "a"]
        """
    })
    monkeypatch.setenv("NAME", "OK2")

    outcome = project.run("c", "-e", "a", "-k", "extras")
    outcome.assert_success()
    outcome.assert_out_err("[testenv:a]\nextras =\n  a\n  b\n", "")


def test_config_in_toml_replace_env_circular_set(
    tox_project: ToxProjectCreator, monkeypatch: pytest.MonkeyPatch
) -> None:
    project = tox_project({
        "pyproject.toml": """
        [tool.tox.env.a]
        set_env.COVERAGE_FILE = { replace = "env", name = "COVERAGE_FILE", default = "{env_name}" }
        """
    })
    monkeypatch.setenv("COVERAGE_FILE", "OK")

    outcome = project.run("c", "-e", "a", "-k", "set_env")
    outcome.assert_success()
    assert "COVERAGE_FILE=OK" in outcome.out, outcome.out


def test_config_in_toml_replace_env_circular_unset(
    tox_project: ToxProjectCreator, monkeypatch: pytest.MonkeyPatch
) -> None:
    project = tox_project({
        "pyproject.toml": """
        [tool.tox.env.a]
        set_env.COVERAGE_FILE = { replace = "env", name = "COVERAGE_FILE", default = "{env_name}" }
        """
    })
    monkeypatch.delenv("COVERAGE_FILE", raising=False)

    outcome = project.run("c", "-e", "a", "-k", "set_env")
    outcome.assert_success()
    assert "COVERAGE_FILE=a" in outcome.out, outcome.out


def test_config_in_toml_replace_fails(tox_project: ToxProjectCreator) -> None:
    project = tox_project({
        "pyproject.toml": """
        [tool.tox.env.B]
        description = "{[tool.tox.extra]ok:d}"
        """
    })
    outcome = project.run("c", "-e", "B", "-k", "description")
    outcome.assert_success()
    outcome.assert_out_err("[testenv:B]\ndescription = d\n", "")


def test_config_in_toml_replace_from_core(tox_project: ToxProjectCreator) -> None:
    project = tox_project({
        "pyproject.toml": """
        [tool.tox.env.B]
        description = "{[tool.tox]no_package}"
        """
    })
    outcome = project.run("c", "-e", "B", "-k", "description")
    outcome.assert_success()
    outcome.assert_out_err("[testenv:B]\ndescription = False\n", "")


def test_config_in_toml_with_legacy(tox_project: ToxProjectCreator) -> None:
    project = tox_project({
        "pyproject.toml": """
        [tool.tox]
        legacy_tox_ini = '''
            [testenv]
            description=legacy
        '''
        """
    })
    outcome = project.run("c", "-e", "py", "-k", "description")
    outcome.assert_success()
    outcome.assert_out_err("[testenv:py]\ndescription = legacy\n", "")


def test_config_in_toml_bad_type_env_name(tox_project: ToxProjectCreator) -> None:
    project = tox_project({
        "pyproject.toml": """
        [tool.tox]
        env = [1]
        """
    })
    outcome = project.run("l")
    outcome.assert_failed()
    outcome.assert_out_err("ROOT: HandledError| Environment key must be string, got 1\n", "")


def test_config_in_toml_bad_type_env(tox_project: ToxProjectCreator) -> None:
    project = tox_project({
        "pyproject.toml": """
        [tool.tox]
        env = {a = 1}
        """
    })
    outcome = project.run("l")
    outcome.assert_failed()
    outcome.assert_out_err("ROOT: HandledError| tool.tox.env.a must be a table, is 'int'\n", "")


def test_config_deps(tox_project: ToxProjectCreator) -> None:
    project = tox_project({
        "pyproject.toml": """
        [tool.tox.env_run_base]
        deps = ['mypy>=1', 'ruff==1']
        """
    })
    outcome = project.run("c", "-k", "deps")
    outcome.assert_success()
    outcome.assert_out_err("[testenv:py]\ndeps =\n  mypy>=1\n  ruff==1\n", "")


def test_config_deps_req(tox_project: ToxProjectCreator) -> None:
    project = tox_project({
        "pyproject.toml": """
        [tool.tox.env_run_base]
        deps = ['-r requirements.txt']
        """
    })
    outcome = project.run("c", "-k", "deps")
    outcome.assert_success()
    outcome.assert_out_err("[testenv:py]\ndeps = -r requirements.txt\n", "")


def test_config_requires(tox_project: ToxProjectCreator) -> None:
    project = tox_project({
        "pyproject.toml": """
        [tool.tox]
        requires = ['tox>=4']
        """
    })
    outcome = project.run("c", "-k", "requires", "--core")
    outcome.assert_success()
    outcome.assert_out_err("[testenv:py]\n\n[tox]\nrequires =\n  tox>=4\n  tox\n", "")


def test_config_set_env_ref(tox_project: ToxProjectCreator) -> None:
    project = tox_project({
        "pyproject.toml": """
        [tool.tox.env_run_base]
        set_env = { A = "1", B = "2"}
        [tool.tox.env.t]
        set_env = [
            { replace = "ref", of = ["tool", "tox", "env_run_base", "set_env"]},
            { C = "3", D = "4"},
        ]
        """
    })
    outcome = project.run("c", "-et", "-k", "set_env", "--hashseed", "1")
    outcome.assert_success()
    out = (
        "[testenv:t]\n"
        "set_env =\n"
        "  A=1\n"
        "  B=2\n"
        "  C=3\n"
        "  D=4\n"
        "  PIP_DISABLE_PIP_VERSION_CHECK=1\n"
        "  PYTHONHASHSEED=1\n"
        "  PYTHONIOENCODING=utf-8\n"
    )
    outcome.assert_out_err(out, "")


def test_config_set_env_substitution_deferred(tox_project: ToxProjectCreator) -> None:
    project = tox_project({
        "tox.toml": """
        [env_run_base]
        package = "skip"
        set_env.COVERAGE_SRC = "{env_site_packages_dir}{/}mypackage"
        """
    })
    outcome = project.run("c", "-e", "py", "-k", "set_env")
    outcome.assert_success()
    assert "COVERAGE_SRC=" in outcome.out
    assert "mypackage" in outcome.out


def test_config_env_run_base_deps_reference_with_additional_deps(tox_project: ToxProjectCreator) -> None:
    project = tox_project({
        "pyproject.toml": """
        [tool.tox.env_run_base]
        deps = ["pytest>=8", "coverage>=7"]

        [tool.tox.env.test]
        deps = ["{[tool.tox.env_run_base]deps}", "pytest-xdist", "pytest-timeout"]
        """
    })
    outcome = project.run("c", "-e", "test", "-k", "deps")
    outcome.assert_success()
    out = "[testenv:test]\ndeps =\n  pytest>=8\n  coverage>=7\n  pytest-xdist\n  pytest-timeout\n"
    outcome.assert_out_err(out, "")


def test_config_env_pkg_base_deps_reference_with_additional_deps(tox_project: ToxProjectCreator) -> None:
    project = tox_project({
        "pyproject.toml": """
        [tool.tox.env_pkg_base]
        deps = ["build", "wheel"]

        [tool.tox.env.pkg]
        deps = ["{[tool.tox.env_pkg_base]deps}", "setuptools>=40"]
        """
    })
    outcome = project.run("c", "-e", "pkg", "-k", "deps")
    outcome.assert_success()
    out = "[testenv:pkg]\ndeps =\n  build\n  wheel\n  setuptools>=40\n"
    outcome.assert_out_err(out, "")


def test_config_env_base_inherit_from_arbitrary_section(tox_project: ToxProjectCreator) -> None:
    project = tox_project({
        "pyproject.toml": """
        [tool.tox]
        env_list = ["a", "b"]

        [tool.tox.env.shared]
        description = "shared config"
        skip_install = true

        [tool.tox.env.a]
        base = ["shared"]

        [tool.tox.env.b]
        base = ["shared"]
        """
    })
    outcome = project.run("c", "-e", "a,b", "-k", "description")
    outcome.assert_success()
    out = "[testenv:a]\ndescription = shared config\n\n[testenv:b]\ndescription = shared config\n"
    outcome.assert_out_err(out, "")


def test_config_in_toml_replace_glob_match(tox_project: ToxProjectCreator, tmp_path: Path) -> None:
    (tmp_path / "p" / "dist").mkdir(parents=True)
    (tmp_path / "p" / "dist" / "pkg-1.0.whl").touch()
    project = tox_project({
        "pyproject.toml": dedent("""
        [tool.tox.env.A]
        description = { replace = "glob", pattern = "dist/*.whl" }
        """),
    })
    outcome = project.run("c", "-e", "A", "-k", "description")
    outcome.assert_success()
    assert "pkg-1.0.whl" in outcome.out


def test_config_in_toml_replace_glob_no_match_default(tox_project: ToxProjectCreator) -> None:
    project = tox_project({
        "pyproject.toml": dedent("""
        [tool.tox.env.A]
        description = { replace = "glob", pattern = "dist/*.xyz", default = "none" }
        """),
    })
    outcome = project.run("c", "-e", "A", "-k", "description")
    outcome.assert_success()
    outcome.assert_out_err("[testenv:A]\ndescription = none\n", "")


def test_config_in_toml_replace_glob_extend(tox_project: ToxProjectCreator, tmp_path: Path) -> None:
    (tmp_path / "p" / "dist").mkdir(parents=True)
    (tmp_path / "p" / "dist" / "a.whl").touch()
    (tmp_path / "p" / "dist" / "b.whl").touch()
    project = tox_project({
        "pyproject.toml": dedent("""
        [tool.tox.env.A]
        commands = [["echo", { replace = "glob", pattern = "dist/*.whl", extend = true }]]
        """),
    })
    outcome = project.run("c", "-e", "A", "-k", "commands")
    outcome.assert_success()
    assert "a.whl" in outcome.out
    assert "b.whl" in outcome.out


@pytest.mark.parametrize(
    ("env_vars", "condition", "then", "else_val", "expected"),
    [
        pytest.param({"TAG": "v1"}, "env.TAG", "yes", "no", "yes", id="env_set"),
        pytest.param({}, "env.TAG", "yes", "no", "no", id="env_unset"),
        pytest.param({"TAG": ""}, "env.TAG", "yes", "no", "no", id="env_empty"),
        pytest.param({"CI": "true"}, "env.CI == 'true'", "ci", "local", "ci", id="eq_match"),
        pytest.param({"CI": "false"}, "env.CI == 'true'", "ci", "local", "local", id="eq_no_match"),
        pytest.param({"M": "s"}, "env.M != 'prod'", "dev", "prod", "dev", id="neq"),
        pytest.param({"CI": "1", "D": "1"}, "env.CI and env.D", "y", "n", "y", id="and_true"),
        pytest.param({"CI": "1"}, "env.CI and env.D", "y", "n", "n", id="and_partial"),
        pytest.param({}, "env.CI or env.L", "y", "n", "n", id="or_false"),
        pytest.param({"L": "1"}, "env.CI or env.L", "y", "n", "y", id="or_true"),
        pytest.param({}, "not env.CI", "local", "ci", "local", id="not_true"),
        pytest.param({"CI": "1"}, "not env.CI", "local", "ci", "ci", id="not_false"),
    ],
)
def test_config_in_toml_replace_if(  # noqa: PLR0913
    tox_project: ToxProjectCreator,
    monkeypatch: pytest.MonkeyPatch,
    env_vars: dict[str, str],
    condition: str,
    then: str,
    else_val: str,
    expected: str,
) -> None:
    for k in ("TAG", "CI", "D", "L", "M"):
        monkeypatch.delenv(k, raising=False)
    for k, v in env_vars.items():
        monkeypatch.setenv(k, v)
    project = tox_project({
        "pyproject.toml": dedent(f"""
        [tool.tox.env.A]
        description = {{ replace = "if", condition = "{condition}", then = "{then}", "else" = "{else_val}" }}
        """),
    })
    outcome = project.run("c", "-e", "A", "-k", "description")
    outcome.assert_success()
    outcome.assert_out_err(f"[testenv:A]\ndescription = {expected}\n", "")


def test_config_in_toml_replace_if_no_else(tox_project: ToxProjectCreator, monkeypatch: pytest.MonkeyPatch) -> None:
    monkeypatch.delenv("DEPLOY", raising=False)
    project = tox_project({
        "pyproject.toml": """\
        [tool.tox.env.A]
        description = { replace = "if", condition = "env.DEPLOY", then = "deploy mode" }
        """
    })
    outcome = project.run("c", "-e", "A", "-k", "description")
    outcome.assert_success()
    outcome.assert_out_err("[testenv:A]\ndescription = \n", "")


def test_config_in_toml_replace_if_nested_substitution(
    tox_project: ToxProjectCreator, monkeypatch: pytest.MonkeyPatch
) -> None:
    monkeypatch.setenv("DEPLOY", "yes")
    project = tox_project({
        "pyproject.toml": """\
        [tool.tox.env.A]
        description = { replace = "if", condition = "env.DEPLOY", then = "{env_name}", "else" = "none" }
        """
    })
    outcome = project.run("c", "-e", "A", "-k", "description")
    outcome.assert_success()
    outcome.assert_out_err("[testenv:A]\ndescription = A\n", "")


def test_config_in_toml_replace_if_set_env(tox_project: ToxProjectCreator, monkeypatch: pytest.MonkeyPatch) -> None:
    monkeypatch.setenv("TAG_NAME", "v2.0")
    project = tox_project({
        "pyproject.toml": """\
        [tool.tox.env.A]
        set_env.MATURITY = { replace = "if", condition = "env.TAG_NAME", then = "production", "else" = "testing" }
        """
    })
    outcome = project.run("c", "-e", "A", "-k", "set_env")
    outcome.assert_success()
    assert "MATURITY=production" in outcome.out


def test_config_in_toml_replace_if_extend(tox_project: ToxProjectCreator, monkeypatch: pytest.MonkeyPatch) -> None:
    monkeypatch.setenv("V", "1")
    toml = """\
    [tool.tox.env.A]
    commands = [["echo", { replace = "if", condition = "env.V", then = ["-v"], "else" = ["-q"], extend = true }]]
    """
    project = tox_project({"pyproject.toml": toml})
    outcome = project.run("c", "-e", "A", "-k", "commands")
    outcome.assert_success()
    assert "-v" in outcome.out


@pytest.mark.parametrize(
    ("condition_toml", "error_match"),
    [
        pytest.param(
            'then = "yes", "else" = "no"',
            "No condition was supplied in if replacement",
            id="missing_condition",
        ),
        pytest.param(
            'condition = "env.CI", "else" = "no"',
            "No 'then' value was supplied in if replacement",
            id="missing_then",
        ),
        pytest.param(
            'condition = "env.CI ===", then = "yes"',
            r"Invalid condition expression: env\.CI ===",
            id="invalid_syntax",
        ),
        pytest.param(
            'condition = "env.CI > env.X", then = "yes"',
            r"Unsupported comparison operator in condition: env\.CI > env\.X",
            id="unsupported_compare",
        ),
        pytest.param(
            'condition = "1 + 2", then = "yes"',
            r"Unsupported expression in condition: ",
            id="unsupported_expr",
        ),
    ],
)
def test_config_in_toml_replace_if_error(tox_project: ToxProjectCreator, condition_toml: str, error_match: str) -> None:
    project = tox_project({
        "pyproject.toml": dedent(f"""
        [tool.tox.env.A]
        description = {{ replace = "if", {condition_toml} }}
        """),
    })
    with pytest.raises(MatchError, match=error_match):
        project.run("c", "-e", "A", "-k", "description")
