// SPDX-License-Identifier: ISC
// https://codeberg.org/amonakov/elf-stubber
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>

#include <link.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/mman.h>
#include <sys/stat.h>

int main(int argc, char **argv)
{
	if (argc != 2) {
		fprintf(stderr, "Usage: %s <elf>\n", argv[0]);
		return 1;
	}
	int fd;
	if ((fd = open(argv[1], O_RDONLY)) < 0) {
		perror(argv[1]);
		return 1;
	}
	struct stat st;
	if (fstat(fd, &st) < 0) {
		perror("stat");
		return 1;
	}
	char *map;
	if ((map = mmap(0, st.st_size, PROT_READ, MAP_SHARED, fd, 0)) == MAP_FAILED) {
		perror("mmap");
		return 1;
	}
	ElfW(Ehdr) *ehdr = (void *)map;
	ElfW(Shdr) *shdr = (void *)(map + ehdr->e_shoff);
	assert(ehdr->e_shentsize == sizeof *shdr);
	int i;
	for (i = ehdr->e_shnum; i; shdr++, i--) {
		if (shdr->sh_type == SHT_DYNSYM)
			break;
	}
	if (!i) {
		fputs("section .dynsym not found", stderr);
		return 1;
	}
	ElfW(Sym) *sym = (void *)(map + shdr->sh_offset);
	assert(shdr->sh_entsize == sizeof *sym);
	assert(shdr->sh_size % sizeof *sym == 0);
	ElfW(Sym) *end = (void *)(map + shdr->sh_offset + shdr->sh_size);
	ElfW(Shdr) *strtab = (void *)(map + ehdr->e_shoff);
	strtab += shdr->sh_link;
	assert(strtab->sh_type == SHT_STRTAB);
	char *str = map + strtab->sh_offset;
	for (sym++; sym != end; sym++) {
		int type = sym->st_info & 15;
		int bind = sym->st_info >> 4;
		int vis  = sym->st_other & 3;
		assert(bind == STB_GLOBAL || bind == STB_WEAK);
		assert(vis == STV_DEFAULT || vis == STV_PROTECTED);
		if (!sym->st_value) continue;
		switch (type) {
		case STT_NOTYPE:
		case STT_SECTION:
		case STT_FILE:
			break;
		case STT_OBJECT:
		case STT_TLS:
			fprintf(stderr, "TODO object %s [size %zd]\n",
			       str + sym->st_name, (size_t)sym->st_size);
			break;
		case STT_FUNC:
		case STT_GNU_IFUNC:
			puts(str + sym->st_name);
			break;
		default:
			fprintf(stderr, "TODO symbol type %d\n", type);
		}
	}

	return 0;
}
