"""Load from a pyproject.toml file, native format."""

from __future__ import annotations

import sys
from collections.abc import Iterator, Mapping
from itertools import product
from typing import TYPE_CHECKING, Any, Final, cast

from tox.config.loader.section import Section
from tox.config.loader.toml import TomlLoader
from tox.config.loader.toml._product import expand_factor_group
from tox.config.types import MissingRequiredConfigKeyError
from tox.report import HandledError

from .api import Source

if sys.version_info >= (3, 11):  # pragma: no cover (py311+)
    import tomllib
else:  # pragma: no cover (py311+)
    import tomli as tomllib

if TYPE_CHECKING:
    from collections.abc import Iterable
    from pathlib import Path

    from tox.config.loader.api import Loader, OverrideMap
    from tox.config.sets import CoreConfigSet


class TomlSection(Section):
    SEP: str = "."
    PREFIX: tuple[str, ...]
    ENV: Final[str] = "env"
    ENV_BASE: Final[str] = "env_base"
    RUN_ENV_BASE: Final[str] = "env_run_base"
    PKG_ENV_BASE: Final[str] = "env_pkg_base"

    @classmethod
    def test_env(cls, name: str) -> TomlSection:
        return cls(cls.env_prefix(), name)

    @classmethod
    def env_prefix(cls) -> str:
        return cls.SEP.join((*cls.PREFIX, cls.ENV))

    @classmethod
    def core_prefix(cls) -> str:
        return cls.SEP.join(cls.PREFIX)

    @classmethod
    def package_env_base(cls) -> str:
        return cls.SEP.join((*cls.PREFIX, cls.PKG_ENV_BASE))

    @classmethod
    def run_env_base(cls) -> str:
        return cls.SEP.join((*cls.PREFIX, cls.RUN_ENV_BASE))

    @classmethod
    def env_base_prefix(cls) -> str:
        return cls.SEP.join((*cls.PREFIX, cls.ENV_BASE))

    @classmethod
    def env_base(cls, name: str) -> TomlSection:
        return cls(cls.env_base_prefix(), name)

    @property
    def keys(self) -> Iterable[str]:
        # Build keys from prefix + name directly, preserving dots in names (e.g. env "py3.11").
        prefix, name = self._prefix, self._name
        if prefix is None and not name:
            return []
        parts: list[str] = prefix.split(self.SEP) if prefix else []
        if self.PREFIX and len(parts) >= len(self.PREFIX) and tuple(parts[: len(self.PREFIX)]) == self.PREFIX:
            parts = parts[len(self.PREFIX) :]  # strip global PREFIX (e.g. ("tool", "tox"))
        if name:
            parts.append(name)
        return parts


class TomlPyProjectSection(TomlSection):
    PREFIX = ("tool", "tox")


class TomlPyProject(Source):
    """Configuration sourced from a pyproject.toml files."""

    FILENAME = "pyproject.toml"
    _Section: type[TomlSection] = TomlPyProjectSection

    def __init__(self, path: Path) -> None:
        if path.name != self.FILENAME or not path.exists():
            raise ValueError
        with path.open("rb") as file_handler:
            self._content = tomllib.load(file_handler)
        try:
            our_content: Mapping[str, Any] = self._content
            for key in self._Section.PREFIX:
                our_content = our_content[key]
            self._our_content = our_content
        except KeyError as exc:
            raise MissingRequiredConfigKeyError(path) from exc
        if set(self._our_content.keys()) == {"legacy_tox_ini"}:
            raise MissingRequiredConfigKeyError(path)
        self._env_base_generated: dict[str, str] = _build_env_base_map(
            dict(self._our_content.get(self._Section.ENV_BASE, {})),
        )
        super().__init__(path)

    def get_core_section(self) -> Section:
        return self._Section(prefix=None, name="")

    def transform_section(self, section: Section) -> Section:
        return self._Section(section.prefix, section.name)

    def get_loader(self, section: Section, override_map: OverrideMap) -> Loader[Any] | None:
        current = self._our_content
        sec = cast("TomlSection", section)
        for key in sec.keys:
            if key in current:
                current = current[key]
            else:
                return None
        if not isinstance(current, Mapping):
            msg = f"{sec.key} must be a table, is {current.__class__.__name__!r}"
            raise HandledError(msg)
        is_core = section.prefix is None
        is_env_base = not is_core and sec.prefix == self._Section.env_base_prefix()
        unused_exclude: set[str] = set()
        if is_core:
            unused_exclude = {sec.ENV, sec.ENV_BASE, sec.RUN_ENV_BASE, sec.PKG_ENV_BASE}
        elif is_env_base:
            unused_exclude = {"factors"}
        return TomlLoader(
            section=section,
            overrides=override_map.get(section.key, []),
            content=current,
            root_content=self._content,
            unused_exclude=unused_exclude,
        )

    def envs(self, core_conf: CoreConfigSet) -> Iterator[str]:
        yield from core_conf["env_list"]
        yield from [section.name for section in self.sections()]
        yield from self._env_base_generated

    def sections(self) -> Iterator[Section]:
        for env_name in self._our_content.get(self._Section.ENV, {}):
            if not isinstance(env_name, str):
                msg = f"Environment key must be string, got {env_name!r}"
                raise HandledError(msg)
            yield self._Section.test_env(env_name)

    def get_base_sections(self, base: list[str], in_section: Section) -> Iterator[Section]:
        core_prefix = self._Section.core_prefix()
        strip = f"{core_prefix}{self._Section.SEP}" if core_prefix else ""
        env_base_pfx = self._Section.env_base_prefix()
        env_base_dot = f"{env_base_pfx}{self._Section.SEP}"
        for entry in base:
            if entry.startswith(env_base_dot):
                yield self._Section.env_base(entry[len(env_base_dot) :])
            else:
                yield self._Section(prefix=core_prefix or None, name=entry.removeprefix(strip))
                if in_section.prefix is not None:
                    yield self._Section(prefix=in_section.prefix, name=entry)

    def get_tox_env_section(self, item: str) -> tuple[Section, list[str], list[str]]:
        if base_name := self._env_base_generated.get(item):
            return (
                self._Section.test_env(item),
                [self._Section.env_base_prefix() + self._Section.SEP + base_name, self._Section.run_env_base()],
                [self._Section.package_env_base()],
            )
        return self._Section.test_env(item), [self._Section.run_env_base()], [self._Section.package_env_base()]


def _build_env_base_map(env_base_content: dict[str, Any]) -> dict[str, str]:
    result: dict[str, str] = {}
    for base_name, config in env_base_content.items():
        if not isinstance(config, Mapping):
            msg = f"env_base.{base_name} must be a table"
            raise HandledError(msg)
        factors_raw = config.get("factors")
        if factors_raw is None:
            msg = f"env_base.{base_name} requires a 'factors' key; use [env.{base_name}] for single environments"
            raise HandledError(msg)
        if not isinstance(factors_raw, list):
            msg = f"env_base.{base_name}.factors must be a list, got {type(factors_raw).__name__}"
            raise HandledError(msg)
        if factors_raw and isinstance(factors_raw[0], list | dict):
            expanded = [expand_factor_group(g) for g in factors_raw]
            names = ["-".join(combo) for combo in product(*expanded)]
        else:
            names = [str(f) for f in factors_raw]
        for factor_suffix in names:
            result[f"{base_name}-{factor_suffix}"] = base_name
    return result


__all__ = [
    "TomlPyProject",
]
