/+ D bindings to the GNU gettext runtime library.
   Copyright © 2025 Free Software Foundation, Inc.

   Boost Software License - Version 1.0 - August 17th, 2003

   Permission is hereby granted, free of charge, to any person or organization
   obtaining a copy of the software and accompanying documentation covered by
   this license (the "Software") to use, reproduce, display, distribute,
   execute, and transmit the Software, and to prepare derivative works of the
   Software, and to permit third-parties to whom the Software is furnished to
   do so, all subject to the following:

   The copyright notices in the Software and this entire statement, including
   the above license grant, this restriction and the following disclaimer,
   must be included in all copies of the Software, in whole or in part, and
   all derivative works of the Software, unless such copies or derivative
   works are solely in the form of machine-executable object code generated by
   a source language processor.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
   SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
   FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
   ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
   DEALINGS IN THE SOFTWARE.
 +/

/* Written by Bruno Haible <bruno@clisp.org>, 2025.  */

module gnu.libintl.libintl;

import gnu.libintl.internal.low;

import std.conv;
import std.string;

/* Argument conversion: While string *literals* in D are NUL-terminated, strings
   in general are not. Therefore we need to call toStringz, to ensure the C
   function receives a NUL-terminated 'const char *'.  */
/* Result conversion: When a 'const char *' is passed to a function such as
   writeln(), it prints it like a pointer. But we want to print it as a string.
   We therefore need to convert the result to 'string'.
   For this purpose, to!string works, whereas fromStringz does not.  */

public string gettext (string msgid)
{
  return to!string(unsafe_gettext(toStringz(msgid)));
}

public string dgettext (string domainname, string msgid)
{
  return to!string(unsafe_dgettext(toStringz(domainname), toStringz(msgid)));
}

public string dcgettext (string domainname, string msgid, int category)
{
  return to!string(unsafe_dcgettext(toStringz(domainname), toStringz(msgid), category));
}

public string ngettext (string msgid, string msgid_plural, ulong n)
{
  return to!string(unsafe_ngettext(toStringz(msgid), toStringz(msgid_plural), n));
}

public string dngettext (string domainname, string msgid, string msgid_plural, ulong n)
{
  return to!string(unsafe_dngettext(toStringz(domainname), toStringz(msgid), toStringz(msgid_plural), n));
}

public string dcngettext (string domainname, string msgid, string msgid_plural, ulong n, int category)
{
  return to!string(unsafe_dcngettext(toStringz(domainname), toStringz(msgid), toStringz(msgid_plural), n, category));
}

public string pgettext (string msgctxt, string msgid)
{
  const(char) * msg_ctxt_id = toStringz(msgctxt ~ "\x04" ~ msgid);
  const(char) * translation = unsafe_gettext (msg_ctxt_id);
  if (translation == msg_ctxt_id)
    return msgid;
  else
    return to!string(translation);
}

public string dpgettext (string domainname, string msgctxt, string msgid)
{
  const(char) * msg_ctxt_id = toStringz(msgctxt ~ "\x04" ~ msgid);
  const(char) * translation = unsafe_dgettext (toStringz(domainname), msg_ctxt_id);
  if (translation == msg_ctxt_id)
    return msgid;
  else
    return to!string(translation);
}

public string dcpgettext (string domainname, string msgctxt, string msgid, int category)
{
  const(char) * msg_ctxt_id = toStringz(msgctxt ~ "\x04" ~ msgid);
  const(char) * translation = unsafe_dcgettext (toStringz(domainname), msg_ctxt_id, category);
  if (translation == msg_ctxt_id)
    return msgid;
  else
    return to!string(translation);
}

public string npgettext (string msgctxt, string msgid, string msgid_plural, ulong n)
{
  const(char) * msg_ctxt_id = toStringz(msgctxt ~ "\x04" ~ msgid);
  const(char) * translation = unsafe_ngettext (msg_ctxt_id, toStringz(msgid_plural), n);
  if (translation == msg_ctxt_id)
    return (n == 1 ? msgid : msgid_plural);
  else
    return to!string(translation);
}

public string dnpgettext (string domainname, string msgctxt, string msgid, string msgid_plural, ulong n)
{
  const(char) * msg_ctxt_id = toStringz(msgctxt ~ "\x04" ~ msgid);
  const(char) * translation = unsafe_dngettext (toStringz(domainname), msg_ctxt_id, toStringz(msgid_plural), n);
  if (translation == msg_ctxt_id)
    return (n == 1 ? msgid : msgid_plural);
  else
    return to!string(translation);
}

public string dcnpgettext (string domainname, string msgctxt, string msgid, string msgid_plural, ulong n, int category)
{
  const(char) * msg_ctxt_id = toStringz(msgctxt ~ "\x04" ~ msgid);
  const(char) * translation = unsafe_dcngettext (toStringz(domainname), msg_ctxt_id, toStringz(msgid_plural), n, category);
  if (translation == msg_ctxt_id)
    return (n == 1 ? msgid : msgid_plural);
  else
    return to!string(translation);
}

public string textdomain (string domainname)
{
  return to!string(unsafe_textdomain(toStringz(domainname)));
}

public string bindtextdomain (string domainname, string dirname)
{
  // D uses UTF-8 for its string representation.
  unsafe_bindtextdomain(toStringz(domainname), "UTF-8");
  return to!string(unsafe_bindtextdomain(toStringz(domainname), toStringz(dirname)));
}

public const(char) * setlocale (int category, const(char) * locale)
{
  return unsafe_setlocale (category, locale);
}

unittest
{
  import std.stdio;
  writeln(gettext("Hello world!"));
}
