//! Rules from [pygrep-hooks](https://github.com/pre-commit/pygrep-hooks).
pub(crate) mod rules;

#[cfg(test)]
mod tests {
    use std::path::Path;

    use anyhow::Result;
    use test_case::test_case;

    use crate::registry::Rule;

    use crate::settings::LinterSettings;
    use crate::settings::types::PreviewMode;
    use crate::test::test_path;
    use crate::{assert_diagnostics, assert_diagnostics_diff, settings};

    #[test_case(Rule::BlanketTypeIgnore, Path::new("PGH003_0.py"))]
    #[test_case(Rule::BlanketTypeIgnore, Path::new("PGH003_1.py"))]
    #[test_case(Rule::BlanketNOQA, Path::new("PGH004_0.py"))]
    #[test_case(Rule::BlanketNOQA, Path::new("PGH004_1.py"))]
    #[test_case(Rule::BlanketNOQA, Path::new("PGH004_2.py"))]
    #[test_case(Rule::BlanketNOQA, Path::new("PGH004_3.py"))]
    #[test_case(Rule::InvalidMockAccess, Path::new("PGH005_0.py"))]
    fn rules(rule_code: Rule, path: &Path) -> Result<()> {
        let snapshot = format!("{}_{}", rule_code.noqa_code(), path.to_string_lossy());
        let diagnostics = test_path(
            Path::new("pygrep_hooks").join(path).as_path(),
            &settings::LinterSettings::for_rule(rule_code),
        )?;
        assert_diagnostics!(snapshot, diagnostics);
        Ok(())
    }

    #[test_case(Rule::InvalidMockAccess, Path::new("PGH005_0.py"))]
    fn preview_rules(rule_code: Rule, path: &Path) -> Result<()> {
        let snapshot = format!(
            "preview__{}_{}",
            rule_code.noqa_code(),
            path.to_string_lossy()
        );

        assert_diagnostics_diff!(
            snapshot,
            Path::new("pygrep_hooks").join(path).as_path(),
            &LinterSettings {
                preview: PreviewMode::Disabled,
                ..LinterSettings::for_rule(rule_code)
            },
            &LinterSettings {
                preview: PreviewMode::Enabled,
                ..LinterSettings::for_rule(rule_code)
            }
        );
        Ok(())
    }
}
