/*
 *
 * Copyright (c) 2020, Alliance for Open Media. All rights reserved.
 *
 * This source code is subject to the terms of the BSD 2 Clause License and
 * the Alliance for Open Media Patent License 1.0. If the BSD 2 Clause License
 * was not distributed with this source code in the LICENSE file, you can
 * obtain it at www.aomedia.org/license/software. If the Alliance for Open
 * Media Patent License 1.0 was not distributed with this source code in the
 * PATENTS file, you can obtain it at www.aomedia.org/license/patent.
 */

#include <tmmintrin.h>  // SSSE3
#include "config/av1_rtcd.h"
#include "config/aom_scale_rtcd.h"

#include "aom_dsp/x86/convolve_sse2.h"
#include "aom_dsp/x86/convolve_ssse3.h"
#include "aom_dsp/x86/mem_sse2.h"
#include "aom_dsp/x86/transpose_sse2.h"
#include "av1/common/resize.h"

static inline __m128i scale_plane_2_to_1_phase_0_kernel(
    const uint8_t *const src, const __m128i *const mask) {
  const __m128i a = _mm_loadu_si128((const __m128i *)(&src[0]));
  const __m128i b = _mm_loadu_si128((const __m128i *)(&src[16]));
  const __m128i a_and = _mm_and_si128(a, *mask);
  const __m128i b_and = _mm_and_si128(b, *mask);
  return _mm_packus_epi16(a_and, b_and);
}

static inline void shuffle_filter_odd_ssse3(const int16_t *const filter,
                                            __m128i *const f) {
  const __m128i f_values = _mm_load_si128((const __m128i *)filter);
  // pack and duplicate the filter values
  // It utilizes the fact that the high byte of filter[3] is always 0 to clean
  // half of f[0] and f[4].
  assert(filter[3] >= 0 && filter[3] < 256);
  f[0] = _mm_shuffle_epi8(f_values, _mm_set1_epi16(0x0007u));
  f[1] = _mm_shuffle_epi8(f_values, _mm_set1_epi16(0x0402u));
  f[2] = _mm_shuffle_epi8(f_values, _mm_set1_epi16(0x0806u));
  f[3] = _mm_shuffle_epi8(f_values, _mm_set1_epi16(0x0c0au));
  f[4] = _mm_shuffle_epi8(f_values, _mm_set1_epi16(0x070eu));
}

static inline __m128i convolve8_8_even_offset_ssse3(const __m128i *const s,
                                                    const __m128i *const f) {
  // multiply 2 adjacent elements with the filter and add the result
  const __m128i k_64 = _mm_set1_epi16(1 << 6);
  const __m128i x0 = _mm_maddubs_epi16(s[0], f[0]);
  const __m128i x1 = _mm_maddubs_epi16(s[1], f[1]);
  const __m128i x2 = _mm_maddubs_epi16(s[2], f[2]);
  const __m128i x3 = _mm_maddubs_epi16(s[3], f[3]);
  // compensate the subtracted 64 in f[1]. x4 is always non negative.
  const __m128i x4 = _mm_maddubs_epi16(s[1], _mm_set1_epi8(64));
  // add and saturate the results together
  __m128i temp = _mm_adds_epi16(x0, x3);
  temp = _mm_adds_epi16(temp, x1);
  temp = _mm_adds_epi16(temp, x2);
  temp = _mm_adds_epi16(temp, x4);
  // round and shift by 7 bit each 16 bit
  temp = _mm_adds_epi16(temp, k_64);
  temp = _mm_srai_epi16(temp, 7);
  return temp;
}

static inline __m128i convolve8_8_odd_offset_ssse3(const __m128i *const s,
                                                   const __m128i *const f) {
  // multiply 2 adjacent elements with the filter and add the result
  const __m128i k_64 = _mm_set1_epi16(1 << 6);
  const __m128i x0 = _mm_maddubs_epi16(s[0], f[0]);
  const __m128i x1 = _mm_maddubs_epi16(s[1], f[1]);
  const __m128i x2 = _mm_maddubs_epi16(s[2], f[2]);
  const __m128i x3 = _mm_maddubs_epi16(s[3], f[3]);
  const __m128i x4 = _mm_maddubs_epi16(s[4], f[4]);
  // compensate the subtracted 64 in f[2]. x5 is always non negative.
  const __m128i x5 = _mm_maddubs_epi16(s[2], _mm_set1_epi8(64));
  __m128i temp;

  // add and saturate the results together
  temp = _mm_adds_epi16(x0, x1);
  temp = _mm_adds_epi16(temp, x2);
  temp = _mm_adds_epi16(temp, x3);
  temp = _mm_adds_epi16(temp, x4);
  temp = _mm_adds_epi16(temp, x5);
  // round and shift by 7 bit each 16 bit
  temp = _mm_adds_epi16(temp, k_64);
  temp = _mm_srai_epi16(temp, 7);
  return temp;
}

static void scale_plane_2_to_1_phase_0(const uint8_t *src,
                                       const ptrdiff_t src_stride, uint8_t *dst,
                                       const ptrdiff_t dst_stride,
                                       const int dst_w, const int dst_h) {
  const int max_width = (dst_w + 15) & ~15;
  const __m128i mask = _mm_set1_epi16(0x00FF);
  int y = dst_h;

  do {
    int x = max_width;
    do {
      const __m128i d = scale_plane_2_to_1_phase_0_kernel(src, &mask);
      _mm_storeu_si128((__m128i *)dst, d);
      src += 32;
      dst += 16;
      x -= 16;
    } while (x);
    src += 2 * (src_stride - max_width);
    dst += dst_stride - max_width;
  } while (--y);
}

static void scale_plane_4_to_1_phase_0(const uint8_t *src,
                                       const ptrdiff_t src_stride, uint8_t *dst,
                                       const ptrdiff_t dst_stride,
                                       const int dst_w, const int dst_h) {
  const int max_width = (dst_w + 15) & ~15;
  const __m128i mask = _mm_set1_epi32(0x000000FF);
  int y = dst_h;

  do {
    int x = max_width;
    do {
      const __m128i d0 = scale_plane_2_to_1_phase_0_kernel(&src[0], &mask);
      const __m128i d1 = s