// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/Beam.hh"
#include "Rivet/Projections/ChargedFinalState.hh"
#include "Rivet/Projections/Thrust.hh"

namespace Rivet {


  /// @brief Charged particle multiplicities and distributions
  class TASSO_1980_I143691 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(TASSO_1980_I143691);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {

      // Initialise and register projections
      ChargedFinalState cfs;
      declare(cfs, "CFS");
      declare(Thrust(cfs), "Thrust");


      // Book histograms
      book(_p, 1, 1, 1);
      for (const string& en : _p.binning().edges<0>()) {
        if (isCompatibleWithSqrtS(stod(en)*GeV))  _sqs = en;
      }
      raiseBeamErrorIf(_sqs.empty());

      if (sqrtS() > 25*GeV)       _pre = "hi";
      else if (sqrtS() > 15*GeV)  _pre = "me";
      else                        _pre = "lo";
      book(_h["lorap"], 2,1,1);
      book(_h["lox"],   5,1,1);
      book(_h["merap"], 3,1,1);
      book(_h["mex"],   6,1,1);
      book(_h["hirap"], 4,1,1);
      book(_h["hix"],   7,1,1);
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {
      const ChargedFinalState& cfs = apply<ChargedFinalState>(event, "CFS");
      // thrust
      const Thrust& thrust = apply<Thrust>(event, "Thrust");
      Vector3 axis=thrust.thrustAxis();
      _p->fill(_sqs, cfs.particles().size());
      for (const Particle& p : cfs.particles()) {
        const Vector3 mom3 = p.p3();
        double pp = mom3.mod();
        double xP = 2.*pp/sqrtS();
        _h[_pre+"x"]->fill(xP);
        const double mom = dot(axis, mom3);
        const double rap = 0.5*log((p.E() + mom) / (p.E() - mom));
        _h[_pre+"rap"]->fill(fabs(rap));
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {

      for (auto& item : _h) {
        if (item.first.find("rap") != string::npos) {
          scale(item.second, 1.0/sumOfWeights());
        }
        else {
          // These observables aren't well defined: the histograms are filled
          // for an energy range but the scaling requires a specific energy.
          scale(item.second, crossSection()*sqr(sqrtS())/sumOfWeights()/microbarn);
        }
      }
    }

    /// @}


    /// @name Histograms
    /// @{
    map<string,Histo1DPtr> _h;
    BinnedProfilePtr<string> _p;
    string _sqs = "", _pre = "";
    /// @}


  };


  RIVET_DECLARE_PLUGIN(TASSO_1980_I143691);
}
