// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/FinalState.hh"
#include "Rivet/Projections/FastJets.hh"
#include "Rivet/Projections/DISFinalState.hh"
#include "Rivet/Tools/HistoGroup.hh"

namespace Rivet {


  /// @brief ZEUS inclusive jet cross sections in DIS
  class ZEUS_2023_I2694205 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(ZEUS_2023_I2694205);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {

      // The final-state particles are clustered in Breit frame
      // using FastJet with the kT algorithm and a jet-radius parameter of 1.
      const DISFinalState DISfs(DISFrame::BREIT);
      declare(DISfs, "fs");

      FastJets jets(DISfs, JetAlg::KT, 1.0);
      declare(jets, "jets");

      // Book histograms.
      // Transverse jet energies separated into pseudorapidity ranges.
      book(_h, {150., 200., 270., 400., 700., 5000., 15000.});
      for (auto& b : _h->bins()) {
        book(b, 7, 1, 9 + b.index());
      }
    }

    /// Perform the per-event analysis
    void analyze(const Event& event) {

      // Lorentz invariant DIS quantities
      const DISKinematics& dis = apply<DISFinalState>(event, "fs").kinematics();
      if (dis.failed())  vetoEvent;
      const double Q2 = dis.Q2();
      const double y  = dis.y();

      // Kinematic cuts on virtuality and inelasticity.
      if ( !inRange(Q2, 150.*GeV2, 15000.*GeV2) )  vetoEvent;
      if ( !inRange(y, 0.2, 0.7) )                 vetoEvent;

      // Lorentz boosts for Breit and lab frames.
      const LorentzTransform breitboost = dis.boostBreit();
      const LorentzTransform labboost = breitboost.inverse();

      // Retrieve clustered jets in Breit frame, sorted by pT.
      Jets alljets = apply<FastJets>(event, "jets").jetsByPt(Cuts::pT > 5*GeV);

      // Cut on Pseurdorapidity in lab frame.
      // 1 if hadron in "conventional" +z direction, -1 if in -z.
      Jets jets;
      const int orientation = dis.orientation();
      for (auto& jet : alljets) {
        jet.transformBy(labboost); // boost to lab frame
        if ( inRange(jet.eta()*orientation, -1., 2.5) ) {
          jet.transformBy(breitboost);
          jets += jet;
        }
      }

      // Fill histograms.
      for (const Jet& jet : jets) {
        _h->fill(Q2/GeV2, jet.pT()/GeV);
      }
    }

    /// Normalise histograms after the run
    void finalize() {

      scale(_h, crossSection()/picobarn/sumW());

    }

    /// @}

  private:

    /// @name Histograms
    /// @{
    Histo1DGroupPtr _h;
    /// @}
  };


  RIVET_DECLARE_PLUGIN(ZEUS_2023_I2694205);

}
