/*
 * Simple URL decoding function
 * Copyright (c) 2012 Antti Seppälä
 *
 * References:
 *  RFC 3986: Uniform Resource Identifier (URI): Generic Syntax
 *       T. Berners-Lee et al. The Internet Society, 2005
 *
 * based on http://www.icosaedro.it/apache/urldecode.c
 *          from Umberto Salsi (salsi@icosaedro.it)
 *
 * This file is part of FFmpeg.
 *
 * FFmpeg is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * FFmpeg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with FFmpeg; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <string.h>

#include "libavutil/error.h"
#include "libavutil/macros.h"
#include "libavutil/mem.h"
#include "libavutil/avstring.h"
#include "urldecode.h"

static size_t urldecode(char *dest, const char *url, size_t url_len, int decode_plus_sign)
{
    size_t s = 0, d = 0;
    char c;

    while (s < url_len) {
        c = url[s++];

        if (c == '%' && s + 1 < url_len) {
            char c2 = url[s++];
            char c3 = url[s++];
            if (av_isxdigit(c2) && av_isxdigit(c3)) {
                c2 = av_tolower(c2);
                c3 = av_tolower(c3);

                if (c2 <= '9')
                    c2 = c2 - '0';
                else
                    c2 = c2 - 'a' + 10;

                if (c3 <= '9')
                    c3 = c3 - '0';
                else
                    c3 = c3 - 'a' + 10;

                dest[d++] = 16 * c2 + c3;

            } else { /* %zz or something other invalid */
                dest[d++] = c;
                dest[d++] = c2;
                dest[d++] = c3;
            }
        } else if (c == '+' && decode_plus_sign) {
            dest[d++] = ' ';
        } else {
            dest[d++] = c;
        }

    }

    return d;
}

char *ff_urldecode(const char *url, int decode_plus_sign)
{
    char *dest = NULL;
    size_t url_len;

    if (!url)
        return NULL;

    url_len = strlen(url) + 1;
    dest = av_malloc(url_len);

    if (!dest)
        return NULL;

    urldecode(dest, url, url_len, decode_plus_sign);

    return dest;
}

int ff_urldecode_len(char *dest, size_t dest_len, const char *url, size_t url_max_len, int decode_plus_sign)
{
    size_t written_bytes;
    size_t url_len = strlen(url);

    url_len = FFMIN(url_len, url_max_len);

    if (dest_len <= url_len)
        return AVERROR(EINVAL);

    written_bytes = urldecode(dest, url, url_len, decode_plus_sign);
    dest[written_bytes] = '\0';

    return written_bytes;
}
