#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
This module inherits constants for Denon AVR receivers.

:copyright: (c) 2021 by Oliver Goetz.
:license: MIT, see LICENSE for more details.
"""

import re
from collections import namedtuple
from typing import Literal

import attr

# Attr constants
DENON_ATTR_SETATTR = [attr.setters.validate, attr.setters.convert]

# Types for constants
ReceiverType = namedtuple("ReceiverType", ["type", "port"])
DescriptionType = namedtuple("DescriptionType", ["port", "url"])
ReceiverURLs = namedtuple(
    "ReceiverURLs",
    [
        "appcommand",
        "appcommand0300",
        "status",
        "mainzone",
        "deviceinfo",
        "netaudiostatus",
        "tunerstatus",
        "hdtunerstatus",
        "command_sel_src",
        "command_fav_src",
        "command_power_on",
        "command_power_standby",
        "command_volume_up",
        "command_volume_down",
        "command_set_volume",
        "command_mute_on",
        "command_mute_off",
        "command_sel_sound_mode",
        "command_neural_x_on_off",
        "command_imax_auto_off",
        "command_imax_audio_settings",
        "command_imax_hpf",
        "command_imax_lpf",
        "command_imax_subwoofer_mode",
        "command_imax_subwoofer_output",
        "command_cinema_eq",
        "command_netaudio_post",
        "command_set_all_zone_stereo",
        "command_pause",
        "command_play",
        "command_stop",
        "command_cusor_up",
        "command_cusor_down",
        "command_cusor_left",
        "command_cusor_right",
        "command_cusor_enter",
        "command_back",
        "command_info",
        "command_options",
        "command_setup_open",
        "command_setup_close",
        "command_setup_query",
        "command_channel_level_adjust",
        "command_dimmer_toggle",
        "command_dimmer_set",
        "command_channel_volume",
        "command_channel_volumes_reset",
        "command_subwoofer_on_off",
        "command_subwoofer_level",
        "command_lfe",
        "command_tactile_transducer",
        "command_tactile_transducer_level",
        "command_tactile_transducer_lpf",
        "command_delay_up",
        "command_delay_down",
        "command_auromatic_3d_preset",
        "command_auromatic_3d_strength",
        "command_auro_3d_mode",
        "command_dirac_filter",
        "command_eco_mode",
        "command_lfc",
        "command_containment_amount",
        "command_loudness_management",
        "command_bass_sync",
        "command_dialog_enhancer",
        "command_hdmi_output",
        "command_hdmi_audio_decode",
        "command_quick_select_mode",
        "command_quick_select_memory",
        "command_smart_select_mode",  # Marantz
        "command_smart_select_memory",  # Marantz
        "command_auto_standby",
        "command_sleep",
        "command_center_spread",
        "command_video_processing_mode",
        "command_room_size",
        "command_status",
        "command_system_reset",
        "command_network_restart",
        "command_trigger",
        "command_speaker_preset",
        "command_bluetooth_transmitter",
        "command_dialog_control",
        "command_speaker_virtualizer",
        "command_effect_speaker_selection",
        "command_drc",
        "command_delay_time",
        "command_audio_restorer",
        "command_remote_control_lock",
        "command_panel_lock",
        "command_panel_and_volume_lock",
        "command_graphic_eq",
        "command_headphone_eq",
        "command_denon_hdmi_cec_on",
        "command_denon_hdmi_cec_off",
        "command_marantz_hdmi_cec_on",
        "command_marantz_hdmi_cec_off",
        "command_mdax",  # Marantz Only
        "command_dac_filter",  # Marantz Only
        "command_illumination",  # Marantz Only
        "command_auto_lip_sync",  # Marantz Only
        "command_dolby_atmos_toggle_denon",
        "command_dolby_atmos_toggle_marantz",
        "command_page_up_denon",
        "command_page_down_denon",
        "command_page_up_marantz",
        "command_page_down_marantz",
        "command_input_mode_select_denon",
        "command_input_mode_auto_denon",
        "command_input_mode_hdmi_denon",
        "command_input_mode_digital_denon",
        "command_input_mode_analog_denon",
        "command_input_mode_select_marantz",
        "command_input_mode_auto_marantz",
        "command_input_mode_hdmi_marantz",
        "command_input_mode_digital_marantz",
        "command_input_mode_analog_marantz",
    ],
)
TelnetCommands = namedtuple(
    "TelnetCommands",
    [
        "command_sel_src",
        "command_fav_src",
        "command_power_on",
        "command_power_standby",
        "command_volume_up",
        "command_volume_down",
        "command_set_volume",
        "command_mute_on",
        "command_mute_off",
        "command_sel_sound_mode",
        "command_neural_x_on_off",
        "command_imax_auto_off",
        "command_imax_audio_settings",
        "command_imax_hpf",
        "command_imax_lpf",
        "command_imax_subwoofer_mode",
        "command_imax_subwoofer_output",
        "command_cinema_eq",
        "command_set_all_zone_stereo",
        "command_pause",
        "command_play",
        "command_stop",
        "command_multieq",
        "command_dynamiceq",
        "command_reflevoffset",
        "command_dynamicvol",
        "command_tonecontrol",
        "command_bass",
        "command_treble",
        "command_cusor_up",
        "command_cusor_down",
        "command_cusor_left",
        "command_cusor_right",
        "command_cusor_enter",
        "command_back",
        "command_info",
        "command_options",
        "command_setup_open",
        "command_setup_close",
        "command_setup_query",
        "command_channel_level_adjust",
        "command_dimmer_toggle",
        "command_dimmer_set",
        "command_channel_volume",
        "command_channel_volumes_reset",
        "command_subwoofer_on_off",
        "command_subwoofer_level",
        "command_lfe",
        "command_tactile_transducer",
        "command_tactile_transducer_level",
        "command_tactile_transducer_lpf",
        "command_delay_up",
        "command_delay_down",
        "command_auromatic_3d_preset",
        "command_auromatic_3d_strength",
        "command_auro_3d_mode",
        "command_dirac_filter",
        "command_eco_mode",
        "command_lfc",
        "command_containment_amount",
        "command_loudness_management",
        "command_bass_sync",
        "command_dialog_enhancer",
        "command_hdmi_output",
        "command_hdmi_audio_decode",
        "command_quick_select_mode",
        "command_quick_select_memory",
        "command_smart_select_mode",  # Marantz
        "command_smart_select_memory",  # Marantz
        "command_auto_standby",
        "command_sleep",
        "command_center_spread",
        "command_video_processing_mode",
        "command_room_size",
        "command_status",
        "command_system_reset",
        "command_network_restart",
        "command_trigger",
        "command_speaker_preset",
        "command_bluetooth_transmitter",
        "command_dialog_control",
        "command_speaker_virtualizer",
        "command_effect_speaker_selection",
        "command_drc",
        "command_delay_time",
        "command_audio_restorer",
        "command_remote_control_lock",
        "command_panel_lock",
        "command_panel_and_volume_lock",
        "command_graphic_eq",
        "command_headphone_eq",
        "command_denon_hdmi_cec_on",
        "command_denon_hdmi_cec_off",
        "command_marantz_hdmi_cec_on",
        "command_marantz_hdmi_cec_off",
        "command_mdax",  # Marantz Only
        "command_dac_filter",  # Marantz Only
        "command_illumination",  # Marantz Only
        "command_auto_lip_sync",  # Marantz Only
        "command_dolby_atmos_toggle_denon",
        "command_dolby_atmos_toggle_marantz",
        "command_page_up_denon",
        "command_page_down_denon",
        "command_page_up_marantz",
        "command_page_down_marantz",
        "command_input_mode_select_denon",
        "command_input_mode_auto_denon",
        "command_input_mode_hdmi_denon",
        "command_input_mode_digital_denon",
        "command_input_mode_analog_denon",
        "command_input_mode_select_marantz",
        "command_input_mode_auto_marantz",
        "command_input_mode_hdmi_marantz",
        "command_input_mode_digital_marantz",
        "command_input_mode_analog_marantz",
    ],
)

# AVR-X search patterns
DEVICEINFO_AVR_X_PATTERN = re.compile(
    r"(.*AV([CR])-([XSA]).*|.*SR500[6-9]|.*SR60(07|08|09|10|11|12|13)|."
    r"*SR70(07|08|09|10|11|12|13)|.*SR501[3-4]|.*NR1604|.*NR1710)"
)

DEVICEINFO_COMMAPI_PATTERN = re.compile(r"(0210|0220|0250|0300|0301)")

# Receiver types
AVR_NAME = "avr"
AVR_X_NAME = "avr-x"
AVR_X_2016_NAME = "avr-x-2016"

AVR = ReceiverType(type=AVR_NAME, port=80)
AVR_X = ReceiverType(type=AVR_X_NAME, port=80)
AVR_X_2016 = ReceiverType(type=AVR_X_2016_NAME, port=8080)
VALID_RECEIVER_TYPES = (AVR, AVR_X, AVR_X_2016)

DESCRIPTION_TYPES = {
    AVR_NAME: DescriptionType(port=8080, url="/description.xml"),
    AVR_X_NAME: DescriptionType(port=8080, url="/description.xml"),
    AVR_X_2016_NAME: DescriptionType(
        port=60006, url="/upnp/desc/aios_device/aios_device.xml"
    ),
}

SOURCE_MAPPING = {
    "TV AUDIO": "TV",
    "iPod/USB": "USB/IPOD",
    "Bluetooth": "BT",
    "Blu-ray": "BD",
    "CBL/SAT": "SAT/CBL",
    "NETWORK": "NET",
    "Media Player": "MPLAY",
    "AUX": "AUX1",
    "Tuner": "TUNER",
    "FM": "TUNER",
    "SpotifyConnect": "Spotify Connect",
}

CHANGE_INPUT_MAPPING = {
    "Favorites": "FAVORITES",
    "Flickr": "FLICKR",
    "Internet Radio": "IRADIO",
    "Media Server": "SERVER",
    "Online Music": "NET",
    "Spotify": "SPOTIFY",
}

TELNET_SOURCES = {
    "CD",
    "PHONO",
    "TUNER",
    "DVD",
    "BD",
    "TV",
    "SAT/CBL",
    "MPLAY",
    "GAME",
    "HDRADIO",
    "NET",
    "PANDORA",
    "SIRIUSXM",
    "SOURCE",
    "LASTFM",
    "FLICKR",
    "IRADIO",
    "IRP",
    "SERVER",
    "FAVORITES",
    "AUX1",
    "AUX2",
    "AUX3",
    "AUX4",
    "AUX5",
    "AUX6",
    "AUX7",
    "BT",
    "USB/IPOD",
    "USB DIRECT",
    "IPOD DIRECT",
}

TELNET_MAPPING = {
    "FAVORITES": "Favorites",
    "FLICKR": "Flickr",
    "IRADIO": "Internet Radio",
    "IRP": "Internet Radio",
    "SERVER": "Media Server",
    "SPOTIFY": "Spotify",
}

# Sound modes
ALL_ZONE_STEREO = "ALL ZONE STEREO"

SOUND_MODE_MAPPING = {
    "MUSIC": [
        "AAC+NEO:X M",
        "AAC+PL2X M",
        "DOLBY D +NEO:X M",
        "DOLBY D+ +NEO:X M",
        "DOLBY D+ +PL2X M",
        "DOLBY D+NEO:X M",
        "DOLBY D+PL2X M",
        "DOLBY HD+NEO:X M",
        "DOLBY HD+PL2X M",
        "DOLBY PL2 M",
        "DOLBY PL2 MUSIC",
        "DOLBY PL2 X MUSIC",
        "DOLBY PL2X M",
        "DOLBY PLII MS",
        "DTS HD+NEO:X M",
        "DTS HD+PL2X M",
        "DTS NEO:6 M",
        "DTS NEO:6 MUSIC",
        "DTS NEO:X M",
        "DTS NEO:X MUSIC",
        "DTS+NEO:X M",
        "DTS+PL2X M",
        "M CH IN+NEO:X M",
        "M CH IN+PL2X M",
        "PLII MUSIC",
        "PLIIX MUSIC",
    ],
    "MOVIE": [
        "AAC+NEO:X C",
        "AAC+PL2X C",
        "DOLBY D +NEO:X C",
        "DOLBY D+ +NEO:X C",
        "DOLBY D+ +PL2X C",
        "DOLBY D+NEO:X C",
        "DOLBY D+PL2X C",
        "DOLBY HD+NEO:X C",
        "DOLBY HD+PL2X C",
        "DOLBY PL2 C",
        "DOLBY PL2 CINEMA",
        "DOLBY PL2 MOVIE",
        "DOLBY PL2 X MOVIE",
        "DOLBY PL2X C",
        "DOLBY PLII MOVIE",
        "DOLBY PLII MV",
        "DTS HD+NEO:X C",
        "DTS HD+PL2X C",
        "DTS NEO:6 C",
        "DTS NEO:6 CINEMA",
        "DTS NEO:X C",
        "DTS NEO:X CINEMA",
        "DTS+NEO:X C",
        "DTS+PL2X C",
        "M CH IN+NEO:X C",
        "M CH IN+PL2X C",
        "MULTI IN + VIRTUAL:X",
        "PLII CINEMA",
        "PLII MOVIE",
        "PLIIX CINEMA",
    ],
    "GAME": [
        "AAC+NEO:X G",
        "DOLBY D +NEO:X G",
        "DOLBY D+ +NEO:X G",
        "DOLBY D+NEO:X G",
        "DOLBY HD+NEO:X G",
        "DOLBY PL2 G",
        "DOLBY PL2 GAME",
        "DOLBY PL2 X GAME",
        "DOLBY PL2X G",
        "DOLBY PLII GAME",
        "DOLBY PLII GM",
        "DTS HD+NEO:X G",
        "DTS NEO:X G",
        "DTS+NEO:X G",
        "M CH IN+NEO:X G",
        "PLII GAME",
    ],
    "AUTO": ["None"],
    "STANDARD": ["None2"],
    "VIRTUAL": ["VIRTUAL"],
    "MATRIX": ["MATRIX"],
    "ROCK ARENA": ["ROCK ARENA"],
    "JAZZ CLUB": ["JAZZ CLUB"],
    "VIDEO GAME": ["VIDEO GAME"],
    "MONO MOVIE": ["MONO MOVIE"],
    "SUPER STADIUM": ["SUPER STADIUM"],
    "CLASSIC CONCERT": ["CLASSIC CONCERT"],
    "WIDE SCREEN": ["WIDE SCREEN"],
    "DIRECT": ["DIRECT"],
    "PURE DIRECT": ["PURE_DIRECT", "PURE DIRECT"],
    "DOLBY DIGITAL": [
        "AAC+DOLBY EX",
        "AAC+DSUR",
        "AAC+NEURAL:X",
        "AAC+PL2Z H",
        "DOLBY ATMOS",
        "DOLBY AUDIO - DD + DSUR",
        "DOLBY AUDIO - DD + NEURAL:X",
        "DOLBY AUDIO - DD+   + DSUR",
        "DOLBY AUDIO - DD+   + NEURAL:X",
        "DOLBY AUDIO - DOLBY DIGITAL",
        "DOLBY AUDIO - DOLBY DIGITAL PLUS",
        "DOLBY AUDIO - DOLBY SURROUND",
        "DOLBY AUDIO - DOLBY TRUEHD",
        "DOLBY AUDIO - TRUEHD + DSUR",
        "DOLBY AUDIO - TRUEHD + NEURAL:X",
        "DOLBY AUDIO-DD",
        "DOLBY AUDIO-DD+",
        "DOLBY AUDIO-DD+ +DSUR",
        # FY24 specification uses incorrect spelling
        "DOLBY AUDIO-DD+ +NERUAL:X",
        # FY23 specification uses correct spelling, go figure
        "DOLBY AUDIO-DD+ +NEURAL:X",
        "DOLBY AUDIO-DD+DSUR",
        "DOLBY AUDIO-DD+NEURAL:X",
        "DOLBY AUDIO-DSUR",
        "DOLBY AUDIO-TRUEHD",
        "DOLBY AUDIO-TRUEHD+DSUR",
        "DOLBY AUDIO-TRUEHD+NEURAL:X",
        "DOLBY D + +DOLBY SURROUND",
        "DOLBY D + DOLBY SURROUND",
        "DOLBY D + NEURAL:X",
        "DOLBY D EX",
        "DOLBY D+",
        "DOLBY D+ +DS",
        "DOLBY D+ +EX",
        "DOLBY D+ +PL2Z H",
        "DOLBY D+DS",
        "DOLBY D+PL2Z H",
        "DOLBY DIGITAL",
        "DOLBY DIGITAL +",
        "DOLBY DIGITAL + + NEURAL:X",
        "DOLBY DIGITAL + NEURAL:X",
        "DOLBY HD",
        "DOLBY HD + DOLBY SURROUND",
        "DOLBY HD+EX",
        "DOLBY HD+PL2Z H",
        "DOLBY PL2Z H",
        "DOLBY PRO LOGIC",
        "DOLBY SURROUND",
        "DOLBY TRUEHD",
        "M CH IN+DOLBY EX",
        "M CH IN+DSUR",
        "M CH IN+PL2Z H",
        "MPEG-H",
        "MPEG2 AAC",
        "MPEG4 AAC",
        "MULTI IN + DOLBY SURROUND",
        "MULTI IN + DSUR",
        "MULTI IN + NEURAL:X",
        "NEURAL",
        "STANDARD(DOLBY)",
    ],
    "DTS SURROUND": [
        "AAC+NEURAL:X",
        "AAC+VIRTUAL:X",
        "DTS + DOLBY SURROUND",
        "DTS + NEURAL:X",
        "DTS + VIRTUAL:X",
        "DTS ES 8CH DSCRT",
        "DTS ES DSCRT+NEURAL:X",
        "DTS ES DSCRT6.1",
        "DTS ES MTRX+NEURAL:X",
        "DTS ES MTRX6.1",
        "DTS EXPRESS",
        "DTS HD",
        "DTS HD MSTR",
        "DTS HD+DSUR",
        "DTS HD+NEO:6",
        "DTS HD+NEURAL:X",
        "DTS HD+PL2Z H",
        "DTS HD+VIRTUAL:X",
        "DTS NEURAL:X",
        "DTS SURROUND",
        "DTS VIRTUAL:X",
        "DTS-HD",
        "DTS-HD + DOLBY SURROUND",
        "DTS-HD + DSUR",
        "DTS-HD + NEURAL:X",
        "DTS-HD MSTR",
        "DTS:X",
        "DTS:X MSTR",
        "DTS:X+VIRTUAL:X",
        "DTS+DSUR",
        "DTS+NEO:6",
        "DTS+NEUR",
        "DTS+NEURAL:X",
        "DTS+PL2Z H",
        "DTS+VIRTU",
        "DTS+VIRTUAL:X",
        "DTS96 ES MTRX",
        "DTS96/24",
        "IMAX DTS",
        "IMAX DTS:X",
        "IMAX DTS:X+VIRTUAL:X",
        "IMAX DTS+NEURAL:X",
        "IMAX DTS+VIRTUAL:X",
        "M CH IN+NEURAL:X",
        "M CH IN+VIRTUAL:X",
        "MAX DTS+VIRTUAL:X",
        # Exists in both DOLBY and DTS in the specification,
        # but can't be in both here, so we it them in DTS only
        "MULTI CH IN",
        # Exists in both DOLBY and DTS in the specification,
        # but can't be in both here, so we it them in DTS only
        "MULTI CH IN 7.1",
        "NEURAL:X",
        "STANDARD(DTS)",
        "VIRTUAL:X",
    ],
    "AURO3D": ["AURO-3D", "AURO3D"],
    "AURO2DSURR": ["AURO-2D SURROUND", "AURO2DSURR"],
    "MCH STEREO": [
        "MCH STEREO",
        "MULTI CH STEREO",
        "MULTI_CH_STEREO",
    ],
    "STEREO": ["STEREO"],
    ALL_ZONE_STEREO: ["ALL ZONE STEREO"],
}

# Receiver sources
NETAUDIO_SOURCES = {
    "AirPlay",
    "Online Music",
    "Media Server",
    "iPod/USB",
    "Bluetooth",
    "Internet Radio",
    "Favorites",
    "SpotifyConnect",
    "Flickr",
    "NET/USB",
    "Music Server",
    "NETWORK",
    "NET",
}
TUNER_SOURCES = {"Tuner", "TUNER"}
HDTUNER_SOURCES = {"HD Radio", "HDRADIO"}
PLAYING_SOURCES = set().union(*[NETAUDIO_SOURCES, TUNER_SOURCES, HDTUNER_SOURCES])

# Image URLs
STATIC_ALBUM_URL = "http://{host}:{port}/img/album%20art_S.png"
ALBUM_COVERS_URL = "http://{host}:{port}/NetAudio/art.asp-jpg?{hash}"

# General URLs
APPCOMMAND_URL = "/goform/AppCommand.xml"
APPCOMMAND0300_URL = "/goform/AppCommand0300.xml"
DEVICEINFO_URL = "/goform/Deviceinfo.xml"
NETAUDIOSTATUS_URL = "/goform/formNetAudio_StatusXml.xml"
TUNERSTATUS_URL = "/goform/formTuner_TunerXml.xml"
HDTUNERSTATUS_URL = "/goform/formTuner_HdXml.xml"
COMMAND_NETAUDIO_POST_URL = "/NetAudio/index.put.asp"
COMMAND_PAUSE = "/goform/formiPhoneAppDirect.xml?NS9B"
COMMAND_PLAY = "/goform/formiPhoneAppDirect.xml?NS9A"
COMMAND_STOP = "/goform/formiPhoneAppDirect.xml?NS9C"
COMMAND_ECO_MODE = "/goform/formiPhoneAppDirect.xml?ECO{mode}"


# Main Zone URLs
STATUS_URL = "/goform/formMainZone_MainZoneXmlStatus.xml"
MAINZONE_URL = "/goform/formMainZone_MainZoneXml.xml"
COMMAND_SEL_SRC_URL = "/goform/formiPhoneAppDirect.xml?SI"
COMMAND_FAV_SRC_URL = "/goform/formiPhoneAppDirect.xml?ZM"
COMMAND_POWER_ON_URL = "/goform/formiPhoneAppPower.xml?1+PowerOn"
COMMAND_POWER_STANDBY_URL = "/goform/formiPhoneAppPower.xml?1+PowerStandby"
COMMAND_VOLUME_UP_URL = "/goform/formiPhoneAppDirect.xml?MVUP"
COMMAND_VOLUME_DOWN_URL = "/goform/formiPhoneAppDirect.xml?MVDOWN"
COMMAND_SET_VOLUME_URL = "/goform/formiPhoneAppVolume.xml?1+{volume:.1f}"
COMMAND_MUTE_ON_URL = "/goform/formiPhoneAppMute.xml?1+MuteOn"
COMMAND_MUTE_OFF_URL = "/goform/formiPhoneAppMute.xml?1+MuteOff"
COMMAND_SEL_SM_URL = "/goform/formiPhoneAppDirect.xml?MS"
COMMAND_SET_ZST_URL = "/goform/formiPhoneAppDirect.xml?MN"
COMMAND_NEURAL_X_ON_OFF = "/goform/formiPhoneAppDirect.xml?PSNEURAL%20{mode}"
COMMAND_IMAX_AUTO_OFF = "/goform/formiPhoneAppDirect.xml?PSIMAX%20{mode}"
COMMAND_IMAX_AUDIO_SETTINGS = "/goform/formiPhoneAppDirect.xml?PSIMAXAUD%20{mode}"
COMMAND_IMAX_HPF = "/goform/formiPhoneAppDirect.xml?PSIMAXHPF%20{frequency}"
COMMAND_IMAX_LPF = "/goform/formiPhoneAppDirect.xml?PSIMAXLPF%20{frequency}"
COMMAND_IMAX_SUBWOOFER_MODE = "/goform/formiPhoneAppDirect.xml?PSIMAXSWM%20{mode}"
COMMAND_IMAX_SUBWOOFER_OUTPUT = "/goform/formiPhoneAppDirect.xml?PSIMAXSWO%20{mode}"
COMMAND_CINEMA_EQ = "/goform/formiPhoneAppDirect.xml?PSCINEMA%20EQ.{mode}"
COMMAND_CURSOR_UP = "/goform/formiPhoneAppDirect.xml?MNCUP"
COMMAND_CURSOR_DOWN = "/goform/formiPhoneAppDirect.xml?MNCDN"
COMMAND_CURSOR_LEFT = "/goform/formiPhoneAppDirect.xml?MNCLT"
COMMAND_CURSOR_RIGHT = "/goform/formiPhoneAppDirect.xml?MNCRT"
COMMAND_CURSOR_ENTER = "/goform/formiPhoneAppDirect.xml?MNENT"
COMMAND_BACK = "/goform/formiPhoneAppDirect.xml?MNRTN"
COMMAND_INFO = "/goform/formiPhoneAppDirect.xml?MNINF"
COMMAND_OPTIONS = "/goform/formiPhoneAppDirect.xml?MNOPT"
COMMAND_SETUP_OPEN = "/goform/formiPhoneAppDirect.xml?MNMEN%20ON"
COMMAND_SETUP_CLOSE = "/goform/formiPhoneAppDirect.xml?MNMEN%20OFF"
COMMAND_SETUP_QUERY = "/goform/formiPhoneAppDirect.xml?MNMEN?"
COMMAND_CHANNEL_LEVEL_ADJUST = "/goform/formiPhoneAppDirect.xml?MNCHL"
COMMAND_DIMMER_TOGGLE = "/goform/formiPhoneAppDirect.xml?DIM%20SEL"
COMMAND_DIMMER_SET = "/goform/formiPhoneAppDirect.xml?DIM%20{mode}"
COMMAND_CHANNEL_VOLUME = "/goform/formiPhoneAppDirect.xml?CV{channel}%20{value}"
COMMAND_CHANNEL_VOLUMES_RESET = "/goform/formiPhoneAppDirect.xml?CVZRL"
COMMAND_SUBWOOFER_ON_OFF = "/goform/formiPhoneAppDirect.xml?PSSWR%20{mode}"
COMMAND_SUBWOOFER_LEVEL = "/goform/formiPhoneAppDirect.xml?PSSWL{number}%20{mode}"
COMMAND_LFE = "/goform/formiPhoneAppDirect.xml?PSLFE%20{mode}"
COMMAND_TACTILE_TRANSDUCER = "/goform/formiPhoneAppDirect.xml?SSTTR%20{mode}"
COMMAND_TACTILE_TRANSDUCER_LEVEL = "/goform/formiPhoneAppDirect.xml?SSTTRLEV%20{mode}"
COMMAND_TACTILE_TRANSDUCER_LPF = (
    "/goform/formiPhoneAppDirect.xml?SSTTRLPF%20{frequency}"
)
COMMAND_DELAY_UP = "/goform/formiPhoneAppDirect.xml?PSDELAY%20UP"
COMMAND_DELAY_DOWN = "/goform/formiPhoneAppDirect.xml?PSDELAY%20DOWN"
COMMAND_AUROMATIC_3D_PRESET = "/goform/formiPhoneAppDirect.xml?PSAUROPR%20{preset}"
COMMAND_AUROMATIC_3D_STRENGTH = "/goform/formiPhoneAppDirect.xml?PSAUROST%20{value}"
COMMAND_AURO_3D_MODE = "/goform/formiPhoneAppDirect.xml?PSAUROMODEM%20{mode}"
COMMAND_DIRAC_FILTER = "/goform/formiPhoneAppDirect.xml?PSDIRAC%20{filter}"
COMMAND_LFC = "/goform/formiPhoneAppDirect.xml?PSLFC%20{mode}"
COMMAND_CONTAINMENT_AMOUNT = "/goform/formiPhoneAppDirect.xml?PSCNTAMT%20{value}"
COMMAND_LOUDNESS_MANAGEMENT = "/goform/formiPhoneAppDirect.xml?PSLOM%20{mode}"
COMMAND_BASS_SYNC = "/goform/formiPhoneAppDirect.xml?PSBSC%20{mode}"
COMMAND_DIALOG_ENHANCER = "/goform/formiPhoneAppDirect.xml?PSDEH%20{level}"
COMMAND_HDMI_OUTPUT = "/goform/formiPhoneAppDirect.xml?VSMONI{output}"
COMMAND_HDMI_AUDIO_DECODE = "/goform/formiPhoneAppDirect.xml?VSAUDIO%20{mode}"
COMMAND_QUICK_SELECT_MODE = "/goform/formiPhoneAppDirect.xml?MSQUICK{number}"
COMMAND_QUICK_SELECT_MEMORY = "/goform/formiPhoneAppDirect.xml?MSQUICK{number}%20MEMORY"
COMMAND_SMART_SELECT_MODE = "/goform/formiPhoneAppDirect.xml?MSSMART{number}"  # Marantz
COMMAND_SMART_SELECT_MEMORY = (
    "/goform/formiPhoneAppDirect.xml?MSSMART{number}%20MEMORY"  # Marantz
)
COMMAND_AUTO_STANDBY = "/goform/formiPhoneAppDirect.xml?STBY{mode}"
COMMAND_SLEEP = "/goform/formiPhoneAppDirect.xml?SLP{value}"
COMMAND_CENTER_SPREAD = "/goform/formiPhoneAppDirect.xml?PSCES%20{mode}"
COMMAND_VIDEO_PROCESSING_MODE = "/goform/formiPhoneAppDirect.xml?VSVPM{mode}"
COMMAND_ROOM_SIZE = "/goform/formiPhoneAppDirect.xml?PSRSZ%20{size}"
COMMAND_STATUS = "/goform/formiPhoneAppDirect.xml?RCSHP0230030"
COMMAND_SYSTEM_RESET = "/goform/formiPhoneAppDirect.xml?SYRST"
COMMAND_NETWORK_RESTART = "/goform/formiPhoneAppDirect.xml?NSRBT"
COMMAND_TRIGGER = "/goform/formiPhoneAppDirect.xml?TR{number}%20{mode}"
COMMAND_SPEAKER_PRESET = "/goform/formiPhoneAppDirect.xml?SPPR%20{number}"
COMMAND_BLUETOOTH_TRANSMITTER = "/goform/formiPhoneAppDirect.xml?BTTX%20{mode}"
COMMAND_DIALOG_CONTROL = "/goform/formiPhoneAppDirect.xml?PSDIC%20{value}"
COMMAND_SPEAKER_VIRTUALIZER = "/goform/formiPhoneAppDirect.xml?PSSPV%20{mode}"
COMMAND_EFFECT_SPEAKER_SELECTION = "/goform/formiPhoneAppDirect.xml?PSSP:{mode}"
COMMAND_DRC = "/goform/formiPhoneAppDirect.xml?PSDRC%20{mode}"
COMMAND_DELAY_TIME = "/goform/formiPhoneAppDirect.xml?PSDEL%20{value}"
COMMAND_AUDIO_RESTORER = "/goform/formiPhoneAppDirect.xml?PSRSTR%20{mode}"
COMMAND_REMOTE_CONTROL_LOCK = "/goform/formiPhoneAppDirect.xml?SYREMOTE%20LOCK%20{mode}"
COMMAND_PANEL_LOCK = "/goform/formiPhoneAppDirect.xml?SYPANEL%20LOCK%20{mode}"
COMMAND_PANEL_AND_VOLUME_LOCK = "/goform/formiPhoneAppDirect.xml?SYPANEL+V%20LOCK%20ON"
COMMAND_GRAPHIC_EQ = "/goform/formiPhoneAppDirect.xml?PSGEQ%20{mode}"
COMMAND_HEADPHONE_EQ = "/goform/formiPhoneAppDirect.xml?PSHEQ%20{mode}"
COMMAND_DENON_HDMI_CEC_ON = "/goform/formiPhoneAppDirect.xml?RCKSK0410826"
COMMAND_DENON_HDMI_CEC_OFF = "/goform/formiPhoneAppDirect.xml?RCKSK0410827"
COMMAND_MARANTZ_HDMI_CEC_ON = "/goform/formiPhoneAppDirect.xml?RCRC51608408"
COMMAND_MARANTZ_HDMI_CEC_OFF = "/goform/formiPhoneAppDirect.xml?RCRC51608409"
COMMAND_MDAX = "/goform/formiPhoneAppDirect.xml?PSMDAX%20{mode}"  # Marantz Only
COMMAND_DAC_FILTER = "/goform/formiPhoneAppDirect.xml?PSDACFIL%20{mode}"  # Marantz Only
COMMAND_ILLUMINATION = "/goform/formiPhoneAppDirect.xml?ILB%20{mode}"  # Marantz Only
COMMAND_AUTO_LIP_SYNC = (
    "/goform/formiPhoneAppDirect.xml?SSHOSALS%20{mode}"  # Marantz Only
)
COMMAND_DOLBY_ATMOS_TOGGLE_DENON = "/goform/formiPhoneAppDirect.xml?RCKSK0410348"
COMMAND_DOLBY_ATMOS_TOGGLE_MARANTZ = "/goform/formiPhoneAppDirect.xml?RCRC51606425"
COMMAND_PAGE_UP_DENON = "/goform/formiPhoneAppDirect.xml?RCKSK0410945"
COMMAND_PAGE_DOWN_DENON = "/goform/formiPhoneAppDirect.xml?RCKSK0410946"
COMMAND_PAGE_UP_MARANTZ = "/goform/formiPhoneAppDirect.xml?RCRC52708105"
COMMAND_PAGE_DOWN_MARANTZ = "/goform/formiPhoneAppDirect.xml?RCRC52708005"
COMMAND_INPUT_MODE_SELECT_DENON = "/goform/formiPhoneAppDirect.xml?RCKSK0410352"
COMMAND_INPUT_MODE_AUTO_DENON = "/goform/formiPhoneAppDirect.xml?RCKSK0410358"
COMMAND_INPUT_MODE_HDMI_DENON = "/goform/formiPhoneAppDirect.xml?RCKSK0410357"
COMMAND_INPUT_MODE_DIGITAL_DENON = "/goform/formiPhoneAppDirect.xml?RCKSK0410359"
COMMAND_INPUT_MODE_ANALOG_DENON = "/goform/formiPhoneAppDirect.xml?RCKSK0410353"
COMMAND_INPUT_MODE_SELECT_MARANTZ = "/goform/formiPhoneAppDirect.xml?RCRC51600101"
COMMAND_INPUT_MODE_AUTO_MARANTZ = "/goform/formiPhoneAppDirect.xml?RCRC51600115"
COMMAND_INPUT_MODE_HDMI_MARANTZ = "/goform/formiPhoneAppDirect.xml?RCRC51600116"
COMMAND_INPUT_MODE_DIGITAL_MARANTZ = "/goform/formiPhoneAppDirect.xml?RCRC51600117"
COMMAND_INPUT_MODE_ANALOG_MARANTZ = "/goform/formiPhoneAppDirect.xml?RCRC51600118"

# Zone 2 URLs
STATUS_Z2_URL = "/goform/formZone2_Zone2XmlStatus.xml"
COMMAND_SEL_SRC_Z2_URL = "/goform/formiPhoneAppDirect.xml?Z2"
COMMAND_FAV_SRC_Z2_URL = "/goform/formiPhoneAppDirect.xml?Z2"
COMMAND_POWER_ON_Z2_URL = "/goform/formiPhoneAppPower.xml?2+PowerOn"
COMMAND_POWER_STANDBY_Z2_URL = "/goform/formiPhoneAppPower.xml?2+PowerStandby"
COMMAND_VOLUME_UP_Z2_URL = "/goform/formiPhoneAppDirect.xml?Z2UP"
COMMAND_VOLUME_DOWN_Z2_URL = "/goform/formiPhoneAppDirect.xml?Z2DOWN"
COMMAND_SET_VOLUME_Z2_URL = "/goform/formiPhoneAppVolume.xml?2+{volume:.1f}"
COMMAND_MUTE_ON_Z2_URL = "/goform/formiPhoneAppMute.xml?2+MuteOn"
COMMAND_MUTE_OFF_Z2_URL = "/goform/formiPhoneAppMute.xml?2+MuteOff"

# Zone 3 URLs
STATUS_Z3_URL = "/goform/formZone3_Zone3XmlStatus.xml"
COMMAND_SEL_SRC_Z3_URL = "/goform/formiPhoneAppDirect.xml?Z3"
COMMAND_FAV_SRC_Z3_URL = "/goform/formiPhoneAppDirect.xml?Z3"
COMMAND_POWER_ON_Z3_URL = "/goform/formiPhoneAppPower.xml?3+PowerOn"
COMMAND_POWER_STANDBY_Z3_URL = "/goform/formiPhoneAppPower.xml?3+PowerStandby"
COMMAND_VOLUME_UP_Z3_URL = "/goform/formiPhoneAppDirect.xml?Z3UP"
COMMAND_VOLUME_DOWN_Z3_URL = "/goform/formiPhoneAppDirect.xml?Z3DOWN"
COMMAND_SET_VOLUME_Z3_URL = "/goform/formiPhoneAppVolume.xml?3+{volume:.1f}"
COMMAND_MUTE_ON_Z3_URL = "/goform/formiPhoneAppMute.xml?3+MuteOn"
COMMAND_MUTE_OFF_Z3_URL = "/goform/formiPhoneAppMute.xml?3+MuteOff"

DENONAVR_URLS = ReceiverURLs(
    appcommand=APPCOMMAND_URL,
    appcommand0300=APPCOMMAND0300_URL,
    status=STATUS_URL,
    mainzone=MAINZONE_URL,
    deviceinfo=DEVICEINFO_URL,
    netaudiostatus=NETAUDIOSTATUS_URL,
    tunerstatus=TUNERSTATUS_URL,
    hdtunerstatus=HDTUNERSTATUS_URL,
    command_sel_src=COMMAND_SEL_SRC_URL,
    command_fav_src=COMMAND_FAV_SRC_URL,
    command_power_on=COMMAND_POWER_ON_URL,
    command_power_standby=COMMAND_POWER_STANDBY_URL,
    command_volume_up=COMMAND_VOLUME_UP_URL,
    command_volume_down=COMMAND_VOLUME_DOWN_URL,
    command_set_volume=COMMAND_SET_VOLUME_URL,
    command_mute_on=COMMAND_MUTE_ON_URL,
    command_mute_off=COMMAND_MUTE_OFF_URL,
    command_sel_sound_mode=COMMAND_SEL_SM_URL,
    command_neural_x_on_off=COMMAND_NEURAL_X_ON_OFF,
    command_imax_auto_off=COMMAND_IMAX_AUTO_OFF,
    command_imax_audio_settings=COMMAND_IMAX_AUDIO_SETTINGS,
    command_imax_hpf=COMMAND_IMAX_HPF,
    command_imax_lpf=COMMAND_IMAX_LPF,
    command_imax_subwoofer_mode=COMMAND_IMAX_SUBWOOFER_MODE,
    command_imax_subwoofer_output=COMMAND_IMAX_SUBWOOFER_OUTPUT,
    command_cinema_eq=COMMAND_CINEMA_EQ,
    command_netaudio_post=COMMAND_NETAUDIO_POST_URL,
    command_set_all_zone_stereo=COMMAND_SET_ZST_URL,
    command_pause=COMMAND_PAUSE,
    command_play=COMMAND_PLAY,
    command_stop=COMMAND_STOP,
    command_cusor_up=COMMAND_CURSOR_UP,
    command_cusor_down=COMMAND_CURSOR_DOWN,
    command_cusor_left=COMMAND_CURSOR_LEFT,
    command_cusor_right=COMMAND_CURSOR_RIGHT,
    command_cusor_enter=COMMAND_CURSOR_ENTER,
    command_back=COMMAND_BACK,
    command_info=COMMAND_INFO,
    command_options=COMMAND_OPTIONS,
    command_setup_open=COMMAND_SETUP_OPEN,
    command_setup_close=COMMAND_SETUP_CLOSE,
    command_setup_query=COMMAND_SETUP_QUERY,
    command_channel_level_adjust=COMMAND_CHANNEL_LEVEL_ADJUST,
    command_dimmer_toggle=COMMAND_DIMMER_TOGGLE,
    command_dimmer_set=COMMAND_DIMMER_SET,
    command_channel_volume=COMMAND_CHANNEL_VOLUME,
    command_channel_volumes_reset=COMMAND_CHANNEL_VOLUMES_RESET,
    command_subwoofer_on_off=COMMAND_SUBWOOFER_ON_OFF,
    command_subwoofer_level=COMMAND_SUBWOOFER_LEVEL,
    command_lfe=COMMAND_LFE,
    command_tactile_transducer=COMMAND_TACTILE_TRANSDUCER,
    command_tactile_transducer_level=COMMAND_TACTILE_TRANSDUCER_LEVEL,
    command_tactile_transducer_lpf=COMMAND_TACTILE_TRANSDUCER_LPF,
    command_delay_up=COMMAND_DELAY_UP,
    command_delay_down=COMMAND_DELAY_DOWN,
    command_auromatic_3d_preset=COMMAND_AUROMATIC_3D_PRESET,
    command_auromatic_3d_strength=COMMAND_AUROMATIC_3D_STRENGTH,
    command_auro_3d_mode=COMMAND_AURO_3D_MODE,
    command_dirac_filter=COMMAND_DIRAC_FILTER,
    command_eco_mode=COMMAND_ECO_MODE,
    command_lfc=COMMAND_LFC,
    command_containment_amount=COMMAND_CONTAINMENT_AMOUNT,
    command_loudness_management=COMMAND_LOUDNESS_MANAGEMENT,
    command_bass_sync=COMMAND_BASS_SYNC,
    command_dialog_enhancer=COMMAND_DIALOG_ENHANCER,
    command_hdmi_output=COMMAND_HDMI_OUTPUT,
    command_hdmi_audio_decode=COMMAND_HDMI_AUDIO_DECODE,
    command_quick_select_mode=COMMAND_QUICK_SELECT_MODE,
    command_quick_select_memory=COMMAND_QUICK_SELECT_MEMORY,
    command_smart_select_mode=COMMAND_SMART_SELECT_MODE,  # Marantz
    command_smart_select_memory=COMMAND_SMART_SELECT_MEMORY,  # Marantz
    command_auto_standby=COMMAND_AUTO_STANDBY,
    command_sleep=COMMAND_SLEEP,
    command_center_spread=COMMAND_CENTER_SPREAD,
    command_video_processing_mode=COMMAND_VIDEO_PROCESSING_MODE,
    command_room_size=COMMAND_ROOM_SIZE,
    command_status=COMMAND_STATUS,
    command_system_reset=COMMAND_SYSTEM_RESET,
    command_network_restart=COMMAND_NETWORK_RESTART,
    command_trigger=COMMAND_TRIGGER,
    command_speaker_preset=COMMAND_SPEAKER_PRESET,
    command_bluetooth_transmitter=COMMAND_BLUETOOTH_TRANSMITTER,
    command_dialog_control=COMMAND_DIALOG_CONTROL,
    command_speaker_virtualizer=COMMAND_SPEAKER_VIRTUALIZER,
    command_effect_speaker_selection=COMMAND_EFFECT_SPEAKER_SELECTION,
    command_drc=COMMAND_DRC,
    command_delay_time=COMMAND_DELAY_TIME,
    command_audio_restorer=COMMAND_AUDIO_RESTORER,
    command_remote_control_lock=COMMAND_REMOTE_CONTROL_LOCK,
    command_panel_lock=COMMAND_PANEL_LOCK,
    command_panel_and_volume_lock=COMMAND_PANEL_AND_VOLUME_LOCK,
    command_graphic_eq=COMMAND_GRAPHIC_EQ,
    command_headphone_eq=COMMAND_HEADPHONE_EQ,
    command_denon_hdmi_cec_on=COMMAND_DENON_HDMI_CEC_ON,
    command_denon_hdmi_cec_off=COMMAND_DENON_HDMI_CEC_OFF,
    command_marantz_hdmi_cec_on=COMMAND_MARANTZ_HDMI_CEC_ON,
    command_marantz_hdmi_cec_off=COMMAND_MARANTZ_HDMI_CEC_OFF,
    command_mdax=COMMAND_MDAX,  # Marantz Only
    command_dac_filter=COMMAND_DAC_FILTER,  # Marantz Only
    command_illumination=COMMAND_ILLUMINATION,  # Marantz Only
    command_auto_lip_sync=COMMAND_AUTO_LIP_SYNC,  # Marantz Only
    command_dolby_atmos_toggle_denon=COMMAND_DOLBY_ATMOS_TOGGLE_DENON,
    command_dolby_atmos_toggle_marantz=COMMAND_DOLBY_ATMOS_TOGGLE_MARANTZ,
    command_page_up_denon=COMMAND_PAGE_UP_DENON,
    command_page_down_denon=COMMAND_PAGE_DOWN_DENON,
    command_page_up_marantz=COMMAND_PAGE_UP_MARANTZ,
    command_page_down_marantz=COMMAND_PAGE_DOWN_MARANTZ,
    command_input_mode_select_denon=COMMAND_INPUT_MODE_SELECT_DENON,
    command_input_mode_auto_denon=COMMAND_INPUT_MODE_AUTO_DENON,
    command_input_mode_hdmi_denon=COMMAND_INPUT_MODE_HDMI_DENON,
    command_input_mode_digital_denon=COMMAND_INPUT_MODE_DIGITAL_DENON,
    command_input_mode_analog_denon=COMMAND_INPUT_MODE_ANALOG_DENON,
    command_input_mode_select_marantz=COMMAND_INPUT_MODE_SELECT_MARANTZ,
    command_input_mode_auto_marantz=COMMAND_INPUT_MODE_AUTO_MARANTZ,
    command_input_mode_hdmi_marantz=COMMAND_INPUT_MODE_HDMI_MARANTZ,
    command_input_mode_digital_marantz=COMMAND_INPUT_MODE_DIGITAL_MARANTZ,
    command_input_mode_analog_marantz=COMMAND_INPUT_MODE_ANALOG_MARANTZ,
)

ZONE2_URLS = ReceiverURLs(
    appcommand=APPCOMMAND_URL,
    appcommand0300=APPCOMMAND0300_URL,
    status=STATUS_Z2_URL,
    mainzone=MAINZONE_URL,
    deviceinfo=DEVICEINFO_URL,
    netaudiostatus=NETAUDIOSTATUS_URL,
    tunerstatus=TUNERSTATUS_URL,
    hdtunerstatus=HDTUNERSTATUS_URL,
    command_sel_src=COMMAND_SEL_SRC_Z2_URL,
    command_fav_src=COMMAND_FAV_SRC_Z2_URL,
    command_power_on=COMMAND_POWER_ON_Z2_URL,
    command_power_standby=COMMAND_POWER_STANDBY_Z2_URL,
    command_volume_up=COMMAND_VOLUME_UP_Z2_URL,
    command_volume_down=COMMAND_VOLUME_DOWN_Z2_URL,
    command_set_volume=COMMAND_SET_VOLUME_Z2_URL,
    command_mute_on=COMMAND_MUTE_ON_Z2_URL,
    command_mute_off=COMMAND_MUTE_OFF_Z2_URL,
    command_sel_sound_mode=COMMAND_SEL_SM_URL,
    command_neural_x_on_off=COMMAND_NEURAL_X_ON_OFF,
    command_imax_auto_off=COMMAND_IMAX_AUTO_OFF,
    command_imax_audio_settings=COMMAND_IMAX_AUDIO_SETTINGS,
    command_imax_hpf=COMMAND_IMAX_HPF,
    command_imax_lpf=COMMAND_IMAX_LPF,
    command_imax_subwoofer_mode=COMMAND_IMAX_SUBWOOFER_MODE,
    command_imax_subwoofer_output=COMMAND_IMAX_SUBWOOFER_OUTPUT,
    command_cinema_eq=COMMAND_CINEMA_EQ,
    command_netaudio_post=COMMAND_NETAUDIO_POST_URL,
    command_set_all_zone_stereo=COMMAND_SET_ZST_URL,
    command_pause=COMMAND_PAUSE,
    command_play=COMMAND_PLAY,
    command_stop=COMMAND_STOP,
    command_cusor_up=COMMAND_CURSOR_UP,
    command_cusor_down=COMMAND_CURSOR_DOWN,
    command_cusor_left=COMMAND_CURSOR_LEFT,
    command_cusor_right=COMMAND_CURSOR_RIGHT,
    command_cusor_enter=COMMAND_CURSOR_ENTER,
    command_back=COMMAND_BACK,
    command_info=COMMAND_INFO,
    command_options=COMMAND_OPTIONS,
    command_setup_open=COMMAND_SETUP_OPEN,
    command_setup_close=COMMAND_SETUP_CLOSE,
    command_setup_query=COMMAND_SETUP_QUERY,
    command_channel_level_adjust=COMMAND_CHANNEL_LEVEL_ADJUST,
    command_dimmer_toggle=COMMAND_DIMMER_TOGGLE,
    command_dimmer_set=COMMAND_DIMMER_SET,
    command_channel_volume=COMMAND_CHANNEL_VOLUME,
    command_channel_volumes_reset=COMMAND_CHANNEL_VOLUMES_RESET,
    command_subwoofer_on_off=COMMAND_SUBWOOFER_ON_OFF,
    command_subwoofer_level=COMMAND_SUBWOOFER_LEVEL,
    command_lfe=COMMAND_LFE,
    command_tactile_transducer=COMMAND_TACTILE_TRANSDUCER,
    command_tactile_transducer_level=COMMAND_TACTILE_TRANSDUCER_LEVEL,
    command_tactile_transducer_lpf=COMMAND_TACTILE_TRANSDUCER_LPF,
    command_delay_up=COMMAND_DELAY_UP,
    command_delay_down=COMMAND_DELAY_DOWN,
    command_auromatic_3d_preset=COMMAND_AUROMATIC_3D_PRESET,
    command_auromatic_3d_strength=COMMAND_AUROMATIC_3D_STRENGTH,
    command_auro_3d_mode=COMMAND_AURO_3D_MODE,
    command_dirac_filter=COMMAND_DIRAC_FILTER,
    command_eco_mode=COMMAND_ECO_MODE,
    command_lfc=COMMAND_LFC,
    command_containment_amount=COMMAND_CONTAINMENT_AMOUNT,
    command_loudness_management=COMMAND_LOUDNESS_MANAGEMENT,
    command_bass_sync=COMMAND_BASS_SYNC,
    command_dialog_enhancer=COMMAND_DIALOG_ENHANCER,
    command_hdmi_output=COMMAND_HDMI_OUTPUT,
    command_hdmi_audio_decode=COMMAND_HDMI_AUDIO_DECODE,
    command_quick_select_mode=COMMAND_QUICK_SELECT_MODE,
    command_quick_select_memory=COMMAND_QUICK_SELECT_MEMORY,
    command_smart_select_mode=COMMAND_SMART_SELECT_MODE,  # Marantz
    command_smart_select_memory=COMMAND_SMART_SELECT_MEMORY,  # Marantz
    command_auto_standby=COMMAND_AUTO_STANDBY,
    command_sleep=COMMAND_SLEEP,
    command_center_spread=COMMAND_CENTER_SPREAD,
    command_video_processing_mode=COMMAND_VIDEO_PROCESSING_MODE,
    command_room_size=COMMAND_ROOM_SIZE,
    command_status=COMMAND_STATUS,
    command_system_reset=COMMAND_SYSTEM_RESET,
    command_network_restart=COMMAND_NETWORK_RESTART,
    command_trigger=COMMAND_TRIGGER,
    command_speaker_preset=COMMAND_SPEAKER_PRESET,
    command_bluetooth_transmitter=COMMAND_BLUETOOTH_TRANSMITTER,
    command_dialog_control=COMMAND_DIALOG_CONTROL,
    command_speaker_virtualizer=COMMAND_SPEAKER_VIRTUALIZER,
    command_effect_speaker_selection=COMMAND_EFFECT_SPEAKER_SELECTION,
    command_drc=COMMAND_DRC,
    command_delay_time=COMMAND_DELAY_TIME,
    command_audio_restorer=COMMAND_AUDIO_RESTORER,
    command_remote_control_lock=COMMAND_REMOTE_CONTROL_LOCK,
    command_panel_lock=COMMAND_PANEL_LOCK,
    command_panel_and_volume_lock=COMMAND_PANEL_AND_VOLUME_LOCK,
    command_graphic_eq=COMMAND_GRAPHIC_EQ,
    command_headphone_eq=COMMAND_HEADPHONE_EQ,
    command_denon_hdmi_cec_on=COMMAND_DENON_HDMI_CEC_ON,
    command_denon_hdmi_cec_off=COMMAND_DENON_HDMI_CEC_OFF,
    command_marantz_hdmi_cec_on=COMMAND_MARANTZ_HDMI_CEC_ON,
    command_marantz_hdmi_cec_off=COMMAND_MARANTZ_HDMI_CEC_OFF,
    command_mdax=COMMAND_MDAX,  # Marantz Only
    command_dac_filter=COMMAND_DAC_FILTER,  # Marantz Only
    command_illumination=COMMAND_ILLUMINATION,  # Marantz Only
    command_auto_lip_sync=COMMAND_AUTO_LIP_SYNC,  # Marantz Only
    command_dolby_atmos_toggle_denon=COMMAND_DOLBY_ATMOS_TOGGLE_DENON,
    command_dolby_atmos_toggle_marantz=COMMAND_DOLBY_ATMOS_TOGGLE_MARANTZ,
    command_page_up_denon=COMMAND_PAGE_UP_DENON,
    command_page_down_denon=COMMAND_PAGE_DOWN_DENON,
    command_page_up_marantz=COMMAND_PAGE_UP_MARANTZ,
    command_page_down_marantz=COMMAND_PAGE_DOWN_MARANTZ,
    command_input_mode_select_denon=COMMAND_INPUT_MODE_SELECT_DENON,
    command_input_mode_auto_denon=COMMAND_INPUT_MODE_AUTO_DENON,
    command_input_mode_hdmi_denon=COMMAND_INPUT_MODE_HDMI_DENON,
    command_input_mode_digital_denon=COMMAND_INPUT_MODE_DIGITAL_DENON,
    command_input_mode_analog_denon=COMMAND_INPUT_MODE_ANALOG_DENON,
    command_input_mode_select_marantz=COMMAND_INPUT_MODE_SELECT_MARANTZ,
    command_input_mode_auto_marantz=COMMAND_INPUT_MODE_AUTO_MARANTZ,
    command_input_mode_hdmi_marantz=COMMAND_INPUT_MODE_HDMI_MARANTZ,
    command_input_mode_digital_marantz=COMMAND_INPUT_MODE_DIGITAL_MARANTZ,
    command_input_mode_analog_marantz=COMMAND_INPUT_MODE_ANALOG_MARANTZ,
)

ZONE3_URLS = ReceiverURLs(
    appcommand=APPCOMMAND_URL,
    appcommand0300=APPCOMMAND0300_URL,
    status=STATUS_Z3_URL,
    mainzone=MAINZONE_URL,
    deviceinfo=DEVICEINFO_URL,
    netaudiostatus=NETAUDIOSTATUS_URL,
    tunerstatus=TUNERSTATUS_URL,
    hdtunerstatus=HDTUNERSTATUS_URL,
    command_sel_src=COMMAND_SEL_SRC_Z3_URL,
    command_fav_src=COMMAND_FAV_SRC_Z3_URL,
    command_power_on=COMMAND_POWER_ON_Z3_URL,
    command_power_standby=COMMAND_POWER_STANDBY_Z3_URL,
    command_volume_up=COMMAND_VOLUME_UP_Z3_URL,
    command_volume_down=COMMAND_VOLUME_DOWN_Z3_URL,
    command_set_volume=COMMAND_SET_VOLUME_Z3_URL,
    command_mute_on=COMMAND_MUTE_ON_Z3_URL,
    command_mute_off=COMMAND_MUTE_OFF_Z3_URL,
    command_sel_sound_mode=COMMAND_SEL_SM_URL,
    command_neural_x_on_off=COMMAND_NEURAL_X_ON_OFF,
    command_imax_auto_off=COMMAND_IMAX_AUTO_OFF,
    command_imax_audio_settings=COMMAND_IMAX_AUDIO_SETTINGS,
    command_imax_hpf=COMMAND_IMAX_HPF,
    command_imax_lpf=COMMAND_IMAX_LPF,
    command_imax_subwoofer_mode=COMMAND_IMAX_SUBWOOFER_MODE,
    command_imax_subwoofer_output=COMMAND_IMAX_SUBWOOFER_OUTPUT,
    command_cinema_eq=COMMAND_CINEMA_EQ,
    command_netaudio_post=COMMAND_NETAUDIO_POST_URL,
    command_set_all_zone_stereo=COMMAND_SET_ZST_URL,
    command_pause=COMMAND_PAUSE,
    command_play=COMMAND_PLAY,
    command_stop=COMMAND_STOP,
    command_cusor_up=COMMAND_CURSOR_UP,
    command_cusor_down=COMMAND_CURSOR_DOWN,
    command_cusor_left=COMMAND_CURSOR_LEFT,
    command_cusor_right=COMMAND_CURSOR_RIGHT,
    command_cusor_enter=COMMAND_CURSOR_ENTER,
    command_back=COMMAND_BACK,
    command_info=COMMAND_INFO,
    command_options=COMMAND_OPTIONS,
    command_setup_open=COMMAND_SETUP_OPEN,
    command_setup_close=COMMAND_SETUP_CLOSE,
    command_setup_query=COMMAND_SETUP_QUERY,
    command_channel_level_adjust=COMMAND_CHANNEL_LEVEL_ADJUST,
    command_dimmer_toggle=COMMAND_DIMMER_TOGGLE,
    command_dimmer_set=COMMAND_DIMMER_SET,
    command_channel_volume=COMMAND_CHANNEL_VOLUME,
    command_channel_volumes_reset=COMMAND_CHANNEL_VOLUMES_RESET,
    command_subwoofer_on_off=COMMAND_SUBWOOFER_ON_OFF,
    command_subwoofer_level=COMMAND_SUBWOOFER_LEVEL,
    command_lfe=COMMAND_LFE,
    command_tactile_transducer=COMMAND_TACTILE_TRANSDUCER,
    command_tactile_transducer_level=COMMAND_TACTILE_TRANSDUCER_LEVEL,
    command_tactile_transducer_lpf=COMMAND_TACTILE_TRANSDUCER_LPF,
    command_delay_up=COMMAND_DELAY_UP,
    command_delay_down=COMMAND_DELAY_DOWN,
    command_auromatic_3d_preset=COMMAND_AUROMATIC_3D_PRESET,
    command_auromatic_3d_strength=COMMAND_AUROMATIC_3D_STRENGTH,
    command_auro_3d_mode=COMMAND_AURO_3D_MODE,
    command_dirac_filter=COMMAND_DIRAC_FILTER,
    command_eco_mode=COMMAND_ECO_MODE,
    command_lfc=COMMAND_LFC,
    command_containment_amount=COMMAND_CONTAINMENT_AMOUNT,
    command_loudness_management=COMMAND_LOUDNESS_MANAGEMENT,
    command_bass_sync=COMMAND_BASS_SYNC,
    command_dialog_enhancer=COMMAND_DIALOG_ENHANCER,
    command_hdmi_output=COMMAND_HDMI_OUTPUT,
    command_hdmi_audio_decode=COMMAND_HDMI_AUDIO_DECODE,
    command_quick_select_mode=COMMAND_QUICK_SELECT_MODE,
    command_quick_select_memory=COMMAND_QUICK_SELECT_MEMORY,
    command_smart_select_mode=COMMAND_SMART_SELECT_MODE,  # Marantz
    command_smart_select_memory=COMMAND_SMART_SELECT_MEMORY,  # Marantz
    command_auto_standby=COMMAND_AUTO_STANDBY,
    command_sleep=COMMAND_SLEEP,
    command_center_spread=COMMAND_CENTER_SPREAD,
    command_video_processing_mode=COMMAND_VIDEO_PROCESSING_MODE,
    command_room_size=COMMAND_ROOM_SIZE,
    command_status=COMMAND_STATUS,
    command_system_reset=COMMAND_SYSTEM_RESET,
    command_network_restart=COMMAND_NETWORK_RESTART,
    command_trigger=COMMAND_TRIGGER,
    command_speaker_preset=COMMAND_SPEAKER_PRESET,
    command_bluetooth_transmitter=COMMAND_BLUETOOTH_TRANSMITTER,
    command_dialog_control=COMMAND_DIALOG_CONTROL,
    command_speaker_virtualizer=COMMAND_SPEAKER_VIRTUALIZER,
    command_effect_speaker_selection=COMMAND_EFFECT_SPEAKER_SELECTION,
    command_drc=COMMAND_DRC,
    command_delay_time=COMMAND_DELAY_TIME,
    command_audio_restorer=COMMAND_AUDIO_RESTORER,
    command_remote_control_lock=COMMAND_REMOTE_CONTROL_LOCK,
    command_panel_lock=COMMAND_PANEL_LOCK,
    command_panel_and_volume_lock=COMMAND_PANEL_AND_VOLUME_LOCK,
    command_graphic_eq=COMMAND_GRAPHIC_EQ,
    command_headphone_eq=COMMAND_HEADPHONE_EQ,
    command_denon_hdmi_cec_on=COMMAND_DENON_HDMI_CEC_ON,
    command_denon_hdmi_cec_off=COMMAND_DENON_HDMI_CEC_OFF,
    command_marantz_hdmi_cec_on=COMMAND_MARANTZ_HDMI_CEC_ON,
    command_marantz_hdmi_cec_off=COMMAND_MARANTZ_HDMI_CEC_OFF,
    command_mdax=COMMAND_MDAX,  # Marantz Only
    command_dac_filter=COMMAND_DAC_FILTER,  # Marantz Only
    command_illumination=COMMAND_ILLUMINATION,  # Marantz Only
    command_auto_lip_sync=COMMAND_AUTO_LIP_SYNC,  # Marantz Only
    command_dolby_atmos_toggle_denon=COMMAND_DOLBY_ATMOS_TOGGLE_DENON,
    command_dolby_atmos_toggle_marantz=COMMAND_DOLBY_ATMOS_TOGGLE_MARANTZ,
    command_page_up_denon=COMMAND_PAGE_UP_DENON,
    command_page_down_denon=COMMAND_PAGE_DOWN_DENON,
    command_page_up_marantz=COMMAND_PAGE_UP_MARANTZ,
    command_page_down_marantz=COMMAND_PAGE_DOWN_MARANTZ,
    command_input_mode_select_denon=COMMAND_INPUT_MODE_SELECT_DENON,
    command_input_mode_auto_denon=COMMAND_INPUT_MODE_AUTO_DENON,
    command_input_mode_hdmi_denon=COMMAND_INPUT_MODE_HDMI_DENON,
    command_input_mode_digital_denon=COMMAND_INPUT_MODE_DIGITAL_DENON,
    command_input_mode_analog_denon=COMMAND_INPUT_MODE_ANALOG_DENON,
    command_input_mode_select_marantz=COMMAND_INPUT_MODE_SELECT_MARANTZ,
    command_input_mode_auto_marantz=COMMAND_INPUT_MODE_AUTO_MARANTZ,
    command_input_mode_hdmi_marantz=COMMAND_INPUT_MODE_HDMI_MARANTZ,
    command_input_mode_digital_marantz=COMMAND_INPUT_MODE_DIGITAL_MARANTZ,
    command_input_mode_analog_marantz=COMMAND_INPUT_MODE_ANALOG_MARANTZ,
)

# Telnet Events
ALL_TELNET_EVENTS = "ALL"
TELNET_EVENTS = {
    "BT",
    "CV",
    "DC",
    "DIM",
    "ECO",
    "HD",
    "ILB",  # Marantz Only
    "MN",
    "MS",
    "MU",
    "MV",
    "NS",
    "NSA",
    "NSE",
    "OP",
    "PS",
    "PV",
    "PW",
    "RM",
    "SD",
    "SI",
    "SLP",
    "SP",
    "SR",
    "SS",
    "STBY",
    "SV",
    "SY",
    "TF",
    "TM",
    "TP",
    "TR",
    "UG",
    "VS",
    "ZM",
    "Z2",
    "Z3",
}
ALL_ZONE_TELNET_EVENTS = {
    "DIM",
    "HD",
    "ILB",  # Marantz Only
    "NS",
    "NSA",
    "NSE",
    "MN",
    "PW",
    "RM",
    "SY",
    "TF",
    "TM",
    "TP",
    "TR",
    "UG",
}

DENONAVR_TELNET_COMMANDS = TelnetCommands(
    command_sel_src="SI",
    command_fav_src="ZM",
    command_power_on="ZMON",
    command_power_standby="ZMOFF",
    command_volume_up="MVUP",
    command_volume_down="MVDOWN",
    command_set_volume="MV{volume:02d}",
    command_mute_on="MUON",
    command_mute_off="MUOFF",
    command_sel_sound_mode="MS",
    command_neural_x_on_off="PSNEURAL {mode}",
    command_imax_auto_off="PSIMAX {mode}",
    command_imax_audio_settings="PSIMAXAUD {mode}",
    command_imax_hpf="PSIMAXHPF {frequency}",
    command_imax_lpf="PSIMAXLPF {frequency}",
    command_imax_subwoofer_mode="PSIMAXSWM {mode}",
    command_imax_subwoofer_output="PSIMAXSWO {mode}",
    command_cinema_eq="PSCINEMA EQ.{mode}",
    command_set_all_zone_stereo="MN",
    command_pause="NS9B",
    command_play="NS9A",
    command_stop="NS9C",
    command_multieq="PSMULTEQ:",
    command_dynamiceq="PSDYNEQ ",
    command_reflevoffset="PSREFLEV ",
    command_dynamicvol="PSDYNVOL ",
    command_tonecontrol="PSTONE CTRL ",
    command_bass="PSBAS ",
    command_treble="PSTRE ",
    command_cusor_up="MNCUP",
    command_cusor_down="MNCDN",
    command_cusor_left="MNCLT",
    command_cusor_right="MNCRT",
    command_cusor_enter="MNENT",
    command_back="MNRTN",
    command_info="MNINF",
    command_options="MNOPT",
    command_setup_open="MNMEN ON",
    command_setup_close="MNMEN OFF",
    command_setup_query="MNMEN?",
    command_channel_level_adjust="MNCHL",
    command_dimmer_toggle="DIM SEL",
    command_dimmer_set="DIM {mode}",
    command_channel_volume="CV{channel} {value}",
    command_channel_volumes_reset="CVZRL",
    command_subwoofer_on_off="PSSWR {mode}",
    command_subwoofer_level="PSSWL{number} {mode}",
    command_lfe="PSLFE {mode}",
    command_tactile_transducer="SSTTR {mode}",
    command_tactile_transducer_level="SSTTRLEV {mode}",
    command_tactile_transducer_lpf="SSTTRLPF {frequency}",
    command_delay_up="PSDELAY UP",
    command_delay_down="PSDELAY DOWN",
    command_auromatic_3d_preset="PSAUROPR {preset}",
    command_auromatic_3d_strength="PSAUROST {value}",
    command_auro_3d_mode="PSAUROMODE {mode}",
    command_dirac_filter="PSDIRAC {filter}",
    command_eco_mode="ECO{mode}",
    command_lfc="PSLFC {mode}",
    command_containment_amount="PSCNTAMT {value}",
    command_loudness_management="PSLOM {mode}",
    command_bass_sync="PSBSC {mode}",
    command_dialog_enhancer="PSDEH {level}",
    command_hdmi_output="VSMONI{output}",
    command_hdmi_audio_decode="VSAUDIO {mode}",
    command_quick_select_mode="MSQUICK{number}",
    command_quick_select_memory="MSQUICK{number} MEMORY",
    command_smart_select_mode="MSSMART{number}",  # Marantz
    command_smart_select_memory="MSSMART{number} MEMORY",  # Marantz
    command_auto_standby="STBY{mode}",
    command_sleep="SLP{value}",
    command_center_spread="PSCES {mode}",
    command_video_processing_mode="VSVPM{mode}",
    command_room_size="PSRSZ {size}",
    command_status="RCSHP0230030",
    command_system_reset="SYRST",
    command_network_restart="NSRBT",
    command_trigger="TR{number} {mode}",
    command_speaker_preset="SPPR {number}",
    command_bluetooth_transmitter="BTTX {mode}",
    command_dialog_control="PSDIC {value}",
    command_speaker_virtualizer="PSSPV {mode}",
    command_effect_speaker_selection="PSSP:{mode}",
    command_drc="PSDRC {mode}",
    command_delay_time="PSDEL {value}",
    command_audio_restorer="PSRSTR {mode}",
    command_remote_control_lock="SYREMOTE LOCK {mode}",
    command_panel_lock="SYPANEL LOCK {mode}",
    command_panel_and_volume_lock="SYPANEL+V LOCK ON",
    command_graphic_eq="PSGEQ {mode}",
    command_headphone_eq="PSHEQ {mode}",
    command_denon_hdmi_cec_on="RCKSK0410826",
    command_denon_hdmi_cec_off="RCKSK0410827",
    command_marantz_hdmi_cec_on="RCRC51608408",
    command_marantz_hdmi_cec_off="RCRC51608409",
    command_mdax="PSMDAX {mode}",  # Marantz Only
    command_dac_filter="PSDACFIL {mode}",  # Marantz Only
    command_illumination="ILB {mode}",  # Marantz Only
    command_auto_lip_sync="SSHOSALS {mode}",  # Marantz Only
    command_dolby_atmos_toggle_denon="RCKSK0410348",
    command_dolby_atmos_toggle_marantz="RCRC51606425",
    command_page_up_denon="RCKSK0410945",
    command_page_down_denon="RCKSK0410946",
    command_page_up_marantz="RCRC52708105",
    command_page_down_marantz="RCRC52708005",
    command_input_mode_select_denon="RCKSK0410352",
    command_input_mode_auto_denon="RCKSK0410358",
    command_input_mode_hdmi_denon="RCKSK0410357",
    command_input_mode_digital_denon="RCKSK0410359",
    command_input_mode_analog_denon="RCKSK0410353",
    command_input_mode_select_marantz="RCRC51600101",
    command_input_mode_auto_marantz="RCRC51600115",
    command_input_mode_hdmi_marantz="RCRC51600116",
    command_input_mode_digital_marantz="RCRC51600117",
    command_input_mode_analog_marantz="RCRC51600118",
)

ZONE2_TELNET_COMMANDS = TelnetCommands(
    command_sel_src="Z2",
    command_fav_src="Z2",
    command_power_on="Z2ON",
    command_power_standby="Z2OFF",
    command_volume_up="Z2UP",
    command_volume_down="Z2DOWN",
    command_set_volume="Z2{volume:02d}",
    command_mute_on="Z2MUON",
    command_mute_off="Z2MUOFF",
    command_sel_sound_mode="MS",
    command_neural_x_on_off="PSNEURAL {mode}",
    command_imax_auto_off="PSIMAX {mode}",
    command_imax_audio_settings="PSIMAXAUD {mode}",
    command_imax_hpf="PSIMAXHPF {frequency}",
    command_imax_lpf="PSIMAXLPF {frequency}",
    command_imax_subwoofer_mode="PSIMAXSWM {mode}",
    command_imax_subwoofer_output="PSIMAXSWO {mode}",
    command_cinema_eq="PSCINEMA EQ.{mode}",
    command_set_all_zone_stereo="MN",
    command_pause="NS9B",
    command_play="NS9A",
    command_stop="NS9C",
    command_multieq="PSMULTEQ:",
    command_dynamiceq="PSDYNEQ ",
    command_reflevoffset="PSREFLEV ",
    command_dynamicvol="PSDYNVOL ",
    command_tonecontrol="PSTONE CTRL ",
    command_bass="PSBAS ",
    command_treble="PSTRE ",
    command_cusor_up="MNCUP",
    command_cusor_down="MNCDN",
    command_cusor_left="MNCLT",
    command_cusor_right="MNCRT",
    command_cusor_enter="MNENT",
    command_back="MNRTN",
    command_info="MNINF",
    command_options="MNOPT",
    command_setup_open="MNMEN ON",
    command_setup_close="MNMEN OFF",
    command_setup_query="MNMEN?",
    command_channel_level_adjust="MNCHL",
    command_dimmer_toggle="DIM SEL",
    command_dimmer_set="DIM {mode}",
    command_channel_volume="CV{channel} {value}",
    command_channel_volumes_reset="CVZRL",
    command_subwoofer_on_off="PSSWR {mode}",
    command_subwoofer_level="PSSWL{number} {mode}",
    command_lfe="PSLFE {mode}",
    command_tactile_transducer="SSTTR {mode}",
    command_tactile_transducer_level="SSTTRLEV {mode}",
    command_tactile_transducer_lpf="SSTTRLPF {frequency}",
    command_delay_up="PSDELAY UP",
    command_delay_down="PSDELAY DOWN",
    command_auromatic_3d_preset="PSAUROPR {preset}",
    command_auromatic_3d_strength="PSAUROST {value}",
    command_auro_3d_mode="PSAUROMODE {mode}",
    command_dirac_filter="PSDIRAC {filter}",
    command_eco_mode="ECO{mode}",
    command_lfc="PSLFC {mode}",
    command_containment_amount="PSCNTAMT {value}",
    command_loudness_management="PSLOM {mode}",
    command_bass_sync="PSBSC {mode}",
    command_dialog_enhancer="PSDEH {level}",
    command_hdmi_output="VSMONI{output}",
    command_hdmi_audio_decode="VSAUDIO {mode}",
    command_quick_select_mode="MSQUICK{number}",
    command_quick_select_memory="MSQUICK{number} MEMORY",
    command_smart_select_mode="MSSMART{number}",  # Marantz
    command_smart_select_memory="MSSMART{number} MEMORY",  # Marantz
    command_auto_standby="STBY{mode}",
    command_sleep="SLP{value}",
    command_center_spread="PSCES {mode}",
    command_video_processing_mode="VSVPM{mode}",
    command_room_size="PSRSZ {size}",
    command_status="RCSHP0230030",
    command_system_reset="SYRST",
    command_network_restart="NSRBT",
    command_trigger="TR{number} {mode}",
    command_speaker_preset="SPPR {number}",
    command_bluetooth_transmitter="BTTX {mode}",
    command_dialog_control="PSDIC {value}",
    command_speaker_virtualizer="PSSPV {mode}",
    command_effect_speaker_selection="PSSP:{mode}",
    command_drc="PSDRC {mode}",
    command_delay_time="PSDEL {value}",
    command_audio_restorer="PSRSTR {mode}",
    command_remote_control_lock="SYREMOTE LOCK {mode}",
    command_panel_lock="SYPANEL LOCK {mode}",
    command_panel_and_volume_lock="SYPANEL+V LOCK ON",
    command_graphic_eq="PSGEQ {mode}",
    command_headphone_eq="PSHEQ {mode}",
    command_denon_hdmi_cec_on="RCKSK0410826",
    command_denon_hdmi_cec_off="RCKSK0410827",
    command_marantz_hdmi_cec_on="RCRC51608408",
    command_marantz_hdmi_cec_off="RCRC51608409",
    command_mdax="PSMDAX {mode}",  # Marantz Only
    command_dac_filter="PSDACFIL {mode}",  # Marantz Only
    command_illumination="ILB {mode}",  # Marantz Only
    command_auto_lip_sync="SSHOSALS {mode}",  # Marantz Only
    command_dolby_atmos_toggle_denon="RCKSK0410348",
    command_dolby_atmos_toggle_marantz="RCRC51606425",
    command_page_up_denon="RCKSK0410945",
    command_page_down_denon="RCKSK0410946",
    command_page_up_marantz="RCRC52708105",
    command_page_down_marantz="RCRC52708005",
    command_input_mode_select_denon="RCKSK0410352",
    command_input_mode_auto_denon="RCKSK0410358",
    command_input_mode_hdmi_denon="RCKSK0410357",
    command_input_mode_digital_denon="RCKSK0410359",
    command_input_mode_analog_denon="RCKSK0410353",
    command_input_mode_select_marantz="RCRC51600101",
    command_input_mode_auto_marantz="RCRC51600115",
    command_input_mode_hdmi_marantz="RCRC51600116",
    command_input_mode_digital_marantz="RCRC51600117",
    command_input_mode_analog_marantz="RCRC51600118",
)

ZONE3_TELNET_COMMANDS = TelnetCommands(
    command_sel_src="Z3",
    command_fav_src="Z3",
    command_power_on="Z3ON",
    command_power_standby="Z3OFF",
    command_volume_up="Z3UP",
    command_volume_down="Z3DOWN",
    command_set_volume="Z3{volume:02d}",
    command_mute_on="Z3MUON",
    command_mute_off="Z3MUOFF",
    command_sel_sound_mode="MS",
    command_neural_x_on_off="PSNEURAL {mode}",
    command_imax_auto_off="PSIMAX {mode}",
    command_imax_audio_settings="PSIMAXAUD {mode}",
    command_imax_hpf="PSIMAXHPF {frequency}",
    command_imax_lpf="PSIMAXLPF {frequency}",
    command_imax_subwoofer_mode="PSIMAXSWM {mode}",
    command_imax_subwoofer_output="PSIMAXSWO {mode}",
    command_cinema_eq="PSCINEMA EQ.{mode}",
    command_set_all_zone_stereo="MN",
    command_pause="NS9B",
    command_play="NS9A",
    command_stop="NS9C",
    command_multieq="PSMULTEQ:",
    command_dynamiceq="PSDYNEQ ",
    command_reflevoffset="PSREFLEV ",
    command_dynamicvol="PSDYNVOL ",
    command_tonecontrol="PSTONE CTRL ",
    command_bass="PSBAS ",
    command_treble="PSTRE ",
    command_cusor_up="MNCUP",
    command_cusor_down="MNCDN",
    command_cusor_left="MNCLT",
    command_cusor_right="MNCRT",
    command_cusor_enter="MNENT",
    command_back="MNRTN",
    command_info="MNINF",
    command_options="MNOPT",
    command_setup_open="MNMEN ON",
    command_setup_close="MNMEN OFF",
    command_setup_query="MNMEN?",
    command_channel_level_adjust="MNCHL",
    command_dimmer_toggle="DIM SEL",
    command_dimmer_set="DIM {mode}",
    command_channel_volume="CV{channel} {value}",
    command_channel_volumes_reset="CVZRL",
    command_subwoofer_on_off="PSSWR {mode}",
    command_subwoofer_level="PSSWL{number} {mode}",
    command_lfe="PSLFE {mode}",
    command_tactile_transducer="SSTTR {mode}",
    command_tactile_transducer_level="SSTTRLEV {mode}",
    command_tactile_transducer_lpf="SSTTRLPF {frequency}",
    command_delay_up="PSDELAY UP",
    command_delay_down="PSDELAY DOWN",
    command_auromatic_3d_preset="PSAUROPR {preset}",
    command_auromatic_3d_strength="PSAUROST {value}",
    command_auro_3d_mode="PSAUROMODE {mode}",
    command_dirac_filter="PSDIRAC {filter}",
    command_eco_mode="ECO{mode}",
    command_lfc="PSLFC {mode}",
    command_containment_amount="PSCNTAMT {value}",
    command_loudness_management="PSLOM {mode}",
    command_bass_sync="PSBSC {mode}",
    command_dialog_enhancer="PSDEH {level}",
    command_hdmi_output="VSMONI{output}",
    command_hdmi_audio_decode="VSAUDIO {mode}",
    command_quick_select_mode="MSQUICK{number}",
    command_quick_select_memory="MSQUICK{number} MEMORY",
    command_smart_select_mode="MSSMART{number}",  # Marantz
    command_smart_select_memory="MSSMART{number} MEMORY",  # Marantz
    command_auto_standby="STBY{mode}",
    command_sleep="SLP{value}",
    command_center_spread="PSCES {mode}",
    command_video_processing_mode="VSVPM{mode}",
    command_room_size="PSRSZ {size}",
    command_status="RCSHP0230030",
    command_system_reset="SYRST",
    command_network_restart="NSRBT",
    command_trigger="TR{number} {mode}",
    command_speaker_preset="SPPR {number}",
    command_bluetooth_transmitter="BTTX {mode}",
    command_dialog_control="PSDIC {value}",
    command_speaker_virtualizer="PSSPV {mode}",
    command_effect_speaker_selection="PSSP:{mode}",
    command_drc="PSDRC {mode}",
    command_delay_time="PSDEL {value}",
    command_audio_restorer="PSRSTR {mode}",
    command_remote_control_lock="SYREMOTE LOCK {mode}",
    command_panel_lock="SYPANEL LOCK {mode}",
    command_panel_and_volume_lock="SYPANEL+V LOCK ON",
    command_graphic_eq="PSGEQ {mode}",
    command_headphone_eq="PSHEQ {mode}",
    command_denon_hdmi_cec_on="RCKSK0410826",
    command_denon_hdmi_cec_off="RCKSK0410827",
    command_marantz_hdmi_cec_on="RCRC51608408",
    command_marantz_hdmi_cec_off="RCRC51608409",
    command_mdax="PSMDAX {mode}",  # Marantz Only
    command_dac_filter="PSDACFIL {mode}",  # Marantz Only
    command_illumination="ILB {mode}",  # Marantz Only
    command_auto_lip_sync="SSHOSALS {mode}",  # Marantz Only
    command_dolby_atmos_toggle_denon="RCKSK0410348",
    command_dolby_atmos_toggle_marantz="RCRC51606425",
    command_page_up_denon="RCKSK0410945",
    command_page_down_denon="RCKSK0410946",
    command_page_up_marantz="RCRC52708105",
    command_page_down_marantz="RCRC52708005",
    command_input_mode_select_denon="RCKSK0410352",
    command_input_mode_auto_denon="RCKSK0410358",
    command_input_mode_hdmi_denon="RCKSK0410357",
    command_input_mode_digital_denon="RCKSK0410359",
    command_input_mode_analog_denon="RCKSK0410353",
    command_input_mode_select_marantz="RCRC51600101",
    command_input_mode_auto_marantz="RCRC51600115",
    command_input_mode_hdmi_marantz="RCRC51600116",
    command_input_mode_digital_marantz="RCRC51600117",
    command_input_mode_analog_marantz="RCRC51600118",
)

# States
POWER_ON = "ON"
POWER_OFF = "OFF"
POWER_STANDBY = "STANDBY"
POWER_STATES = {POWER_ON, POWER_OFF, POWER_STANDBY}
STATE_ON = "on"
STATE_OFF = "off"
STATE_PLAYING = "playing"
STATE_PAUSED = "paused"
STATE_STOPPED = "stopped"
SETTINGS_MENU_ON = "ON"
SETTINGS_MENU_OFF = "OFF"
SETTINGS_MENU_STATES = {SETTINGS_MENU_ON, SETTINGS_MENU_OFF}
DIMER_BRIGHT = "BRI"
DIMER_DIM = "DIM"
DIMER_DARK = "DAR"
DIMER_OFF = "OFF"
DIMMER_STATES = {DIMER_BRIGHT, DIMER_DIM, DIMER_DARK, DIMER_OFF}
NETAUDIO_PLAYING = "Now Playing"

# Zones
ALL_ZONES = "All"
MAIN_ZONE = "Main"
ZONE2 = "Zone2"
ZONE3 = "Zone3"
VALID_ZONES = {MAIN_ZONE, ZONE2, ZONE3}

# Setup additional zones
NO_ZONES = None
ZONE2_NAME = {ZONE2: None}
ZONE3_NAME = {ZONE3: None}
ZONE2_ZONE3_NAME = {ZONE2: None, ZONE3: None}

# AppCommand related
APPCOMMAND_CMD_TEXT = "cmd_text"
APPCOMMAND_NAME = "name"

# _MAP dictionaries map receiver outputs to denonavr properties
# _MAP_REVERSE dictionaries map them the opposite way

MultiEQModes = Literal["Off", "Flat", "L/R Bypass", "Reference", "Manual"]
"""Multi EQ modes."""

MULTI_EQ_MAP_APPCOMMAND = {"0": "Off", "1": "Flat", "2": "L/R Bypass", "3": "Reference"}
MULTI_EQ_MAP_TELNET = {
    "OFF": "Off",
    "FLAT": "Flat",
    "BYP.LR": "L/R Bypass",
    "AUDYSSEY": "Reference",
    "MANUAL": "Manual",
}
MULTI_EQ_MAP = {**MULTI_EQ_MAP_APPCOMMAND, **MULTI_EQ_MAP_TELNET}
MULTI_EQ_MAP_REVERSE_APPCOMMAND = {
    value: key for key, value in MULTI_EQ_MAP_APPCOMMAND.items()
}
MULTI_EQ_MAP_REVERSE_TELNET = {value: key for key, value in MULTI_EQ_MAP_TELNET.items()}

ReferenceLevelOffsets = Literal["0dB", "+5dB", "+10dB", "+15dB"]
"""Reference Level Offsets."""

REF_LVL_OFFSET_MAP_APPCOMMAND = {"0": "0dB", "1": "+5dB", "2": "+10dB", "3": "+15dB"}
REF_LVL_OFFSET_MAP_TELNET = {"0": "0dB", "5": "+5dB", "10": "+10dB", "15": "+15dB"}
REF_LVL_OFFSET_MAP = {**REF_LVL_OFFSET_MAP_APPCOMMAND, **REF_LVL_OFFSET_MAP_TELNET}
REF_LVL_OFFSET_MAP_REVERSE_APPCOMMAND = {
    value: key for key, value in REF_LVL_OFFSET_MAP_APPCOMMAND.items()
}
REF_LVL_OFFSET_MAP_REVERSE_TELNET = {
    value: key for key, value in REF_LVL_OFFSET_MAP_TELNET.items()
}

DynamicVolumeSettings = Literal["Off", "Light", "Medium", "Heavy"]
"""Dynamic Volume Settings."""

DYNAMIC_VOLUME_MAP_APPCOMMAND = {"0": "Off", "1": "Light", "2": "Medium", "3": "Heavy"}
DYNAMIC_VOLUME_MAP_TELNET = {
    "OFF": "Off",
    "LIT": "Light",
    "MED": "Medium",
    "HEV": "Heavy",
}
DYNAMIC_VOLUME_MAP = {**DYNAMIC_VOLUME_MAP_APPCOMMAND, **DYNAMIC_VOLUME_MAP_TELNET}
DYNAMIC_VOLUME_MAP_REVERSE_APPCOMMAND = {
    value: key for key, value in DYNAMIC_VOLUME_MAP_APPCOMMAND.items()
}
DYNAMIC_VOLUME_MAP_REVERSE_TELNET = {
    value: key for key, value in DYNAMIC_VOLUME_MAP_TELNET.items()
}

AudioRestorers = Literal["Off", "Low", "Medium", "High"]
"""Audio Restorer settings."""

AUDIO_RESTORER_MAP = {
    "OFF": "Off",
    "LOW": "Low",
    "MED": "Medium",
    "HI": "High",
}
AUDIO_RESTORER_MAP_REVERSE = {value: key for key, value in AUDIO_RESTORER_MAP.items()}

AuroMatic3DPresets = Literal[
    "Small",
    "Medium",
    "Large",
    "Speech",
    "Movie",
]
"""Auro-Matic 3D Presets."""

AURO_MATIC_3D_PRESET_MAP = {
    "SMA": "Small",
    "MED": "Medium",
    "LAR": "Large",
    "SPE": "Speech",
    "MOV": "Movie",
}
AURO_MATIC_3D_PRESET_MAP_REVERSE = {
    value: key for key, value in AURO_MATIC_3D_PRESET_MAP.items()
}

Auro3DModes = Literal["Direct", "Channel Expansion"]
"""Auro 3D Modes."""

AURO_3D_MODE_MAP = {"DRCT": "Direct", "EXP": "Channel Expansion"}
AURO_3D_MODE_MAP_REVERSE = {value: key for key, value in AURO_3D_MODE_MAP.items()}

AutoStandbys = Literal["OFF", "15M", "30M", "60M", "2H", "4H", "8H"]

AUTO_STANDBY_MAP_APPCOMMAND = {
    "0": "OFF",
    "15": "15M",
    "30": "30M",
    "60": "60M",
    "120": "2H",
    "240": "4H",
    "480": "8H",
}
AUTO_STANDBY_MAP_TELNET = {
    "OFF": "OFF",
    "15M": "15M",
    "30M": "30M",
    "60M": "60M",
    "2H": "2H",
    "4H": "4H",
    "8H": "8H",
}
AUTO_STANDBY_MAP = {**AUTO_STANDBY_MAP_APPCOMMAND, **AUTO_STANDBY_MAP_TELNET}

BluetoothOutputModes = Literal["Bluetooth + Speakers", "Bluetooth Only"]
"""Bluetooth Output Modes."""

BLUETOOTH_OUTPUT_MODES_MAP = {
    "SP": "Bluetooth + Speakers",
    "BT": "Bluetooth Only",
}
BLUETOOTH_OUTPUT_MODES_MAP_REVERSE = {
    value: key for key, value in BLUETOOTH_OUTPUT_MODES_MAP.items()
}

DIMMER_MODE_MAP_APPCOMMAND = {
    "0": "Off",
    "1": "Dark",
    "2": "Dim",
    "3": "Bright",
}
DIMMER_MODE_MAP_TELNET = {
    DIMER_OFF: "Off",
    DIMER_DARK: "Dark",
    DIMER_DIM: "Dim",
    DIMER_BRIGHT: "Bright",
}
DIMMER_MODE_MAP = {**DIMMER_MODE_MAP_TELNET, **DIMMER_MODE_MAP_APPCOMMAND}
# Telnet literals are correct for telnet and HTTP setters
DIMMER_MODE_MAP_REVERSE = {value: key for key, value in DIMMER_MODE_MAP_TELNET.items()}

DimmerModes = Literal["Off", "Dark", "Dim", "Bright"]
"""Dimmer Modes."""

DIRAC_FILTER_MAP = {"OFF": "Off", "1": "Slot 1", "2": "Slot 2", "3": "Slot 3"}
DIRAC_FILTER_MAP_REVERSE = {value: key for key, value in DIRAC_FILTER_MAP.items()}

DiracFilters = Literal["Slot 1", "Slot 2", "Slot 3", "Off"]
"""Dirac Filters."""

ECO_MODE_MAP_APPCOMMAND = {
    "0": "Off",
    "1": "On",
    "2": "Auto",
}
ECO_MODE_MAP_TELNET = {
    "ON": "On",
    "AUTO": "Auto",
    "OFF": "Off",
}
ECO_MODE_MAP = {**ECO_MODE_MAP_APPCOMMAND, **ECO_MODE_MAP_TELNET}
# Telnet literals are correct for telnet and HTTP setters
ECO_MODE_MAP_REVERSE = {value: key for key, value in ECO_MODE_MAP_TELNET.items()}

EcoModes = Literal["On", "Auto", "Off"]
"""Eco Modes."""

EffectSpeakers = Literal[
    "Floor",
    "Front",
    "Front Height",
    "Front Wide",
    "Front Height + Front Wide",
    "Height + Floor",
    "Surround Back",
    "Surround Back + Front Height",
    "Surround Back + Front Wide",
]
"""Effect Speakers."""

EFFECT_SPEAKER_SELECTION_MAP = {
    "FL": "Floor",
    "FR": "Front",
    "FH": "Front Height",
    "FW": "Front Wide",
    "HW": "Front Height + Front Wide",
    "HF": "Height + Floor",
    "SB": "Surround Back",
    "BH": "Surround Back + Front Height",
    "BW": "Surround Back + Front Wide",
}
EFFECT_SPEAKER_SELECTION_MAP_REVERSE = {
    value: key for key, value in EFFECT_SPEAKER_SELECTION_MAP.items()
}

DRCs = Literal["AUTO", "LOW", "MID", "HI", "OFF"]
"""Dynamic Range Control (DRC) Settings."""

HDMI_OUTPUT_MAP = {
    "MONIAUTO": "Auto",
    "MONI1": "HDMI1",
    "MONI2": "HDMI2",
}
HDMI_OUTPUT_MAP_REVERSE = {
    "Auto": "AUTO",
    "HDMI1": "1",
    "HDMI2": "2",
}

HDMIOutputs = Literal["Auto", "HDMI1", "HDMI2"]
"""HDMI Output Modes."""

HDMIAudioDecodes = Literal["AMP", "TV"]

Subwoofers = Literal["Subwoofer", "Subwoofer 2", "Subwoofer 3", "Subwoofer 4"]
"""Subwoofers."""

SUBWOOFERS_MAP = {
    "SWL": "Subwoofer",
    "SWL2": "Subwoofer 2",
    "SWL3": "Subwoofer 3",
    "SWL4": "Subwoofer 4",
}

SUBWOOFERS_MAP_REVERSE = {
    "Subwoofer": "",
    "Subwoofer 2": "2",
    "Subwoofer 3": "3",
    "Subwoofer 4": "4",
}

CHANNEL_MAP = {
    "FL": "Front Left",
    "FR": "Front Right",
    "C": "Center",
    "SW": "Subwoofer",
    "SW2": "Subwoofer 2",
    "SW3": "Subwoofer 3",
    "SW4": "Subwoofer 4",
    "SL": "Surround Left",
    "SR": "Surround Right",
    "SBL": "Surround Back Left",
    "SBR": "Surround Back Right",
    "FHL": "Front Height Left",
    "FHR": "Front Height Right",
    "FWL": "Front Wide Left",
    "FWR": "Front Wide Right",
    "TFL": "Top Front Left",
    "TFR": "Top Front Right",
    "TML": "Top Middle Left",
    "TMR": "Top Middle Right",
    "TRL": "Top Rear Left",
    "TRR": "Top Rear Right",
    "RHL": "Rear Height Left",
    "RHR": "Rear Height Right",
    "FDL": "Front Dolby Left",
    "FDR": "Front Dolby Right",
    "SDL": "Surround Dolby Left",
    "SDR": "Surround Dolby Right",
    "BDL": "Back Dolby Left",
    "BDR": "Back Dolby Right",
    "SHL": "Surround Height Left",
    "SHR": "Surround Height Right",
    "TS": "Top Surround",
    "CH": "Center Height",
}
CHANNEL_MAP_REVERSE = {value: key for key, value in CHANNEL_MAP.items()}

Channels = Literal[
    "Front Left",
    "Front Right",
    "Center",
    "Subwoofer",
    "Subwoofer 2",
    "Subwoofer 3",
    "Subwoofer 4",
    "Surround Left",
    "Surround Right",
    "Surround Back Left",
    "Surround Back Right",
    "Front Height Left",
    "Front Height Right",
    "Front Wide Left",
    "Front Wide Right",
    "Top Front Left",
    "Top Front Right",
    "Top Middle Left",
    "Top Middle Right",
    "Top Rear Left",
    "Top Rear Right",
    "Rear Height Left",
    "Rear Height Right",
    "Front Dolby Left",
    "Front Dolby Right",
    "Surround Dolby Left",
    "Surround Dolby Right",
    "Back Dolby Left",
    "Back Dolby Right",
    "Surround Height Left",
    "Surround Height Right",
    "Top Surround",
    "Center Height",
]
"""Receiver Channels."""

CHANNEL_VOLUME_MAP = {
    "38": -12.0,
    "385": -11.5,
    "39": -11.0,
    "395": -10.5,
    "40": -10.0,
    "405": -9.5,
    "41": -9.0,
    "415": -8.5,
    "42": -8.0,
    "425": -7.5,
    "43": -7.0,
    "435": -6.5,
    "44": -6.0,
    "445": -5.5,
    "45": -5.0,
    "455": -4.5,
    "46": -4.0,
    "465": -3.5,
    "47": -3.0,
    "475": -2.5,
    "48": -2.0,
    "485": -1.5,
    "49": -1.0,
    "495": -0.5,
    "50": 0.0,
    "505": 0.5,
    "51": 1.0,
    "515": 1.5,
    "52": 2.0,
    "525": 2.5,
    "53": 3.0,
    "535": 3.5,
    "54": 4.0,
    "545": 4.5,
    "55": 5.0,
    "555": 5.5,
    "56": 6.0,
    "565": 6.5,
    "57": 7.0,
    "575": 7.5,
    "58": 8.0,
    "585": 8.5,
    "59": 9.0,
    "595": 9.5,
    "60": 10.0,
    "605": 10.5,
    "61": 11.0,
    "615": 11.5,
    "62": 12.0,
}
CHANNEL_VOLUME_MAP_REVERSE = {value: key for key, value in CHANNEL_VOLUME_MAP.items()}

DialogEnhancerLevels = Literal["Off", "Low", "Medium", "High"]
"""Dialog Enhancer Levels."""

DIALOG_ENHANCER_LEVEL_MAP = {
    "OFF": "Off",
    "LOW": "Low",
    "MED": "Medium",
    "HIGH": "High",
}

DIALOG_ENHANCER_LEVEL_MAP_REVERSE = {
    value: key for key, value in DIALOG_ENHANCER_LEVEL_MAP.items()
}

TransducerLPFs = Literal[
    "40 Hz",
    "60 Hz",
    "80 Hz",
    "90 Hz",
    "100 Hz",
    "110 Hz",
    "120 Hz",
    "150 Hz",
    "180 Hz",
    "200 Hz",
    "250 Hz",
]
"""Tactile Transducer Low Pass Frequencies."""

IMAXHPFs = Literal[
    "40", "60", "80", "90", "100", "110", "120", "150", "180", "200", "250"
]
"""IMAX High Pass Frequencies."""

IMAXLPFs = Literal["80", "90", "100", "110", "120", "150", "180", "200", "250"]
"""IMAX Low Pass Frequencies."""

PanelLocks = Literal["Panel", "Panel + Master Volume"]
"""Panel Lock Modes."""

RoomSizes = Literal[
    "S",
    "MS",
    "M",
    "ML",
    "L",
]
"""Room Sizes."""

VideoProcessingModes = Literal["Auto", "Game", "Movie", "Bypass"]
"""Video Processing Modes."""

VIDEO_PROCESSING_MODES_MAP = {
    "AUTO": "Auto",
    "GAME": "Game",
    "MOVI": "Movie",
    "BYP": "Bypass",
}

VIDEO_PROCESSING_MODES_MAP_REVERSE = {
    value: key for key, value in VIDEO_PROCESSING_MODES_MAP.items()
}

MDAXs = Literal["Off", "Low", "Medium", "High"]
"""M-DAX Modes."""

MDAX_MAP = {
    "OFF": "Off",
    "LOW": "Low",
    "MED": "Medium",
    "HI": "High",
}
MDAX_MAP_REVERSE = {value: key for key, value in MDAX_MAP.items()}

DACFilters = Literal["Mode 1", "Mode 2"]
"""DAC Filter Modes."""

DAC_FILTERS_MAP = {
    "MODE1": "Mode 1",
    "MODE2": "Mode 2",
}

DAC_FILTERS_MAP_REVERSE = {value: key for key, value in DAC_FILTERS_MAP.items()}

Illuminations = Literal["Auto", "Bright", "Dim", "Dark", "Off"]
"""Illumination Modes."""

ILLUMINATION_MAP = {
    "AUTO": "Auto",
    "BRI": "Bright",
    "DIM": "Dim",
    "DAR": "Dark",
    "OFF": "Off",
}

ILLUMINATION_MAP_REVERSE = {value: key for key, value in ILLUMINATION_MAP.items()}

InputModes = Literal["Select", "Auto", "HDMI", "Digital", "Analog"]
"""Input Mode Selection."""
