/*
 * config_rw.c
 */
#include <cstdlib>
#include <fstream>
#include <vector>
#include <algorithm>
#include <cstring>
#include <iostream>
#include <fstream>
#include <string>
#include <stdexcept>
#include <string>
#include <sstream>

#include "../config.h"
#include "../src/misc_utils.h"
#include "../src/common.h"

#include "config_rw.h"

using std::istream;
using std::ostream;
using std::copy;

_config config;

//read enums by casting to int
istream & operator >> (istream & in, CommonEncoderType &t) 
{
  return in >> (int&)t;
}

//interface for config entries
class ConfigRW::ConfigRwInterface
{
public:
  friend istream & operator >> (istream & in, ConfigRwInterface & e)
  {
    return e.operator >> (in);
  }
  
  friend ostream & operator << (ostream & out, ConfigRwInterface const &e) 
  {
    return e.operator << (out);
  }

private:
  virtual ostream & operator << (ostream & out) const = 0;
  virtual istream & operator >> (istream & in) = 0;
};
  
namespace
{
  
template <class T, class U>
struct ConfigRwData : public ConfigRW::ConfigRwInterface
{
public:
  ConfigRwData(T * dst, const T & default_value)
  {
    m_id = dst;
    *m_id = default_value;
  }

private:
  T *m_id;         //pointer into config

  ostream & operator << (ostream & out) const 
  {
    return out << *m_id;
  }
  
  istream & operator >> (istream & in) 
  {
    return in >> *m_id;
  }
};

// template specialization for std::string
template <>
struct ConfigRwData <std::string, const char*> : public ConfigRW::ConfigRwInterface
{
public:
  ConfigRwData(std::string* dst, const char* default_value)
  {
    m_id = dst;
    *m_id = std::string(default_value);
  }

private:
  std::string* m_id;         //pointer into config
  
  ostream & operator << (ostream & out) const 
  {
    return out << *m_id;
  }
  
  istream & operator >> (istream & in) 
  {
    while (in.peek() == ' ')
      in.ignore(1);
    istream& r = std::getline(in, *m_id, '\n');

    return r;
  }
};

// template specialization for c strings
template <>
struct ConfigRwData <char, const char*> : public ConfigRW::ConfigRwInterface
{
public:
  ConfigRwData(char * dst, const char * default_value)
  {
    m_id = dst;
    copy (default_value, default_value+strlen(default_value)+1, m_id);
  }

private:
  char * m_id;         //pointer into config

  ostream & operator << (ostream & out) const 
  {
    return out << m_id;
  }
  
  istream & operator >> (istream & in) 
  {
    while (in.peek() == ' ')
      in.ignore(1);
    
    return in.getline(m_id, 1024, '\n');
  }
};

}; //unnamed namespace

/*  map of config entries to storage location and default values
 */
template <class T, class U>
  void ConfigRW::insert_pair(const std::string& label, T * dst, const U dft)
{
  config_rw_data.push_back(std::make_pair(label, new ConfigRwData<T,U>(dst,dft)));
}

void ConfigRW::init_config_data (void)
{
  config_rw_data.reserve(42);

  // General
  insert_pair("General::WavRatio", &config.wav_ratio, 0.006f);
  insert_pair("General::Mp3Ratio", &config.mp3_ratio, 0.0008f);
  insert_pair("General::ShellForExecution", &config.shell_for_execution, "/bin/sh");
  insert_pair("General::WavPath", &config.wav_path, "./");
  insert_pair("General::Mp3Path", &config.mp3_path, "./");
  insert_pair("General::CDDBPath", &config.cddb_path, "./.cddbslave");
  insert_pair("General::WavFileNameFormat", &config.wav_file_name_format, "track%");
  insert_pair(
	      "General::Mp3FileNameFormat", &config.encoded_file_name_format, "track%"
	      );
  insert_pair(
	      "General::PrependChar", &config.prepend_char, '_'
	      );
  insert_pair(
	      "General::MakeMp3FromExistingWav", 
	      &config.encode_from_existing_wav, 0
	      );
  insert_pair(
	      "General::AskWhenFileExists", &config.ask_when_file_exists, 1
	      );
  insert_pair(
	      "General::AutoAppendExtension", 
	      &config.auto_append_extension, 1
	      );
  insert_pair(
	      "General::KeepWav", &config.keep_wav, 0
	      );
  insert_pair(
	      "General::EjectDiskWhenFinished", &config.eject_disk_when_finished, 0
	      );
  insert_pair(
			"General::LastConfigPage", &config.last_config_page, 0
			);

  // Ripper
  insert_pair(
	      "Ripper::Ripper", &config.ripper.ripper, "cdparanoia                "
	      );
  insert_pair(
	      "Ripper::Plugin", &config.ripper.plugin, "ripperX_plugin-cdparanoia"
	      );

  // Encoders
  insert_pair(
	      "Encoder::Priority", &config.encoder.priority, 10
	      );

  // MP3
  insert_pair(
	      "Encoder::Mp3::Enabled", &config.encoder.mp3.enabled, 1
	      );
  insert_pair(
	      "Encoder::Mp3::Encoder", &config.encoder.mp3.encoder, "lame"
	      );
  insert_pair(
	      "Encoder::Mp3::Type", &config.encoder.mp3.type, MP3
	      );
  insert_pair(
	      "Encoder::Mp3::Plugin", &config.encoder.mp3.plugin, "ripperX_plugin-lame"
	      );
  insert_pair(
	      "Encoder::Mp3::FullCommand", &config.encoder.mp3.full_command, "lame"
	      );
  insert_pair(
	      "Encoder::Mp3::Bitrate", &config.encoder.mp3.bitrate, 128
	      );
  insert_pair(
	      "Encoder::Mp3::UseVBR", &config.encoder.mp3.use_vbr, 1
	      );
  insert_pair(
	      "Encoder::Mp3::VBRQuality", &config.encoder.mp3.vbr_quality, 4.0
	      );
  insert_pair(
	      "Encoder::Mp3::UseHighQuality", &config.encoder.mp3.use_high_quality, 1
	      );
  insert_pair(
	      "Encoder::Mp3::UseCRC", &config.encoder.mp3.use_crc, 0
	      );
  insert_pair(
	      "Encoder::Mp3::ExtraOptions", &config.encoder.mp3.extra_options, ""
	      );

  // OGG
  insert_pair(
	      "Encoder::Ogg::Enabled", &config.encoder.ogg.enabled, 0
	      );
  insert_pair(
	      "Encoder::Ogg::Encoder", &config.encoder.ogg.encoder, "oggenc"
	      );
  insert_pair(
	      "Encoder::Ogg::Type", &config.encoder.ogg.type, OGG
	      );
  insert_pair(
	      "Encoder::Ogg::Plugin", &config.encoder.ogg.plugin, "ripperX_plugin-oggenc"
	      );
  insert_pair(
	      "Encoder::Ogg::FullCommand", &config.encoder.ogg.full_command, "oggenc"
	      );
  insert_pair(
          "Encoder::Ogg::Bitrate", &config.encoder.ogg.bitrate, 192
		  );
	insert_pair(
	      "Encoder::Ogg::UseQuality", &config.encoder.ogg.use_quality, 1
	      );
  insert_pair(
	      "Encoder::Ogg::VBRQuality", &config.encoder.ogg.vbr_quality, 3.0
	      );
  insert_pair(
	      "Encoder::Ogg::ExtraOptions", &config.encoder.ogg.extra_options, ""
	      );

  // FLAC
  insert_pair(
	      "Encoder::Flac::Enabled", &config.encoder.flac.enabled, 0
	      );
  insert_pair(
	      "Encoder::Flac::Encoder", &config.encoder.flac.encoder, "flac"
	      );
  insert_pair(
	      "Encoder::Flac::Type", &config.encoder.flac.type, FLAC
	      );
  insert_pair(
	      "Encoder::Flac::Plugin", &config.encoder.flac.plugin, "ripperX_plugin-flac"
	      );
  insert_pair(
	      "Encoder::Flac::FullCommand", &config.encoder.flac.full_command, "flac"
	      );
  insert_pair(
	      "Encoder::Flac::CompressionLevel", &config.encoder.flac.compression_level, 5
	      );
  insert_pair(
          "Encoder::Flac::ExtraOptions", &config.encoder.flac.extra_options, ""
          );

  // Musepack
  insert_pair(
	      "Encoder::Musepack::Enabled", &config.encoder.musepack.enabled, 0
	      );
  insert_pair(
	      "Encoder::Musepack::Encoder", &config.encoder.musepack.encoder, "mppenc"
	      );
  insert_pair(
	      "Encoder::Musepack::Type", &config.encoder.musepack.type, MUSE
	      );
  insert_pair(
	      "Encoder::Musepack::Plugin", &config.encoder.musepack.plugin, "ripperX_plugin-musepack"
	      );
  insert_pair(
	      "Encoder::Musepack::FullCommand", &config.encoder.musepack.full_command, "mppenc"
	      );
  insert_pair(
	      "Encoder::Musepack::VBRQuality", &config.encoder.musepack.vbr_quality, 5.00
	      );
  insert_pair(
	      "Encoder::Musepack::ExtraOptions", &config.encoder.musepack.extra_options, ""
	      );

  // OPUS
  insert_pair(
	      "Encoder::Opus::Enabled", &config.encoder.opus.enabled, 0
	      );
  insert_pair(
	      "Encoder::Opus::Encoder", &config.encoder.opus.encoder, "opusenc"
	      );
  insert_pair(
	      "Encoder::Opus::Type", &config.encoder.opus.type, OPUS
	      );
  insert_pair(
	      "Encoder::Opus::Plugin", &config.encoder.opus.plugin, "ripperX_plugin-opus"
	      );
  insert_pair(
	      "Encoder::Opus::FullCommand", &config.encoder.opus.full_command, "opusenc"
	      );
  insert_pair(
	      "Encoder::Opus::Bitrate", &config.encoder.opus.bitrate, 192
	      );
  insert_pair(
	      "Encoder::Opus::ExtraOptions", &config.encoder.opus.extra_options, ""
	      );

  // MP2
  insert_pair(
	      "Encoder::Mp2::Enabled", &config.encoder.mp2.enabled, 0
	      );
  insert_pair(
	      "Encoder::Mp2::Encoder", &config.encoder.mp2.encoder, "twolame"
	      );
  insert_pair(
	      "Encoder::Mp2::Type", &config.encoder.mp2.type, MP2
	      );
  insert_pair(
	      "Encoder::Mp2::Plugin", &config.encoder.mp2.plugin, "ripperX_plugin-toolame"
	      );
  insert_pair(
	      "Encoder::Mp2::FullCommand", &config.encoder.mp2.full_command, "toolame"
	      );
  insert_pair(
	      "Encoder::Mp2::Bitrate", &config.encoder.mp2.bitrate, 192
	      );
  insert_pair(
		  "Encoder::Mp3::UseVBR", &config.encoder.mp3.use_vbr, 1
		  );
insert_pair(
	      "Encoder::Mp2::VBRQuality", &config.encoder.mp2.vbr_quality, 0
	      );
  insert_pair(
	      "Encoder::Mp2::UseCRC", &config.encoder.mp2.use_crc, 0
	      );
  insert_pair(
	      "Encoder::Mp2::ExtraOptions", &config.encoder.mp2.extra_options, ""
	      );

    // Players
  insert_pair(
	      "CdPlayer::Play_command", &config.cd_player.play_command,
	      "cdplay %"
	      );
  insert_pair(
	      "CdPlayer::Stop_command", &config.cd_player.stop_command,
	      "cdstop"
	      );
  insert_pair(
	      "WavPlayer::Command", &config.wav_player.command,
	      "play %"
	      );

  insert_pair(
	      "Mp3Player::Command", &config.mp3_player.command,
	      "mpg123 %"
	      );

  // CDDB
  insert_pair(
	      "CDDBConfig::Server", &config.cddb_config.server,
	      "gnudb.gnudb.org/~cddb/cddb.cgi"
	      );
  insert_pair(
	      "CDDBConfig::Port", &config.cddb_config.port, 80
	      );
  insert_pair(
	      "CDDBConfig::UseHttp", &config.cddb_config.use_http, 1
	      );
  insert_pair(
	      "CDDBConfig::ProxyServer", &config.cddb_config.proxy_server,
	      ""
	      );
  insert_pair(
	      "CDDBConfig::ProxyPort", &config.cddb_config.proxy_port, 8080
	      );
  insert_pair(
	      "CDDBConfig::ConvertSpaces", &config.cddb_config.convert_spaces, 0
	      );
  insert_pair(
	      "CDDBConfig::MakeDirectories", 
	      &config.cddb_config.make_directories, 1
	      );
  insert_pair(
	      "CDDBConfig::CreateID3", &config.cddb_config.create_id3, 1
	      );
  insert_pair(
	      "CDDBConfig::SplitTitle", &config.cddb_config.split_title, 1
	      );
  insert_pair(
	      "CDDBConfig::CreatePlaylist", 
	      &config.cddb_config.create_playlist, 1
	      );
  insert_pair(
	      "CDDBConfig::AutoLookup", &config.cddb_config.auto_lookup, 0
	      );
  insert_pair(
	      "CDDBConfig::FormatString", &config.cddb_config.format_string,
	      "%a - %s"
	      );
  insert_pair(
	      "CDDBConfig::DirFormatString", 
	      &config.cddb_config.dir_format_string,
	      "%a - %v"
	      );

  sort(config_rw_data.begin(),config_rw_data.end());
}

void ConfigRW::write_config(void)
{
  using std::ofstream;
  using std::endl;
  
#ifdef TEST_CONFIG_MAIN
  ofstream file(".ripperX3rc.test");
#else
  ofstream file(construct_file_name(std::getenv("HOME"), ".ripperX3rc"));
#endif

  //check error, see if we should create, etc...
  file << "//\n";
  file << "// ~/.ripperX3rc\n";
  file << "// This is the resource file for ripperX.\n";
  file << "// If you edit this file with an editor, note\n";
  file << "// that this file is overwritten each time ripperX is run.\n";
  file << "//\n// You can configure everything in the config menu within ripperX.\n";
  file << "//\n\n";
  file << "//-v " << VERSION << "\n\n";
  for (config_rw_type::iterator it=config_rw_data.begin(); 
       it != config_rw_data.end(); 
       ++it) 
    {
      file << it->first << " = " << *it->second << std::endl;
    }
}

void ConfigRW::read_config(void)
{
  using std::cout;
  using std::endl;
  using std::string;

#ifdef TEST_CONFIG_MAIN
  std::ifstream file(".ripperX3rc.test");
#else
  std::ifstream file(construct_file_name(getenv("HOME"), ".ripperX3rc"));
#endif

  //each non-comment line is label = value \n
  string label, eq, value;
  while (file >> label) 
  {
    if (not label.compare(0, 2, "//"))
	{
	  file.ignore(1024,'\n');
	  continue;
	}

    if (!(file >> eq) or eq.compare("="))
      throw std::runtime_error("error parsing config file");

    config_rw_type::iterator entry =
	  lower_bound (
		     config_rw_data.begin(),
		     config_rw_data.end(),
		     std::make_pair(label, static_cast<ConfigRwInterface*>(0))
		     );
    if ((entry == config_rw_data.end()) or (entry->first != label))
	{
      // throw std::runtime_error(string("Invalid config entry: ")+label);
	  std::cout << "Invalid config entry: " << label << std::endl;
	  file.ignore(1024,'\n');
	}
    else if (entry->first == label)
	{
      //std::getline(file >> entry->second, '\n');
      file >> *(entry->second);
	  //TODO error check,...          
	}
  }
}

#ifdef TEST_CONFIG_MAIN
int main (void) 
{
  ConfigRW data;
  data.read_config();
  data.write_config();
}
#endif
