/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common.ssl;

import java.io.IOException;
import java.nio.file.Path;
import java.security.AccessControlException;
import java.security.GeneralSecurityException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.PrivateKey;
import java.security.UnrecoverableKeyException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Enumeration;
import java.util.List;
import java.util.Objects;
import java.util.function.Function;
import javax.net.ssl.TrustManagerFactory;
import javax.net.ssl.X509ExtendedKeyManager;
import org.elasticsearch.common.ssl.KeyStoreUtil;
import org.elasticsearch.common.ssl.SslConfigException;
import org.elasticsearch.common.ssl.SslFileUtil;
import org.elasticsearch.common.ssl.SslKeyConfig;
import org.elasticsearch.common.ssl.SslTrustConfig;
import org.elasticsearch.common.ssl.StoreTrustConfig;
import org.elasticsearch.common.ssl.StoredCertificate;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.Tuple;

public class StoreKeyConfig
implements SslKeyConfig {
    private final String keystorePath;
    private final String type;
    private final char[] storePassword;
    private final Function<KeyStore, KeyStore> filter;
    private final char[] keyPassword;
    private final String algorithm;
    private final Path configBasePath;

    public StoreKeyConfig(String path, char[] storePassword, String type, @Nullable Function<KeyStore, KeyStore> filter, char[] keyPassword, String algorithm, Path configBasePath) {
        this.keystorePath = Objects.requireNonNull(path, "Keystore path cannot be null");
        this.storePassword = Objects.requireNonNull(storePassword, "Keystore password cannot be null (but may be empty)");
        this.type = Objects.requireNonNull(type, "Keystore type cannot be null");
        this.filter = filter;
        this.keyPassword = Objects.requireNonNull(keyPassword, "Key password cannot be null (but may be empty)");
        this.algorithm = Objects.requireNonNull(algorithm, "Keystore algorithm cannot be null");
        this.configBasePath = Objects.requireNonNull(configBasePath, "Config path cannot be null");
    }

    @Override
    public SslTrustConfig asTrustConfig() {
        String trustStoreAlgorithm = TrustManagerFactory.getDefaultAlgorithm();
        return new StoreTrustConfig(this.keystorePath, this.storePassword, this.type, trustStoreAlgorithm, false, this.configBasePath);
    }

    @Override
    public Collection<Path> getDependentFiles() {
        return List.of(this.resolvePath());
    }

    @Override
    public boolean hasKeyMaterial() {
        return true;
    }

    private Path resolvePath() {
        return this.configBasePath.resolve(this.keystorePath);
    }

    @Override
    public List<Tuple<PrivateKey, X509Certificate>> getKeys() {
        return this.getKeys(false);
    }

    public List<Tuple<PrivateKey, X509Certificate>> getKeys(boolean filterKeystore) {
        Path path = this.resolvePath();
        KeyStore keyStore = this.readKeyStore(path);
        if (filterKeystore) {
            keyStore = this.processKeyStore(keyStore);
        }
        return KeyStoreUtil.stream(keyStore, ex -> this.keystoreException(path, (GeneralSecurityException)ex)).filter(KeyStoreUtil.KeyStoreEntry::isKeyEntry).map(entry -> {
            X509Certificate certificate = entry.getX509Certificate();
            if (certificate != null) {
                return new Tuple((Object)entry.getKey(this.keyPassword), (Object)certificate);
            }
            return null;
        }).filter(Objects::nonNull).toList();
    }

    @Override
    public Collection<StoredCertificate> getConfiguredCertificates() {
        Path path = this.resolvePath();
        KeyStore keyStore = this.readKeyStore(path);
        return KeyStoreUtil.stream(keyStore, ex -> this.keystoreException(path, (GeneralSecurityException)ex)).flatMap(entry -> {
            ArrayList<StoredCertificate> certificates = new ArrayList<StoredCertificate>();
            boolean firstElement = true;
            for (X509Certificate x509Certificate : entry.getX509CertificateChain()) {
                certificates.add(new StoredCertificate(x509Certificate, this.keystorePath, this.type, entry.getAlias(), firstElement));
                firstElement = false;
            }
            return certificates.stream();
        }).toList();
    }

    @Override
    public X509ExtendedKeyManager createKeyManager() {
        Path path = this.resolvePath();
        return this.createKeyManager(path);
    }

    private X509ExtendedKeyManager createKeyManager(Path path) {
        try {
            KeyStore keyStore = this.readKeyStore(path);
            keyStore = this.processKeyStore(keyStore);
            StoreKeyConfig.checkKeyStore(keyStore, path);
            return KeyStoreUtil.createKeyManager(keyStore, this.keyPassword, this.algorithm);
        }
        catch (GeneralSecurityException e) {
            throw this.keystoreException(path, e);
        }
    }

    private KeyStore processKeyStore(KeyStore keyStore) {
        if (this.filter == null) {
            return keyStore;
        }
        return Objects.requireNonNull(this.filter.apply(keyStore), "A keystore filter may not return null");
    }

    private KeyStore readKeyStore(Path path) {
        try {
            return KeyStoreUtil.readKeyStore(path, this.type, this.storePassword);
        }
        catch (AccessControlException e) {
            throw SslFileUtil.accessControlFailure("[" + this.type + "] keystore", List.of(path), e, this.configBasePath);
        }
        catch (IOException e) {
            throw SslFileUtil.ioException("[" + this.type + "] keystore", List.of(path), e);
        }
        catch (GeneralSecurityException e) {
            throw this.keystoreException(path, e);
        }
    }

    private SslConfigException keystoreException(Path path, GeneralSecurityException e) {
        Object extra = null;
        if (e instanceof UnrecoverableKeyException) {
            extra = "this is usually caused by an incorrect key-password";
            if (this.keyPassword.length == 0) {
                extra = (String)extra + " (no key-password was provided)";
            } else if (Arrays.equals(this.storePassword, this.keyPassword)) {
                extra = (String)extra + " (we tried to access the key using the same password as the keystore)";
            }
        }
        return SslFileUtil.securityException("[" + this.type + "] keystore", path == null ? List.of() : List.of(path), e, (String)extra);
    }

    private static void checkKeyStore(KeyStore keyStore, Path path) throws KeyStoreException {
        Enumeration<String> aliases = keyStore.aliases();
        while (aliases.hasMoreElements()) {
            String alias = aliases.nextElement();
            if (!keyStore.isKeyEntry(alias)) continue;
            return;
        }
        String message = "the " + keyStore.getType() + " keystore";
        if (path != null) {
            message = message + " [" + String.valueOf(path) + "]";
        }
        message = message + "does not contain a private key entry";
        throw new SslConfigException(message);
    }

    public String toString() {
        StringBuilder sb = new StringBuilder(this.getClass().getSimpleName());
        sb.append('{');
        String path = this.keystorePath;
        if (path != null) {
            sb.append("path=").append(path).append(", ");
        }
        sb.append("type=").append(this.type);
        sb.append(", storePassword=").append(this.storePassword.length == 0 ? "<empty>" : "<non-empty>");
        sb.append(", keyPassword=");
        if (this.keyPassword.length == 0) {
            sb.append("<empty>");
        } else if (Arrays.equals(this.storePassword, this.keyPassword)) {
            sb.append("<same-as-store-password>");
        } else {
            sb.append("<set>");
        }
        sb.append(", algorithm=").append(this.algorithm);
        sb.append('}');
        return sb.toString();
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        StoreKeyConfig that = (StoreKeyConfig)o;
        return this.keystorePath.equals(that.keystorePath) && this.type.equals(that.type) && this.algorithm.equals(that.algorithm) && Arrays.equals(this.storePassword, that.storePassword) && Arrays.equals(this.keyPassword, that.keyPassword);
    }

    public int hashCode() {
        int result = Objects.hash(this.keystorePath, this.type, this.algorithm);
        result = 31 * result + Arrays.hashCode(this.storePassword);
        result = 31 * result + Arrays.hashCode(this.keyPassword);
        return result;
    }
}

