/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.watcher;

import java.io.IOException;
import java.time.Clock;
import java.time.Instant;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.ElasticsearchParseException;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateListener;
import org.elasticsearch.cluster.block.ClusterBlockLevel;
import org.elasticsearch.cluster.metadata.IndexMetadata;
import org.elasticsearch.cluster.routing.AllocationId;
import org.elasticsearch.cluster.routing.IndexRoutingTable;
import org.elasticsearch.cluster.routing.Murmur3HashFunction;
import org.elasticsearch.cluster.routing.RoutingNode;
import org.elasticsearch.cluster.routing.ShardRouting;
import org.elasticsearch.cluster.routing.ShardRoutingState;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.util.Maps;
import org.elasticsearch.common.util.set.Sets;
import org.elasticsearch.index.engine.Engine;
import org.elasticsearch.index.shard.IndexingOperationListener;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.xcontent.XContentType;
import org.elasticsearch.xpack.core.watcher.WatcherState;
import org.elasticsearch.xpack.core.watcher.watch.Watch;
import org.elasticsearch.xpack.watcher.trigger.TriggerService;
import org.elasticsearch.xpack.watcher.watch.WatchParser;
import org.elasticsearch.xpack.watcher.watch.WatchStoreUtils;

final class WatcherIndexingListener
implements IndexingOperationListener,
ClusterStateListener {
    private static final Logger logger = LogManager.getLogger(WatcherIndexingListener.class);
    static final Configuration INACTIVE = new Configuration(null, Collections.emptyMap());
    private final WatchParser parser;
    private final Clock clock;
    private final TriggerService triggerService;
    private final Supplier<WatcherState> watcherState;
    private volatile Configuration configuration = INACTIVE;

    WatcherIndexingListener(WatchParser parser, Clock clock, TriggerService triggerService, Supplier<WatcherState> watcherState) {
        this.parser = parser;
        this.clock = clock;
        this.triggerService = triggerService;
        this.watcherState = watcherState;
    }

    Configuration getConfiguration() {
        return this.configuration;
    }

    void setConfiguration(Configuration configuration) {
        this.configuration = configuration;
    }

    public void postIndex(ShardId shardId, Engine.Index operation, Engine.IndexResult result) {
        if (this.isWatchDocument(shardId.getIndexName())) {
            if (result.getResultType() == Engine.Result.Type.FAILURE) {
                this.postIndex(shardId, operation, result.getFailure());
                return;
            }
            ZonedDateTime now = Instant.ofEpochMilli(this.clock.millis()).atZone(ZoneOffset.UTC);
            try {
                Watch watch = this.parser.parseWithSecrets(operation.id(), true, operation.source(), now, XContentType.JSON, operation.getIfSeqNo(), operation.getIfPrimaryTerm());
                ShardAllocationConfiguration shardAllocationConfiguration = this.configuration.localShards.get(shardId);
                if (shardAllocationConfiguration == null) {
                    logger.debug("no distributed watch execution info found for watch [{}] on shard [{}], got configuration for {}", (Object)watch.id(), (Object)shardId, this.configuration.localShards.keySet());
                    return;
                }
                boolean shouldBeTriggered = shardAllocationConfiguration.shouldBeTriggered(watch.id());
                WatcherState currentState = this.watcherState.get();
                if (shouldBeTriggered && !EnumSet.of(WatcherState.STOPPING, WatcherState.STOPPED).contains(currentState)) {
                    if (watch.status().state().isActive()) {
                        logger.debug("adding watch [{}] to trigger service", (Object)watch.id());
                        this.triggerService.add(watch);
                    } else {
                        logger.debug("removing watch [{}] from trigger service", (Object)watch.id());
                        this.triggerService.remove(watch.id());
                    }
                } else {
                    logger.debug("watch [{}] should not be triggered. watcher state [{}]", (Object)watch.id(), (Object)currentState);
                }
            }
            catch (IOException e) {
                throw new ElasticsearchParseException("Could not parse watch with id [{}]", (Throwable)e, new Object[]{operation.id()});
            }
        }
    }

    public void postIndex(ShardId shardId, Engine.Index index, Exception ex) {
        if (this.isWatchDocument(shardId.getIndexName())) {
            logger.debug(() -> "failed to add watch [" + index.id() + "] to trigger service", (Throwable)ex);
        }
    }

    public Engine.Delete preDelete(ShardId shardId, Engine.Delete delete) {
        if (this.isWatchDocument(shardId.getIndexName())) {
            logger.debug("removing watch [{}] from trigger service via delete", (Object)delete.id());
            this.triggerService.remove(delete.id());
        }
        return delete;
    }

    private boolean isWatchDocument(String index) {
        return this.configuration.isIndexAndActive(index);
    }

    public void clusterChanged(ClusterChangedEvent event) {
        if (Strings.isNullOrEmpty((String)event.state().nodes().getMasterNodeId()) || event.state().getBlocks().hasGlobalBlockWithLevel(ClusterBlockLevel.WRITE)) {
            this.configuration = INACTIVE;
            return;
        }
        if (event.state().nodes().getLocalNode().canContainData() && event.metadataChanged()) {
            try {
                IndexMetadata metadata = WatchStoreUtils.getConcreteIndex(".watches", event.state().metadata());
                if (metadata == null) {
                    this.configuration = INACTIVE;
                } else {
                    this.checkWatchIndexHasChanged(metadata, event);
                }
            }
            catch (IllegalStateException e) {
                logger.error("error loading watches index", (Throwable)e);
                this.configuration = INACTIVE;
            }
        }
    }

    private void checkWatchIndexHasChanged(IndexMetadata metadata, ClusterChangedEvent event) {
        String watchIndex = metadata.getIndex().getName();
        ClusterState state = event.state();
        String localNodeId = state.nodes().getLocalNode().getId();
        RoutingNode routingNode = state.getRoutingNodes().node(localNodeId);
        List<ShardRouting> localShardRouting = routingNode.shardsWithState(watchIndex, new ShardRoutingState[]{ShardRoutingState.STARTED, ShardRoutingState.RELOCATING}).toList();
        if (localShardRouting.isEmpty()) {
            this.configuration = INACTIVE;
        } else {
            this.reloadConfiguration(watchIndex, localShardRouting, event);
        }
    }

    private void reloadConfiguration(String watchIndex, List<ShardRouting> localShardRouting, ClusterChangedEvent event) {
        boolean isAliasChanged;
        boolean bl = isAliasChanged = !watchIndex.equals(this.configuration.index);
        if (isAliasChanged || this.hasShardAllocationIdChanged(watchIndex, event.state())) {
            IndexRoutingTable watchIndexRoutingTable = event.state().routingTable().index(watchIndex);
            Map<ShardId, ShardAllocationConfiguration> ids = WatcherIndexingListener.getLocalShardAllocationIds(localShardRouting, watchIndexRoutingTable);
            this.configuration = new Configuration(watchIndex, ids);
        }
    }

    private boolean hasShardAllocationIdChanged(String watchIndex, ClusterState state) {
        HashSet<ShardId> configuredLocalShardIds;
        List allStartedRelocatedShards = state.getRoutingTable().index(watchIndex).shardsWithState(ShardRoutingState.STARTED);
        allStartedRelocatedShards.addAll(state.getRoutingTable().index(watchIndex).shardsWithState(ShardRoutingState.RELOCATING));
        if (!allStartedRelocatedShards.isEmpty() && this.configuration == INACTIVE) {
            return true;
        }
        String localNodeId = state.nodes().getLocalNodeId();
        Set clusterStateLocalShardIds = state.getRoutingNodes().node(localNodeId).shardsWithState(watchIndex, new ShardRoutingState[]{ShardRoutingState.STARTED, ShardRoutingState.RELOCATING}).map(ShardRouting::shardId).collect(Collectors.toSet());
        Set differenceSet = Sets.difference(clusterStateLocalShardIds, configuredLocalShardIds = new HashSet<ShardId>(this.configuration.localShards.keySet()));
        if (!differenceSet.isEmpty()) {
            return true;
        }
        Map<ShardId, List> shards = allStartedRelocatedShards.stream().collect(Collectors.groupingBy(ShardRouting::shardId, Collectors.mapping(sr -> sr.allocationId().getId(), Collectors.toCollection(ArrayList::new))));
        shards.values().forEach(Collections::sort);
        for (Map.Entry<ShardId, ShardAllocationConfiguration> entry : this.configuration.localShards.entrySet()) {
            if (!shards.containsKey(entry.getKey())) {
                return true;
            }
            Collection allocationIds = shards.get(entry.getKey());
            if (allocationIds.equals(entry.getValue().allocationIds)) continue;
            return true;
        }
        return false;
    }

    static Map<ShardId, ShardAllocationConfiguration> getLocalShardAllocationIds(List<ShardRouting> localShards, IndexRoutingTable routingTable) {
        Map data = Maps.newMapWithExpectedSize((int)localShards.size());
        for (ShardRouting shardRouting : localShards) {
            ShardId shardId = shardRouting.shardId();
            List<String> allocationIds = routingTable.shard(shardId.getId()).activeShards().stream().map(ShardRouting::allocationId).map(AllocationId::getId).sorted().toList();
            String allocationId = shardRouting.allocationId().getId();
            int idx = allocationIds.indexOf(allocationId);
            data.put(shardId, new ShardAllocationConfiguration(idx, allocationIds.size(), allocationIds));
        }
        return data;
    }

    static final class Configuration {
        final Map<ShardId, ShardAllocationConfiguration> localShards;
        final boolean active;
        final String index;

        Configuration(String index, Map<ShardId, ShardAllocationConfiguration> localShards) {
            this.active = !localShards.isEmpty();
            this.index = index;
            this.localShards = Collections.unmodifiableMap(localShards);
        }

        public boolean isIndexAndActive(String index) {
            return this.active && index.equals(this.index);
        }
    }

    static final class ShardAllocationConfiguration {
        final int index;
        final int shardCount;
        final List<String> allocationIds;

        ShardAllocationConfiguration(int index, int shardCount, List<String> allocationIds) {
            this.index = index;
            this.shardCount = shardCount;
            this.allocationIds = allocationIds;
        }

        public boolean shouldBeTriggered(String id) {
            int hash = Murmur3HashFunction.hash((String)id);
            int shardIndex = Math.floorMod(hash, this.shardCount);
            return shardIndex == this.index;
        }
    }
}

