/* SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * Copyright:
 *   2020      Evan Nemerson <evan@nemerson.com>
 *   2020      kitegi <kitegi@users.noreply.github.com>
 */

#include <stddef.h>
#define SIMDE_TEST_X86_AVX512_INSN fnmadd
#include "../../test.h"
#include "test-avx512.h"
#include <simde/simde-common.h>
#include <simde/x86/avx512/types.h>
#include <simde/x86/avx512/loadu.h>
#include <simde/x86/avx512/fnmadd.h>

static int
test_simde_mm512_fnmadd_ps (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde_float32 a[16];
    const simde_float32 b[16];
    const simde_float32 c[16];
    const simde_float32 r[16];
  } test_vec[] = {
    { { SIMDE_FLOAT32_C(   -61.24), SIMDE_FLOAT32_C(    66.05), SIMDE_FLOAT32_C(   -37.34), SIMDE_FLOAT32_C(   -97.14),
        SIMDE_FLOAT32_C(    46.24), SIMDE_FLOAT32_C(    -6.81), SIMDE_FLOAT32_C(    13.43), SIMDE_FLOAT32_C(   -17.56),
        SIMDE_FLOAT32_C(   -58.55), SIMDE_FLOAT32_C(   -25.60), SIMDE_FLOAT32_C(   -35.79), SIMDE_FLOAT32_C(    89.11),
        SIMDE_FLOAT32_C(    42.18), SIMDE_FLOAT32_C(    22.19), SIMDE_FLOAT32_C(    87.14), SIMDE_FLOAT32_C(   -55.50) },
      { SIMDE_FLOAT32_C(    51.65), SIMDE_FLOAT32_C(   -43.85), SIMDE_FLOAT32_C(   -44.69), SIMDE_FLOAT32_C(     2.09),
        SIMDE_FLOAT32_C(    37.82), SIMDE_FLOAT32_C(     8.27), SIMDE_FLOAT32_C(   -31.95), SIMDE_FLOAT32_C(    70.84),
        SIMDE_FLOAT32_C(   -62.34), SIMDE_FLOAT32_C(    50.49), SIMDE_FLOAT32_C(   -46.36), SIMDE_FLOAT32_C(   -70.84),
        SIMDE_FLOAT32_C(    55.23), SIMDE_FLOAT32_C(   -80.94), SIMDE_FLOAT32_C(    23.06), SIMDE_FLOAT32_C(    93.98) },
      { SIMDE_FLOAT32_C(    85.11), SIMDE_FLOAT32_C(    85.72), SIMDE_FLOAT32_C(    96.84), SIMDE_FLOAT32_C(    31.35),
        SIMDE_FLOAT32_C(   -21.08), SIMDE_FLOAT32_C(    10.28), SIMDE_FLOAT32_C(   -86.21), SIMDE_FLOAT32_C(    20.37),
        SIMDE_FLOAT32_C(    84.68), SIMDE_FLOAT32_C(   -22.01), SIMDE_FLOAT32_C(     9.48), SIMDE_FLOAT32_C(    26.86),
        SIMDE_FLOAT32_C(   -99.81), SIMDE_FLOAT32_C(    -3.39), SIMDE_FLOAT32_C(    71.36), SIMDE_FLOAT32_C(    51.84) },
      { SIMDE_FLOAT32_C(  3248.16), SIMDE_FLOAT32_C(  2982.01), SIMDE_FLOAT32_C( -1571.88), SIMDE_FLOAT32_C(   234.37),
        SIMDE_FLOAT32_C( -1769.88), SIMDE_FLOAT32_C(    66.60), SIMDE_FLOAT32_C(   342.88), SIMDE_FLOAT32_C(  1264.32),
        SIMDE_FLOAT32_C( -3565.33), SIMDE_FLOAT32_C(  1270.53), SIMDE_FLOAT32_C( -1649.74), SIMDE_FLOAT32_C(  6339.41),
        SIMDE_FLOAT32_C( -2429.41), SIMDE_FLOAT32_C(  1792.67), SIMDE_FLOAT32_C( -1938.09), SIMDE_FLOAT32_C(  5267.73) } },
    { { SIMDE_FLOAT32_C(    52.77), SIMDE_FLOAT32_C(   -73.33), SIMDE_FLOAT32_C(   -46.07), SIMDE_FLOAT32_C(    -9.41),
        SIMDE_FLOAT32_C(    34.95), SIMDE_FLOAT32_C(    21.97), SIMDE_FLOAT32_C(   -38.57), SIMDE_FLOAT32_C(    72.60),
        SIMDE_FLOAT32_C(   -27.54), SIMDE_FLOAT32_C(    15.07), SIMDE_FLOAT32_C(   -98.24), SIMDE_FLOAT32_C(   -72.31),
        SIMDE_FLOAT32_C(    34.13), SIMDE_FLOAT32_C(    24.83), SIMDE_FLOAT32_C(   -78.32), SIMDE_FLOAT32_C(    19.24) },
      { SIMDE_FLOAT32_C(    10.55), SIMDE_FLOAT32_C(   -81.48), SIMDE_FLOAT32_C(   -49.41), SIMDE_FLOAT32_C(    89.47),
        SIMDE_FLOAT32_C(    28.80), SIMDE_FLOAT32_C(   -35.62), SIMDE_FLOAT32_C(     9.83), SIMDE_FLOAT32_C(    13.47),
        SIMDE_FLOAT32_C(    42.37), SIMDE_FLOAT32_C(   -80.69), SIMDE_FLOAT32_C(   -59.67), SIMDE_FLOAT32_C(    42.56),
        SIMDE_FLOAT32_C(    15.92), SIMDE_FLOAT32_C(   -88.31), SIMDE_FLOAT32_C(    -5.60), SIMDE_FLOAT32_C(   -31.31) },
      { SIMDE_FLOAT32_C(   -61.63), SIMDE_FLOAT32_C(    48.33), SIMDE_FLOAT32_C(    59.28), SIMDE_FLOAT32_C(    73.32),
        SIMDE_FLOAT32_C(   -29.70), SIMDE_FLOAT32_C(   -79.29), SIMDE_FLOAT32_C(    45.92), SIMDE_FLOAT32_C(    42.77),
        SIMDE_FLOAT32_C(    35.79), SIMDE_FLOAT32_C(    47.68), SIMDE_FLOAT32_C(    70.46), SIMDE_FLOAT32_C(   -30.08),
        SIMDE_FLOAT32_C(   -27.50), SIMDE_FLOAT32_C(    92.14), SIMDE_FLOAT32_C(    89.16), SIMDE_FLOAT32_C(    83.05) },
      { SIMDE_FLOAT32_C(  -618.35), SIMDE_FLOAT32_C( -5926.60), SIMDE_FLOAT32_C( -2217.04), SIMDE_FLOAT32_C(   915.23),
        SIMDE_FLOAT32_C( -1036.26), SIMDE_FLOAT32_C(   703.28), SIMDE_FLOAT32_C(   425.06), SIMDE_FLOAT32_C(  -935.15),
        SIMDE_FLOAT32_C(  1202.66), SIMDE_FLOAT32_C(  1263.68), SIMDE_FLOAT32_C( -5791.52), SIMDE_FLOAT32_C(  3047.43),
        SIMDE_FLOAT32_C(  -570.85), SIMDE_FLOAT32_C(  2284.88), SIMDE_FLOAT32_C(  -349.43), SIMDE_FLOAT32_C(   685.45) } },
    { { SIMDE_FLOAT32_C(   -89.34), SIMDE_FLOAT32_C(   -60.25), SIMDE_FLOAT32_C(    72.52), SIMDE_FLOAT32_C(    39.45),
        SIMDE_FLOAT32_C(     4.13), SIMDE_FLOAT32_C(   -17.65), SIMDE_FLOAT32_C(   -47.07), SIMDE_FLOAT32_C(   -53.49),
        SIMDE_FLOAT32_C(     1.66), SIMDE_FLOAT32_C(    -6.74), SIMDE_FLOAT32_C(    89.07), SIMDE_FLOAT32_C(   -82.41),
        SIMDE_FLOAT32_C(     4.96), SIMDE_FLOAT32_C(   -16.53), SIMDE_FLOAT32_C(   -13.72), SIMDE_FLOAT32_C(    43.33) },
      { SIMDE_FLOAT32_C(   -68.20), SIMDE_FLOAT32_C(   -54.44), SIMDE_FLOAT32_C(    16.64), SIMDE_FLOAT32_C(     2.10),
        SIMDE_FLOAT32_C(   -33.73), SIMDE_FLOAT32_C(   -37.44), SIMDE_FLOAT32_C(   -55.13), SIMDE_FLOAT32_C(   -97.94),
        SIMDE_FLOAT32_C(   -89.76), SIMDE_FLOAT32_C(   -84.67), SIMDE_FLOAT32_C(   -28.02), SIMDE_FLOAT32_C(   -17.26),
        SIMDE_FLOAT32_C(   -92.53), SIMDE_FLOAT32_C(   -38.86), SIMDE_FLOAT32_C(   -34.20), SIMDE_FLOAT32_C(   -81.88) },
      { SIMDE_FLOAT32_C(     0.89), SIMDE_FLOAT32_C(   -61.69), SIMDE_FLOAT32_C(    57.58), SIMDE_FLOAT32_C(   -94.97),
        SIMDE_FLOAT32_C(    20.67), SIMDE_FLOAT32_C(   -89.49), SIMDE_FLOAT32_C(   -48.46), SIMDE_FLOAT32_C(   -77.67),
        SIMDE_FLOAT32_C(     3.77), SIMDE_FLOAT32_C(   -59.39), SIMDE_FLOAT32_C(   -60.08), SIMDE_FLOAT32_C(   -91.27),
        SIMDE_FLOAT32_C(    24.08), SIMDE_FLOAT32_C(    26.19), SIMDE_FLOAT32_C(    52.05), SIMDE_FLOAT32_C(    55.88) },
      { SIMDE_FLOAT32_C( -6092.10), SIMDE_FLOAT32_C( -3341.70), SIMDE_FLOAT32_C( -1149.15), SIMDE_FLOAT32_C(  -177.82),
        SIMDE_FLOAT32_C(   159.97), SIMDE_FLOAT32_C(  -750.31), SIMDE_FLOAT32_C( -2643.43), SIMDE_FLOAT32_C( -5316.48),
        SIMDE_FLOAT32_C(   152.77), SIMDE_FLOAT32_C(  -630.07), SIMDE_FLOAT32_C(  2435.66), SIMDE_FLOAT32_C( -1513.67),
        SIMDE_FLOAT32_C(   483.03), SIMDE_FLOAT32_C(  -616.17), SIMDE_FLOAT32_C(  -417.17), SIMDE_FLOAT32_C(  3603.74) } },
    { { SIMDE_FLOAT32_C(    71.75), SIMDE_FLOAT32_C(   -31.30), SIMDE_FLOAT32_C(   -42.02), SIMDE_FLOAT32_C(   -61.98),
        SIMDE_FLOAT32_C(    31.26), SIMDE_FLOAT32_C(     2.85), SIMDE_FLOAT32_C(   -59.92), SIMDE_FLOAT32_C(    41.49),
        SIMDE_FLOAT32_C(    18.19), SIMDE_FLOAT32_C(    12.07), SIMDE_FLOAT32_C(   -75.76), SIMDE_FLOAT32_C(    25.65),
        SIMDE_FLOAT32_C(    73.21), SIMDE_FLOAT32_C(    -9.97), SIMDE_FLOAT32_C(    43.78), SIMDE_FLOAT32_C(   -25.90) },
      { SIMDE_FLOAT32_C(    28.35), SIMDE_FLOAT32_C(     1.36), SIMDE_FLOAT32_C(   -20.87), SIMDE_FLOAT32_C(   -50.99),
        SIMDE_FLOAT32_C(    11.86), SIMDE_FLOAT32_C(    30.66), SIMDE_FLOAT32_C(   -28.66), SIMDE_FLOAT32_C(   -84.37),
        SIMDE_FLOAT32_C(    71.27), SIMDE_FLOAT32_C(    11.26), SIMDE_FLOAT32_C(   -75.64), SIMDE_FLOAT32_C(    -4.65),
        SIMDE_FLOAT32_C(   -62.54), SIMDE_FLOAT32_C(    76.41), SIMDE_FLOAT32_C(   -48.78), SIMDE_FLOAT32_C(   -90.79) },
      { SIMDE_FLOAT32_C(   -54.89), SIMDE_FLOAT32_C(     9.21), SIMDE_FLOAT32_C(   -52.77), SIMDE_FLOAT32_C(    76.37),
        SIMDE_FLOAT32_C(   -87.94), SIMDE_FLOAT32_C(   -12.68), SIMDE_FLOAT32_C(    17.86), SIMDE_FLOAT32_C(    30.24),
        SIMDE_FLOAT32_C(    99.38), SIMDE_FLOAT32_C(    42.10), SIMDE_FLOAT32_C(   -44.10), SIMDE_FLOAT32_C(    72.59),
        SIMDE_FLOAT32_C(   -67.87), SIMDE_FLOAT32_C(    99.68), SIMDE_FLOAT32_C(   -53.31), SIMDE_FLOAT32_C(    60.48) },
      { SIMDE_FLOAT32_C( -2089.00), SIMDE_FLOAT32_C(    51.78), SIMDE_FLOAT32_C(  -929.73), SIMDE_FLOAT32_C( -3083.99),
        SIMDE_FLOAT32_C(  -458.68), SIMDE_FLOAT32_C(  -100.06), SIMDE_FLOAT32_C( -1699.45), SIMDE_FLOAT32_C(  3530.75),
        SIMDE_FLOAT32_C( -1197.02), SIMDE_FLOAT32_C(   -93.81), SIMDE_FLOAT32_C( -5774.59), SIMDE_FLOAT32_C(   191.86),
        SIMDE_FLOAT32_C(  4510.68), SIMDE_FLOAT32_C(   861.49), SIMDE_FLOAT32_C(  2082.28), SIMDE_FLOAT32_C( -2290.98) } },
    { { SIMDE_FLOAT32_C(     1.03), SIMDE_FLOAT32_C(    25.82), SIMDE_FLOAT32_C(   -90.51), SIMDE_FLOAT32_C(   -87.11),
        SIMDE_FLOAT32_C(   -43.51), SIMDE_FLOAT32_C(   -19.16), SIMDE_FLOAT32_C(   -71.47), SIMDE_FLOAT32_C(   -72.24),
        SIMDE_FLOAT32_C(    92.10), SIMDE_FLOAT32_C(   -47.11), SIMDE_FLOAT32_C(    23.11), SIMDE_FLOAT32_C(   -70.45),
        SIMDE_FLOAT32_C(   -70.70), SIMDE_FLOAT32_C(    74.33), SIMDE_FLOAT32_C(   -61.24), SIMDE_FLOAT32_C(   -25.59) },
      { SIMDE_FLOAT32_C(   -16.46), SIMDE_FLOAT32_C(   -14.01), SIMDE_FLOAT32_C(   -49.22), SIMDE_FLOAT32_C(    -4.41),
        SIMDE_FLOAT32_C(    73.31), SIMDE_FLOAT32_C(    68.64), SIMDE_FLOAT32_C(   -74.16), SIMDE_FLOAT32_C(    72.69),
        SIMDE_FLOAT32_C(    10.74), SIMDE_FLOAT32_C(   -18.27), SIMDE_FLOAT32_C(    45.28), SIMDE_FLOAT32_C(    42.87),
        SIMDE_FLOAT32_C(   -18.59), SIMDE_FLOAT32_C(    91.98), SIMDE_FLOAT32_C(     3.34), SIMDE_FLOAT32_C(    82.44) },
      { SIMDE_FLOAT32_C(    17.80), SIMDE_FLOAT32_C(    12.84), SIMDE_FLOAT32_C(    95.33), SIMDE_FLOAT32_C(    74.28),
        SIMDE_FLOAT32_C(    93.67), SIMDE_FLOAT32_C(   -76.14), SIMDE_FLOAT32_C(   -97.96), SIMDE_FLOAT32_C(    85.77),
        SIMDE_FLOAT32_C(   -23.25), SIMDE_FLOAT32_C(    25.15), SIMDE_FLOAT32_C(   -84.68), SIMDE_FLOAT32_C(     6.05),
        SIMDE_FLOAT32_C(    -0.52), SIMDE_FLOAT32_C(   -45.92), SIMDE_FLOAT32_C(    80.46), SIMDE_FLOAT32_C(    83.01) },
      { SIMDE_FLOAT32_C(    34.75), SIMDE_FLOAT32_C(   374.58), SIMDE_FLOAT32_C( -4359.57), SIMDE_FLOAT32_C(  -309.88),
        SIMDE_FLOAT32_C(  3283.39), SIMDE_FLOAT32_C(  1239.00), SIMDE_FLOAT32_C( -5398.18), SIMDE_FLOAT32_C(  5336.90),
        SIMDE_FLOAT32_C( -1012.40), SIMDE_FLOAT32_C(  -835.55), SIMDE_FLOAT32_C( -1131.10), SIMDE_FLOAT32_C(  3026.24),
        SIMDE_FLOAT32_C( -1314.83), SIMDE_FLOAT32_C( -6882.79), SIMDE_FLOAT32_C(   285.00), SIMDE_FLOAT32_C(  2192.65) } },
    { { SIMDE_FLOAT32_C(    40.08), SIMDE_FLOAT32_C(   -68.77), SIMDE_FLOAT32_C(   -21.40), SIMDE_FLOAT32_C(    13.39),
        SIMDE_FLOAT32_C(    99.87), SIMDE_FLOAT32_C(     4.44), SIMDE_FLOAT32_C(   -13.92), SIMDE_FLOAT32_C(    10.61),
        SIMDE_FLOAT32_C(    86.17), SIMDE_FLOAT32_C(   -68.63), SIMDE_FLOAT32_C(   -46.52), SIMDE_FLOAT32_C(   -32.42),
        SIMDE_FLOAT32_C(   -76.66), SIMDE_FLOAT32_C(    56.82), SIMDE_FLOAT32_C(   -49.98), SIMDE_FLOAT32_C(    41.14) },
      { SIMDE_FLOAT32_C(   -30.34), SIMDE_FLOAT32_C(   -54.64), SIMDE_FLOAT32_C(    15.43), SIMDE_FLOAT32_C(   -36.67),
        SIMDE_FLOAT32_C(   -30.78), SIMDE_FLOAT32_C(    17.47), SIMDE_FLOAT32_C(   -50.90), SIMDE_FLOAT32_C(    45.97),
        SIMDE_FLOAT32_C(   -57.39), SIMDE_FLOAT32_C(   -35.58), SIMDE_FLOAT32_C(   -47.99), SIMDE_FLOAT32_C(    42.09),
        SIMDE_FLOAT32_C(    18.50), SIMDE_FLOAT32_C(   -67.53), SIMDE_FLOAT32_C(    25.10), SIMDE_FLOAT32_C(   -41.42) },
      { SIMDE_FLOAT32_C(   -36.29), SIMDE_FLOAT32_C(   -96.30), SIMDE_FLOAT32_C(    71.97), SIMDE_FLOAT32_C(   -36.42),
        SIMDE_FLOAT32_C(     8.14), SIMDE_FLOAT32_C(   -41.94), SIMDE_FLOAT32_C(    74.19), SIMDE_FLOAT32_C(    -5.68),
        SIMDE_FLOAT32_C(   -10.58), SIMDE_FLOAT32_C(   -72.33), SIMDE_FLOAT32_C(    61.90), SIMDE_FLOAT32_C(    12.77),
        SIMDE_FLOAT32_C(    84.49), SIMDE_FLOAT32_C(   -88.07), SIMDE_FLOAT32_C(   -46.09), SIMDE_FLOAT32_C(   -45.85) },
      { SIMDE_FLOAT32_C(  1179.74), SIMDE_FLOAT32_C( -3853.89), SIMDE_FLOAT32_C(   402.17), SIMDE_FLOAT32_C(   454.59),
        SIMDE_FLOAT32_C(  3082.14), SIMDE_FLOAT32_C(  -119.51), SIMDE_FLOAT32_C(  -634.34), SIMDE_FLOAT32_C(  -493.42),
        SIMDE_FLOAT32_C(  4934.72), SIMDE_FLOAT32_C( -2514.19), SIMDE_FLOAT32_C( -2170.59), SIMDE_FLOAT32_C(  1377.33),
        SIMDE_FLOAT32_C(  1502.70), SIMDE_FLOAT32_C(  3748.98), SIMDE_FLOAT32_C(  1208.41), SIMDE_FLOAT32_C(  1658.17) } },
    { { SIMDE_FLOAT32_C(   -42.71), SIMDE_FLOAT32_C(    69.33), SIMDE_FLOAT32_C(    17.48), SIMDE_FLOAT32_C(    26.51),
        SIMDE_FLOAT32_C(   -13.20), SIMDE_FLOAT32_C(    66.58), SIMDE_FLOAT32_C(   -27.52), SIMDE_FLOAT32_C(    29.41),
        SIMDE_FLOAT32_C(   -69.00), SIMDE_FLOAT32_C(    24.49), SIMDE_FLOAT32_C(   -28.50), SIMDE_FLOAT32_C(    49.50),
        SIMDE_FLOAT32_C(    56.96), SIMDE_FLOAT32_C(    96.60), SIMDE_FLOAT32_C(   -91.91), SIMDE_FLOAT32_C(   -79.34) },
      { SIMDE_FLOAT32_C(   -99.70), SIMDE_FLOAT32_C(    80.06), SIMDE_FLOAT32_C(   -15.76), SIMDE_FLOAT32_C(     8.44),
        SIMDE_FLOAT32_C(   -61.89), SIMDE_FLOAT32_C(   -41.56), SIMDE_FLOAT32_C(   -97.24), SIMDE_FLOAT32_C(    27.54),
        SIMDE_FLOAT32_C(   -13.89), SIMDE_FLOAT32_C(    64.67), SIMDE_FLOAT32_C(   -59.70), SIMDE_FLOAT32_C(   -29.40),
        SIMDE_FLOAT32_C(    76.59), SIMDE_FLOAT32_C(    -5.79), SIMDE_FLOAT32_C(    24.75), SIMDE_FLOAT32_C(   -66.12) },
      { SIMDE_FLOAT32_C(   -36.45), SIMDE_FLOAT32_C(   -57.76), SIMDE_FLOAT32_C(    60.39), SIMDE_FLOAT32_C(    50.35),
        SIMDE_FLOAT32_C(   -91.18), SIMDE_FLOAT32_C(   -67.13), SIMDE_FLOAT32_C(   -20.24), SIMDE_FLOAT32_C(   -60.18),
        SIMDE_FLOAT32_C(    57.36), SIMDE_FLOAT32_C(    51.26), SIMDE_FLOAT32_C(    89.32), SIMDE_FLOAT32_C(    14.31),
        SIMDE_FLOAT32_C(    47.86), SIMDE_FLOAT32_C(    97.41), SIMDE_FLOAT32_C(    34.98), SIMDE_FLOAT32_C(    48.16) },
      { SIMDE_FLOAT32_C( -4294.64), SIMDE_FLOAT32_C( -5608.32), SIMDE_FLOAT32_C(   335.87), SIMDE_FLOAT32_C(  -173.39),
        SIMDE_FLOAT32_C(  -908.13), SIMDE_FLOAT32_C(  2699.94), SIMDE_FLOAT32_C( -2696.28), SIMDE_FLOAT32_C(  -870.13),
        SIMDE_FLOAT32_C(  -901.05), SIMDE_FLOAT32_C( -1532.51), SIMDE_FLOAT32_C( -1612.13), SIMDE_FLOAT32_C(  1469.61),
        SIMDE_FLOAT32_C( -4314.71), SIMDE_FLOAT32_C(   656.72), SIMDE_FLOAT32_C(  2309.75), SIMDE_FLOAT32_C( -5197.80) } },
    { { SIMDE_FLOAT32_C(    77.47), SIMDE_FLOAT32_C(   -80.78), SIMDE_FLOAT32_C(   -43.40), SIMDE_FLOAT32_C(   -84.42),
        SIMDE_FLOAT32_C(   -22.34), SIMDE_FLOAT32_C(   -40.64), SIMDE_FLOAT32_C(    43.12), SIMDE_FLOAT32_C(    63.76),
        SIMDE_FLOAT32_C(   -75.97), SIMDE_FLOAT32_C(    83.43), SIMDE_FLOAT32_C(   -65.64), SIMDE_FLOAT32_C(   -99.38),
        SIMDE_FLOAT32_C(   -22.36), SIMDE_FLOAT32_C(    59.12), SIMDE_FLOAT32_C(   -65.50), SIMDE_FLOAT32_C(    41.19) },
      { SIMDE_FLOAT32_C(   -98.64), SIMDE_FLOAT32_C(    94.89), SIMDE_FLOAT32_C(    -8.47), SIMDE_FLOAT32_C(   -89.83),
        SIMDE_FLOAT32_C(   -72.24), SIMDE_FLOAT32_C(    71.29), SIMDE_FLOAT32_C(   -50.01), SIMDE_FLOAT32_C(    85.11),
        SIMDE_FLOAT32_C(    22.55), SIMDE_FLOAT32_C(   -60.68), SIMDE_FLOAT32_C(    -0.57), SIMDE_FLOAT32_C(   -29.59),
        SIMDE_FLOAT32_C(   -63.27), SIMDE_FLOAT32_C(   -65.59), SIMDE_FLOAT32_C(   -81.44), SIMDE_FLOAT32_C(   -85.80) },
      { SIMDE_FLOAT32_C(   -46.37), SIMDE_FLOAT32_C(   -24.84), SIMDE_FLOAT32_C(   -70.22), SIMDE_FLOAT32_C(    31.29),
        SIMDE_FLOAT32_C(    34.52), SIMDE_FLOAT32_C(    72.90), SIMDE_FLOAT32_C(    -4.95), SIMDE_FLOAT32_C(    58.55),
        SIMDE_FLOAT32_C(    56.33), SIMDE_FLOAT32_C(    29.42), SIMDE_FLOAT32_C(    59.17), SIMDE_FLOAT32_C(   -66.03),
        SIMDE_FLOAT32_C(   -11.46), SIMDE_FLOAT32_C(    93.67), SIMDE_FLOAT32_C(    75.15), SIMDE_FLOAT32_C(   -10.11) },
      { SIMDE_FLOAT32_C(  7595.27), SIMDE_FLOAT32_C(  7640.37), SIMDE_FLOAT32_C(  -437.82), SIMDE_FLOAT32_C( -7552.16),
        SIMDE_FLOAT32_C( -1579.32), SIMDE_FLOAT32_C(  2970.13), SIMDE_FLOAT32_C(  2151.48), SIMDE_FLOAT32_C( -5368.06),
        SIMDE_FLOAT32_C(  1769.45), SIMDE_FLOAT32_C(  5091.95), SIMDE_FLOAT32_C(    21.76), SIMDE_FLOAT32_C( -3006.68),
        SIMDE_FLOAT32_C( -1426.18), SIMDE_FLOAT32_C(  3971.35), SIMDE_FLOAT32_C( -5259.17), SIMDE_FLOAT32_C(  3523.99) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512 a = simde_mm512_loadu_ps(test_vec[i].a);
    simde__m512 b = simde_mm512_loadu_ps(test_vec[i].b);
    simde__m512 c = simde_mm512_loadu_ps(test_vec[i].c);
    simde__m512 r = simde_mm512_fnmadd_ps(a, b, c);
    simde_test_x86_assert_equal_f32x16(r, simde_mm512_loadu_ps(test_vec[i].r), 1);
  }

  return 0;
}

static int
test_simde_mm512_fnmadd_pd (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde_float64 a[8];
    const simde_float64 b[8];
    const simde_float64 c[8];
    const simde_float64 r[8];
  } test_vec[] = {
    { { SIMDE_FLOAT64_C(   -33.42), SIMDE_FLOAT64_C(    18.77), SIMDE_FLOAT64_C(   -64.80), SIMDE_FLOAT64_C(   -31.25),
        SIMDE_FLOAT64_C(    13.14), SIMDE_FLOAT64_C(   -25.39), SIMDE_FLOAT64_C(   -32.34), SIMDE_FLOAT64_C(     4.66) },
      { SIMDE_FLOAT64_C(    75.32), SIMDE_FLOAT64_C(    64.02), SIMDE_FLOAT64_C(    55.28), SIMDE_FLOAT64_C(    78.84),
        SIMDE_FLOAT64_C(    52.87), SIMDE_FLOAT64_C(   -80.75), SIMDE_FLOAT64_C(   -10.62), SIMDE_FLOAT64_C(    16.16) },
      { SIMDE_FLOAT64_C(    95.97), SIMDE_FLOAT64_C(    62.64), SIMDE_FLOAT64_C(   -14.97), SIMDE_FLOAT64_C(   -48.39),
        SIMDE_FLOAT64_C(    80.72), SIMDE_FLOAT64_C(    32.09), SIMDE_FLOAT64_C(    81.91), SIMDE_FLOAT64_C(    89.26) },
      { SIMDE_FLOAT64_C(  2613.16), SIMDE_FLOAT64_C( -1139.02), SIMDE_FLOAT64_C(  3567.17), SIMDE_FLOAT64_C(  2415.36),
        SIMDE_FLOAT64_C(  -613.99), SIMDE_FLOAT64_C( -2018.15), SIMDE_FLOAT64_C(  -261.54), SIMDE_FLOAT64_C(    13.95) } },
    { { SIMDE_FLOAT64_C(   -94.10), SIMDE_FLOAT64_C(    84.15), SIMDE_FLOAT64_C(     9.79), SIMDE_FLOAT64_C(    -9.15),
        SIMDE_FLOAT64_C(     9.78), SIMDE_FLOAT64_C(   -92.82), SIMDE_FLOAT64_C(   -64.66), SIMDE_FLOAT64_C(    76.36) },
      { SIMDE_FLOAT64_C(    25.95), SIMDE_FLOAT64_C(   -29.46), SIMDE_FLOAT64_C(   -54.89), SIMDE_FLOAT64_C(   -60.91),
        SIMDE_FLOAT64_C(    45.15), SIMDE_FLOAT64_C(    12.77), SIMDE_FLOAT64_C(    43.76), SIMDE_FLOAT64_C(    20.47) },
      { SIMDE_FLOAT64_C(   -23.21), SIMDE_FLOAT64_C(    -0.96), SIMDE_FLOAT64_C(    -0.69), SIMDE_FLOAT64_C(   -70.34),
        SIMDE_FLOAT64_C(    18.29), SIMDE_FLOAT64_C(    88.69), SIMDE_FLOAT64_C(    45.82), SIMDE_FLOAT64_C(    14.26) },
      { SIMDE_FLOAT64_C(  2418.68), SIMDE_FLOAT64_C(  2478.10), SIMDE_FLOAT64_C(   536.68), SIMDE_FLOAT64_C(  -627.67),
        SIMDE_FLOAT64_C(  -423.28), SIMDE_FLOAT64_C(  1274.00), SIMDE_FLOAT64_C(  2875.34), SIMDE_FLOAT64_C( -1548.83) } },
    { { SIMDE_FLOAT64_C(    51.33), SIMDE_FLOAT64_C(   -69.15), SIMDE_FLOAT64_C(    65.87), SIMDE_FLOAT64_C(    32.05),
        SIMDE_FLOAT64_C(    62.93), SIMDE_FLOAT64_C(    47.79), SIMDE_FLOAT64_C(    21.31), SIMDE_FLOAT64_C(    68.84) },
      { SIMDE_FLOAT64_C(    31.94), SIMDE_FLOAT64_C(   -68.91), SIMDE_FLOAT64_C(   -40.32), SIMDE_FLOAT64_C(   -58.28),
        SIMDE_FLOAT64_C(   -61.72), SIMDE_FLOAT64_C(    -4.98), SIMDE_FLOAT64_C(   -81.91), SIMDE_FLOAT64_C(    64.23) },
      { SIMDE_FLOAT64_C(    65.56), SIMDE_FLOAT64_C(   -36.80), SIMDE_FLOAT64_C(   -96.68), SIMDE_FLOAT64_C(    10.71),
        SIMDE_FLOAT64_C(    75.97), SIMDE_FLOAT64_C(    47.08), SIMDE_FLOAT64_C(   -68.83), SIMDE_FLOAT64_C(   -47.24) },
      { SIMDE_FLOAT64_C( -1573.92), SIMDE_FLOAT64_C( -4801.93), SIMDE_FLOAT64_C(  2559.20), SIMDE_FLOAT64_C(  1878.58),
        SIMDE_FLOAT64_C(  3960.01), SIMDE_FLOAT64_C(   285.07), SIMDE_FLOAT64_C(  1676.67), SIMDE_FLOAT64_C( -4468.83) } },
    { { SIMDE_FLOAT64_C(   -53.88), SIMDE_FLOAT64_C(    30.48), SIMDE_FLOAT64_C(   -17.58), SIMDE_FLOAT64_C(    64.42),
        SIMDE_FLOAT64_C(    19.17), SIMDE_FLOAT64_C(   -71.76), SIMDE_FLOAT64_C(   -21.32), SIMDE_FLOAT64_C(   -29.50) },
      { SIMDE_FLOAT64_C(   -40.91), SIMDE_FLOAT64_C(   -55.45), SIMDE_FLOAT64_C(   -97.45), SIMDE_FLOAT64_C(   -77.98),
        SIMDE_FLOAT64_C(    92.34), SIMDE_FLOAT64_C(    23.85), SIMDE_FLOAT64_C(    90.86), SIMDE_FLOAT64_C(    24.27) },
      { SIMDE_FLOAT64_C(    54.95), SIMDE_FLOAT64_C(   -49.46), SIMDE_FLOAT64_C(    66.00), SIMDE_FLOAT64_C(    93.22),
        SIMDE_FLOAT64_C(    45.56), SIMDE_FLOAT64_C(    84.08), SIMDE_FLOAT64_C(    57.45), SIMDE_FLOAT64_C(    11.12) },
      { SIMDE_FLOAT64_C( -2149.28), SIMDE_FLOAT64_C(  1640.66), SIMDE_FLOAT64_C( -1647.17), SIMDE_FLOAT64_C(  5116.69),
        SIMDE_FLOAT64_C( -1724.60), SIMDE_FLOAT64_C(  1795.56), SIMDE_FLOAT64_C(  1994.59), SIMDE_FLOAT64_C(   727.09) } },
    { { SIMDE_FLOAT64_C(   -52.72), SIMDE_FLOAT64_C(    60.77), SIMDE_FLOAT64_C(   -78.17), SIMDE_FLOAT64_C(   -76.75),
        SIMDE_FLOAT64_C(     7.85), SIMDE_FLOAT64_C(   -47.00), SIMDE_FLOAT64_C(   -23.99), SIMDE_FLOAT64_C(    53.98) },
      { SIMDE_FLOAT64_C(    83.49), SIMDE_FLOAT64_C(    58.43), SIMDE_FLOAT64_C(    18.39), SIMDE_FLOAT64_C(     2.66),
        SIMDE_FLOAT64_C(    86.67), SIMDE_FLOAT64_C(    97.07), SIMDE_FLOAT64_C(    73.16), SIMDE_FLOAT64_C(   -54.24) },
      { SIMDE_FLOAT64_C(   -58.38), SIMDE_FLOAT64_C(    75.70), SIMDE_FLOAT64_C(   -32.22), SIMDE_FLOAT64_C(   -66.05),
        SIMDE_FLOAT64_C(    -0.44), SIMDE_FLOAT64_C(   -41.36), SIMDE_FLOAT64_C(    58.23), SIMDE_FLOAT64_C(   -45.50) },
      { SIMDE_FLOAT64_C(  4343.21), SIMDE_FLOAT64_C( -3475.09), SIMDE_FLOAT64_C(  1405.33), SIMDE_FLOAT64_C(   138.11),
        SIMDE_FLOAT64_C(  -680.80), SIMDE_FLOAT64_C(  4520.93), SIMDE_FLOAT64_C(  1813.34), SIMDE_FLOAT64_C(  2882.38) } },
    { { SIMDE_FLOAT64_C(     9.18), SIMDE_FLOAT64_C(    24.22), SIMDE_FLOAT64_C(   -52.28), SIMDE_FLOAT64_C(   -45.26),
        SIMDE_FLOAT64_C(     8.30), SIMDE_FLOAT64_C(   -94.83), SIMDE_FLOAT64_C(    65.86), SIMDE_FLOAT64_C(    55.58) },
      { SIMDE_FLOAT64_C(    65.95), SIMDE_FLOAT64_C(    87.69), SIMDE_FLOAT64_C(    78.83), SIMDE_FLOAT64_C(   -26.20),
        SIMDE_FLOAT64_C(   -59.31), SIMDE_FLOAT64_C(   -45.15), SIMDE_FLOAT64_C(   -72.22), SIMDE_FLOAT64_C(   -75.82) },
      { SIMDE_FLOAT64_C(   -86.72), SIMDE_FLOAT64_C(    46.17), SIMDE_FLOAT64_C(    26.84), SIMDE_FLOAT64_C(    99.95),
        SIMDE_FLOAT64_C(    43.24), SIMDE_FLOAT64_C(    -0.00), SIMDE_FLOAT64_C(   -54.29), SIMDE_FLOAT64_C(    84.85) },
      { SIMDE_FLOAT64_C(  -692.14), SIMDE_FLOAT64_C( -2077.68), SIMDE_FLOAT64_C(  4148.07), SIMDE_FLOAT64_C( -1085.86),
        SIMDE_FLOAT64_C(   535.51), SIMDE_FLOAT64_C( -4281.57), SIMDE_FLOAT64_C(  4702.12), SIMDE_FLOAT64_C(  4298.93) } },
    { { SIMDE_FLOAT64_C(   -24.30), SIMDE_FLOAT64_C(    13.49), SIMDE_FLOAT64_C(   -81.19), SIMDE_FLOAT64_C(    75.26),
        SIMDE_FLOAT64_C(    72.13), SIMDE_FLOAT64_C(    77.04), SIMDE_FLOAT64_C(   -70.24), SIMDE_FLOAT64_C(   -18.69) },
      { SIMDE_FLOAT64_C(     1.26), SIMDE_FLOAT64_C(   -22.52), SIMDE_FLOAT64_C(    36.06), SIMDE_FLOAT64_C(   -90.44),
        SIMDE_FLOAT64_C(   -17.34), SIMDE_FLOAT64_C(     1.92), SIMDE_FLOAT64_C(    65.15), SIMDE_FLOAT64_C(   -51.39) },
      { SIMDE_FLOAT64_C(   -10.39), SIMDE_FLOAT64_C(    43.98), SIMDE_FLOAT64_C(    22.41), SIMDE_FLOAT64_C(    30.30),
        SIMDE_FLOAT64_C(    98.83), SIMDE_FLOAT64_C(    50.19), SIMDE_FLOAT64_C(    54.48), SIMDE_FLOAT64_C(   -87.89) },
      { SIMDE_FLOAT64_C(    20.23), SIMDE_FLOAT64_C(   347.77), SIMDE_FLOAT64_C(  2950.12), SIMDE_FLOAT64_C(  6836.81),
        SIMDE_FLOAT64_C(  1349.56), SIMDE_FLOAT64_C(   -97.73), SIMDE_FLOAT64_C(  4630.62), SIMDE_FLOAT64_C( -1048.37) } },
    { { SIMDE_FLOAT64_C(    -3.64), SIMDE_FLOAT64_C(   -18.68), SIMDE_FLOAT64_C(   -87.95), SIMDE_FLOAT64_C(   -60.40),
        SIMDE_FLOAT64_C(    81.32), SIMDE_FLOAT64_C(   -42.24), SIMDE_FLOAT64_C(   -75.55), SIMDE_FLOAT64_C(   -42.97) },
      { SIMDE_FLOAT64_C(    71.25), SIMDE_FLOAT64_C(   -56.74), SIMDE_FLOAT64_C(   -67.71), SIMDE_FLOAT64_C(    43.39),
        SIMDE_FLOAT64_C(   -79.71), SIMDE_FLOAT64_C(   -37.95), SIMDE_FLOAT64_C(   -75.30), SIMDE_FLOAT64_C(    21.55) },
      { SIMDE_FLOAT64_C(    39.53), SIMDE_FLOAT64_C(    60.76), SIMDE_FLOAT64_C(    31.12), SIMDE_FLOAT64_C(   -77.81),
        SIMDE_FLOAT64_C(   -37.33), SIMDE_FLOAT64_C(    -3.74), SIMDE_FLOAT64_C(   -29.20), SIMDE_FLOAT64_C(    52.28) },
      { SIMDE_FLOAT64_C(   298.88), SIMDE_FLOAT64_C(  -999.14), SIMDE_FLOAT64_C( -5923.97), SIMDE_FLOAT64_C(  2542.95),
        SIMDE_FLOAT64_C(  6444.69), SIMDE_FLOAT64_C( -1606.75), SIMDE_FLOAT64_C( -5718.11), SIMDE_FLOAT64_C(   978.28) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512d a = simde_mm512_loadu_pd(test_vec[i].a);
    simde__m512d b = simde_mm512_loadu_pd(test_vec[i].b);
    simde__m512d c = simde_mm512_loadu_pd(test_vec[i].c);
    simde__m512d r = simde_mm512_fnmadd_pd(a, b, c);
    simde_test_x86_assert_equal_f64x8(r, simde_mm512_loadu_pd(test_vec[i].r), 1);
  }

  return 0;
}

SIMDE_TEST_FUNC_LIST_BEGIN
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_fnmadd_ps)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_fnmadd_pd)
SIMDE_TEST_FUNC_LIST_END

#include <test/x86/avx512/test-avx512-footer.h>
