//! Error types for parsing.

use thiserror::Error;

/// Errors that can occur during parsing or processing.
#[derive(Error, Debug)]
pub enum Error {
    /// The provided class string is not a recognized microformats2 type.
    #[error("The type {0:?} is not a known Microformats class supported by this library.")]
    NotKnownClass(String),

    /// JSON parsing or serialization error.
    #[error("JSON: {0}")]
    JSON(#[from] serde_json::Error),

    /// Expected a JSON object but got something else.
    #[error("The provided JSON value was not an object.")]
    NotAnObject,

    /// A required property is missing from a JSON object.
    #[error("Missing property {0:?} when converting from JSON.")]
    JsonObjectMissingProperty(String),

    /// Date/time parsing error.
    #[error(transparent)]
    Temporal(#[from] crate::temporal::Error),
}

impl PartialEq for Error {
    fn eq(&self, other: &Self) -> bool {
        self.to_string().eq(&other.to_string())
    }
}

impl Eq for Error {}
