use super::Element;

impl Element {
    /// Returns the persons and/or institutions involved in the discovery of the element, if known.
    ///
    /// ```
    /// use mendeleev::Element;
    ///
    /// assert_eq!(Element::H.discoverers(), Some(["Henry Cavendish"].as_slice()));
    /// assert_eq!(Element::He.discoverers(), Some(["Sir William Ramsey", "Nils Langet", "P.T.Cleve"].as_slice()));
    /// assert_eq!(Element::Og.discoverers(), Some(["Joint Institute for Nuclear Research"].as_slice()));
    /// assert_eq!(Element::Au.discoverers(), None);
    /// ```
    pub const fn discoverers(&self) -> Option<&'static [&'static str]> {
        use Element as E;
        match self {
            E::H => Some(&["Henry Cavendish"]),
            E::He => Some(&["Sir William Ramsey", "Nils Langet", "P.T.Cleve"]),
            E::Li => Some(&["Johann Arfwedson"]),
            E::Be => Some(&["Fredrich Wöhler", "A.A.Bussy"]),
            E::B => Some(&["Sir H. Davy", "J.L. Gay-Lussac", "L.J. Thénard"]),
            E::C => None,
            E::N => Some(&["Daniel Rutherford"]),
            E::O => Some(&["Joseph Priestly", "Carl Wilhelm Scheele"]),
            E::F => Some(&["Henri Moissan"]),
            E::Ne => Some(&["Sir William Ramsey", "M.W. Travers"]),
            E::Na => Some(&["Sir Humphrey Davy"]),
            E::Mg => Some(&["Sir Humphrey Davy"]),
            E::Al => Some(&["Hans Christian Oersted"]),
            E::Si => Some(&["Jöns Berzelius"]),
            E::P => Some(&["Hennig Brand"]),
            E::S => None,
            E::Cl => Some(&["Carl Wilhelm Scheele"]),
            E::Ar => Some(&["Sir William Ramsey", "Baron Rayleigh"]),
            E::K => Some(&["Sir Humphrey Davy"]),
            E::Ca => Some(&["Sir Humphrey Davy"]),
            E::Sc => Some(&["Lars Nilson"]),
            E::Ti => Some(&["William Gregor"]),
            E::V => Some(&["Nils Sefström"]),
            E::Cr => Some(&["Louis Vauquelin"]),
            E::Mn => Some(&["Johann Gahn"]),
            E::Fe => None,
            E::Co => Some(&["George Brandt"]),
            E::Ni => Some(&["Axel Cronstedt"]),
            E::Cu => None,
            E::Zn => None,
            E::Ga => Some(&["Paul Émile Lecoq de Boisbaudran"]),
            E::Ge => Some(&["Clemens Winkler"]),
            E::As => None,
            E::Se => Some(&["Jöns Berzelius"]),
            E::Br => Some(&["Antoine J. Balard"]),
            E::Kr => Some(&["Sir William Ramsey", "M.W. Travers"]),
            E::Rb => Some(&["R. Bunsen", "G. Kirchoff"]),
            E::Sr => Some(&["A. Crawford"]),
            E::Y => Some(&["Johann Gadolin"]),
            E::Zr => Some(&["Martin Klaproth"]),
            E::Nb => Some(&["Charles Hatchet"]),
            E::Mo => Some(&["Carl Wilhelm Scheele"]),
            E::Tc => Some(&["Carlo Perrier", "Émillo Segrè"]),
            E::Ru => Some(&["Karl Klaus"]),
            E::Rh => Some(&["William Wollaston"]),
            E::Pd => Some(&["William Wollaston"]),
            E::Ag => None,
            E::Cd => Some(&["Fredrich Stromeyer"]),
            E::In => Some(&["Ferdinand Reich", "T. Richter"]),
            E::Sn => None,
            E::Sb => None,
            E::Te => Some(&["Franz Müller von Reichenstein"]),
            E::I => Some(&["Bernard Courtois"]),
            E::Xe => Some(&["Sir William Ramsay", " M. W. Travers"]),
            E::Cs => Some(&["Gustov Kirchoff", "Robert Bunsen"]),
            E::Ba => Some(&["Sir Humphrey Davy"]),
            E::La => Some(&["Carl Mosander"]),
            E::Ce => Some(&["W. von Hisinger", "J. Berzelius", "M. Klaproth"]),
            E::Pr => Some(&["C.F. Aver von Welsbach"]),
            E::Nd => Some(&["C.F. Aver von Welsbach"]),
            E::Pm => Some(&["J.A. Marinsky", "L.E. Glendenin", "C.D. Coryell"]),
            E::Sm => Some(&["Paul Émile Lecoq de Boisbaudran"]),
            E::Eu => Some(&["Eugène Demarçay"]),
            E::Gd => Some(&["Jean de Marignac"]),
            E::Tb => Some(&["Carl Mosander"]),
            E::Dy => Some(&["Paul Émile Lecoq de Boisbaudran"]),
            E::Ho => Some(&["J.L. Soret"]),
            E::Er => Some(&["Carl Mosander"]),
            E::Tm => Some(&["Per Theodor Cleve"]),
            E::Yb => Some(&["Jean de Marignac"]),
            E::Lu => Some(&["Georges Urbain"]),
            E::Hf => Some(&["Dirk Coster", "Georg von Hevesy"]),
            E::Ta => Some(&["Anders Ekeberg"]),
            E::W => Some(&["Fausto and Juan José de Elhuyar"]),
            E::Re => Some(&["Walter Noddack", "Ida Tacke", "Otto Berg"]),
            E::Os => Some(&["Smithson Tenant"]),
            E::Ir => Some(&[
                "S.Tenant",
                "A.F.Fourcory",
                "L.N.Vauquelin",
                "H.V.Collet-Descoltils",
            ]),
            E::Pt => Some(&["Julius Scaliger"]),
            E::Au => None,
            E::Hg => None,
            E::Tl => Some(&["Sir William Crookes"]),
            E::Pb => None,
            E::Bi => None,
            E::Po => Some(&["Pierre and Marie Curie"]),
            E::At => Some(&["D.R.Corson", "K.R.MacKenzie", "E.Segré"]),
            E::Rn => Some(&["Fredrich Ernst Dorn"]),
            E::Fr => Some(&["Marguerite Derey"]),
            E::Ra => Some(&["Pierre and Marie Curie"]),
            E::Ac => Some(&["André Debierne"]),
            E::Th => Some(&["Jöns Berzelius"]),
            E::Pa => Some(&[
                "Fredrich Soddy",
                "John Cranston",
                "Otto Hahn",
                "Lise Meitner",
            ]),
            E::U => Some(&["Martin Klaproth"]),
            E::Np => Some(&["E.M. McMillan", "P.H. Abelson"]),
            E::Pu => Some(&["G.T.Seaborg", "J.W.Kennedy", "E.M.McMillan", "A.C.Wohl"]),
            E::Am => Some(&["G.T.Seaborg", "R.A.James", "L.O.Morgan", "A.Ghiorso"]),
            E::Cm => Some(&["G.T.Seaborg", "R.A.James", "A.Ghiorso"]),
            E::Bk => Some(&["G.T.Seaborg", "S.G.Tompson", "A.Ghiorso"]),
            E::Cf => Some(&["G.T.Seaborg", "S.G.Tompson", "A.Ghiorso", "K.Street Jr."]),
            E::Es => Some(&["Argonne", "Los Alamos", "U of Calif"]),
            E::Fm => Some(&["Argonne", "Los Alamos", "U of Calif"]),
            E::Md => Some(&["G.T.Seaborg", "S.G.Tompson", "A.Ghiorso", "K.Street Jr."]),
            E::No => Some(&["Nobel Institute for Physics"]),
            E::Lr => Some(&["A.Ghiorso", "T.Sikkeland", "A.E.Larsh", "R.M.Latimer"]),
            E::Rf => Some(&["A. Ghiorso, et al"]),
            E::Db => Some(&["A. Ghiorso, et al"]),
            E::Sg => Some(&["Soviet Nuclear Research", "U. of Cal at Berkeley"]),
            E::Bh => Some(&["Heavy Ion Research Laboratory (HIRL)"]),
            E::Hs => Some(&["Heavy Ion Research Laboratory (HIRL)"]),
            E::Mt => Some(&["Heavy Ion Research Laboratory (HIRL)"]),
            E::Ds => Some(&["Heavy Ion Research Laboratory (HIRL)"]),
            E::Rg => Some(&["Heavy Ion Research Laboratory (HIRL)"]),
            E::Cn => Some(&["GSI Helmholtz Centre for Heavy Ion Research"]),
            E::Nh => Some(&["RIKEN"]),
            E::Fl => Some(&["Joint Institute for Nuclear Research"]),
            E::Mc => Some(&["Joint Institute for Nuclear Research"]),
            E::Lv => Some(&["Lawrence Livermore National Laboratory"]),
            E::Ts => Some(&[
                "Joint Institute for Nuclear Research",
                "Oak Ridge National Laboratory",
            ]),
            E::Og => Some(&["Joint Institute for Nuclear Research"]),
        }
    }
}
