% File src/library/methods/man/evalSource.Rd
% Part of the R package, https://www.R-project.org
% Copyright 2010-2 R Core Team
% Distributed under GPL 2 or later

\name{evalSource}
\alias{evalSource}
\alias{insertSource}
\alias{sourceEnvironment-class}
\title{
Use Function Definitions from a Source File without Reinstalling a Package
}
\description{
Definitions of functions and/or methods from a source file are
inserted into a package, using the \code{\link{trace}} mechanism.
Typically, this allows testing or debugging modified versions of a few
functions without reinstalling a large package.
}
\usage{
evalSource(source, package = "", lock = TRUE, cache = FALSE)

insertSource(source, package = "", functions = , methods = ,
           force = )

}
\arguments{
  \item{source}{
A file to be parsed and evaluated by \code{evalSource} to find the new
function and method definitions.

The argument to \code{insertSource} can be an object of class \code{"sourceEnvironment"}
returned from a previous call
to \code{evalSource}   If a file name is passed to \code{insertSource}
it calls \code{evalSource} to obtain the corresponding object.  See
the section on the class for details.

}
  \item{package}{
Optionally, the name of the package to which the new code corresponds
and into which it will be
inserted.  Although the computations will attempt to infer the package
if it is omitted, the safe approach is to supply it.  In the case of a
package that is not attached to the search list, the package name must
be supplied.
}
  \item{functions, methods}{
Optionally, the character-string names of the functions to be
used in the insertion.  Names supplied in the \code{functions}
argument are expected to be defined as functions in the source.
For names supplied in the \code{methods} argument, a table of methods
is expected (as generated by calls to \code{\link{setMethod}}, see the
details section); methods from this table will be inserted by
\code{insertSource}.  In both cases, the revised function or method is
inserted only if it differs from the version in the corresponding
package as loaded.

If \code{what} is omitted, the results of evaluating the source file
will be compared to the contents of the package (see the details section).
}
\item{lock, cache}{
Optional arguments to control the actions taken by \code{evalSource}.
If \code{lock} is \code{TRUE}, the environment in the object returned
will be locked, and so will all its bindings.
If \code{cache} is \code{FALSE}, the normal caching of method and
class definitions will be suppressed during evaluation of the
\code{source} file.

The default settings are generally recommended, the \code{lock} to
support the credibility of the object returned as a snapshot of the
source file, and the second so that method definitions can be inserted
later by \code{insertSource} using the trace mechanism.
}
\item{force}{
If \code{FALSE}, only functions currently in the environment will be
redefined, using \code{\link{trace}}.  If \code{TRUE}, other
objects/functions will be simply assigned.  By default, \code{TRUE} if
neither the \code{functions} nor the \code{methods} argument is supplied.
}

}
\details{
The \code{source} file is parsed and evaluated, suppressing by default
the actual caching of method and class definitions contained in it, so
that functions and methods can be tested out in a reversible way.
The result, if all goes well, is an environment containing the
assigned objects and metadata corresponding to method and class definitions
in the source file.

From this environment, the objects are inserted into the package, into
its namespace if it has one, for use during the current session or
until reverting to the original version by a call to
\code{\link{untrace}}.
The insertion is done by calls to the internal version of
\code{\link{trace}}, to make reversion possible.

Because the trace mechanism is used, only function-type objects will
be inserted, functions themselves or S4 methods.

When the \code{functions} and \code{methods} arguments are both
omitted, \code{insertSource} selects all suitable objects from the
result of evaluating the \code{source} file.

In all cases,
only objects in the source file that differ from
the corresponding objects in the package are inserted.
The definition of \dQuote{differ} is that either the argument list
(including default expressions) or the body of the function is not
identical.
Note that in the case of a method, there need be no specific method
for the corresponding signature in the package: the comparison is made
to the method that would be selected for that signature.

Nothing in the computation requires that the source file supplied be
the same file as in the original package source, although that case is
both likely and sensible if one is revising the package.  Nothing in
the computations compares source files:  the objects generated by
evaluating \code{source} are compared as objects to the content of the package.
}
\value{
An object from class \code{"sourceEnvironment"}, a subclass of
\code{"environment"} (see the section on the class)
The environment contains the versions
of \emph{all}  object resulting from evaluation of the source file.
The class also has slots for the time of creation, the source file
and the package name.
Future extensions may use these objects for versioning or other code tools.


The object returned can be used in debugging (see the section on that
topic) or as the \code{source}
argument in a future call to \code{insertSource}.  If only some of the
revised functions were inserted in the first call, others can be
inserted in a later call without re-evaluating the source file, by
supplying the environment and optionally suitable \code{functions}
and/or \code{methods} argument.
}

\section{Debugging}{
Once a function or method has been inserted into a package by
\code{insertSource}, it can be studied by the standard debugging tools;
for example, \code{\link{debug}} or the various versions of
\code{\link{trace}}.

Calls to \code{\link{trace}} should take the extra argument \code{edit
= env}, where \code{env} is the value returned by the call to
\code{evalSource}.
The trace mechanism has been used to install the revised version from
the source file, and supplying the argument ensures that it is this
version, not the original, that will be traced.  See the example
below.

To turn tracing off, but retain the source version, use \code{trace(x,
edit = env)} as in the example.  To return to the original version
from the package, use \code{untrace(x)}.
}

\section{Class \code{"sourceEnvironment"}}{
Objects from this class can be treated as environments, to extract the
version of functions and methods generated by \code{evalSource}.
The objects also have the following slots:
\describe{
\item{\code{packageName}:}{ The character-string name of the package
    to which the source code corresponds.
}
\item{\code{dateCreated}:}{ The date and time that the source file was
    evaluated (usually from a call to \code{\link{Sys.time}}).
}
\item{\code{sourceFile}:}{ The character-string name of the source file
    used.
}
Note that using the environment does not change the \code{dateCreated}.
}
}

\seealso{
\code{\link{trace}} for the underlying mechanism, and also for the
\code{edit=} argument that can be used for somewhat similar purposes;
that function and also \code{\link{debug}} and
\code{\link{setBreakpoint}}, for techniques more oriented to
traditional debugging styles.
The present function is directly intended for the case that one is
modifying some of the source for an existing package, although it can
be used as well by inserting debugging code in the source (more useful
if the debugging involved is non-trivial).  As noted in the details
section, the source
file need not be the same one in the original package source.
}
\examples{
\dontrun{
## Suppose package P0 has a source file "all.R"
## First, evaluate the source, and from it
## insert the revised version of methods for summary()
  env <- insertSource("./P0/R/all.R", package = "P0",
     methods = "summary")
## now test one of the methods, tracing  the version from the source
  trace("summary", signature = "myMat", browser, edit = env)
## After testing, remove the browser() call but keep the source
  trace("summary", signature = "myMat", edit = env)
## Now insert all the (other) revised functions and methods
## without re-evaluating the source file.
## The package name is included in the object env.
  insertSource(env)
}
}
\keyword{ programming }
\keyword{methods }
