!****h* ROBODoc/H5GLOBAL
!
! NAME
!  MODULE H5GLOBAL
!
! FILE
!  src/fortran/H5f90global.F90
!
! PURPOSE
!  This module is used to pass C stubs for H5 Fortran APIs. The C stubs are
!  packed into arrays in H5_f.c and these arrays are then passed to Fortran.
!  This module then uses EQUIVALENCE to assign elements of the arrays to
!  Fortran equivalent C stubs.
!
! NOTES
!  The size of the C arrays in H5_f.c has to match the values of the variables
!  declared as PARAMETER, hence if the size of an array in H5_f.c is changed
!  then the PARAMETER of that corresponding array in Fortran must also be changed.
!
! USES
!  H5FORTRAN_TYPES 	 - This module is generated at run time. See
!
! COPYRIGHT
! * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
!   Copyright by The HDF Group.                                               *
!   Copyright by the Board of Trustees of the University of Illinois.         *
!   All rights reserved.                                                      *
!                                                                             *
!   This file is part of HDF5.  The full HDF5 copyright notice, including     *
!   terms governing use, modification, and redistribution, is contained in    *
!   the COPYING file, which can be found at the root of the source code       *
!   distribution tree, or in https://support.hdfgroup.org/ftp/HDF5/releases.  *
!   If you do not have access to either file, you may request a copy from     *
!   help@hdfgroup.org.                                                        *
! * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
!
! AUTHOR
!  Elena Pourmal
!
!*****

MODULE H5GLOBAL
  USE H5FORTRAN_TYPES

  IMPLICIT NONE

  ! Enumerate data type that is interoperable with C.
  ENUM, BIND(C)
    ENUMERATOR :: enum_dtype
  END ENUM
  INTEGER, PARAMETER :: ENUM_T = KIND(enum_dtype)
  
  ! Definitions for reference datatypes.
  ! If you change the value of these parameters, do not forget to change corresponding
  ! values in the H5f90.h file.
  INTEGER, PARAMETER :: REF_REG_BUF_LEN = 3

  ! Parameters used in the function 'h5kind_to_type' located in H5_ff.F90.
  ! The flag is used to tell the function whether the kind input variable
  ! is for a REAL or INTEGER data type.

  INTEGER, PARAMETER :: H5_REAL_KIND = 1
  INTEGER, PARAMETER :: H5_INTEGER_KIND = 0

  TYPE :: hobj_ref_t_f
     INTEGER(HADDR_T) ref
  END TYPE hobj_ref_t_f

  TYPE :: hdset_reg_ref_t_f
     INTEGER, DIMENSION(1:REF_REG_BUF_LEN) :: ref
  END TYPE hdset_reg_ref_t_f

  ! Do not forget to change these values when new predefined
  ! datatypes are added
  INTEGER, PARAMETER :: PREDEF_TYPES_LEN = 19
  INTEGER, PARAMETER :: FLOATING_TYPES_LEN = 4
  INTEGER, PARAMETER :: INTEGER_TYPES_LEN = 27

  ! These arrays need to be global because they are used in 
  ! both h5open_f and in h5close_f; initialize to fix linking issues
  ! on OSX and Intel compilers.
  INTEGER(HID_T), DIMENSION(1:PREDEF_TYPES_LEN)   :: predef_types = -1
  INTEGER(HID_T), DIMENSION(1:FLOATING_TYPES_LEN) :: floating_types = -1
  INTEGER(HID_T), DIMENSION(1:INTEGER_TYPES_LEN)  :: integer_types = -1
  
  !DEC$if defined(BUILD_HDF5_DLL)
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_NATIVE_REAL_C_FLOAT
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_NATIVE_REAL_C_DOUBLE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_NATIVE_REAL_C_LONG_DOUBLE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_NATIVE_INTEGER
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_NATIVE_REAL
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_NATIVE_DOUBLE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_NATIVE_CHARACTER 
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_REF_OBJ
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_REF_DSETREG
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_IEEE_F32BE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_IEEE_F32LE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_IEEE_F64BE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_IEEE_F64LE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_I8BE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_I8LE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_I16BE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_I16LE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_I32BE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_I32LE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_I64BE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_I64LE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_U8BE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_U8LE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_U16BE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_U16LE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_U32BE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_U32LE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_U64BE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_U64LE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STRING
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_B8BE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_B8LE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_B16BE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_B16LE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_B32BE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_B32LE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_B64BE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_B64LE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_NATIVE_B8
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_NATIVE_B16
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_NATIVE_B32
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_NATIVE_B64
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_FORTRAN_S1
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_C_S1
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_NATIVE_INTEGER_KIND
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_NATIVE_FLOAT_128
  !DEC$endif
  
  INTEGER(HID_T) :: H5T_NATIVE_REAL_C_FLOAT
  INTEGER(HID_T) :: H5T_NATIVE_REAL_C_DOUBLE
  INTEGER(HID_T) :: H5T_NATIVE_REAL_C_LONG_DOUBLE
  INTEGER(HID_T) :: H5T_NATIVE_INTEGER
  INTEGER(HID_T) :: H5T_NATIVE_REAL
  INTEGER(HID_T) :: H5T_NATIVE_DOUBLE
  INTEGER(HID_T) :: H5T_NATIVE_CHARACTER 
  INTEGER(HID_T) :: H5T_STD_REF_OBJ
  INTEGER(HID_T) :: H5T_STD_REF_DSETREG
  INTEGER(HID_T) :: H5T_IEEE_F32BE
  INTEGER(HID_T) :: H5T_IEEE_F32LE
  INTEGER(HID_T) :: H5T_IEEE_F64BE
  INTEGER(HID_T) :: H5T_IEEE_F64LE
  INTEGER(HID_T) :: H5T_STD_I8BE
  INTEGER(HID_T) :: H5T_STD_I8LE
  INTEGER(HID_T) :: H5T_STD_I16BE
  INTEGER(HID_T) :: H5T_STD_I16LE
  INTEGER(HID_T) :: H5T_STD_I32BE
  INTEGER(HID_T) :: H5T_STD_I32LE
  INTEGER(HID_T) :: H5T_STD_I64BE
  INTEGER(HID_T) :: H5T_STD_I64LE
  INTEGER(HID_T) :: H5T_STD_U8BE
  INTEGER(HID_T) :: H5T_STD_U8LE
  INTEGER(HID_T) :: H5T_STD_U16BE
  INTEGER(HID_T) :: H5T_STD_U16LE
  INTEGER(HID_T) :: H5T_STD_U32BE
  INTEGER(HID_T) :: H5T_STD_U32LE
  INTEGER(HID_T) :: H5T_STD_U64BE
  INTEGER(HID_T) :: H5T_STD_U64LE
  INTEGER(HID_T) :: H5T_STRING
  INTEGER(HID_T) :: H5T_STD_B8BE
  INTEGER(HID_T) :: H5T_STD_B8LE
  INTEGER(HID_T) :: H5T_STD_B16BE
  INTEGER(HID_T) :: H5T_STD_B16LE
  INTEGER(HID_T) :: H5T_STD_B32BE
  INTEGER(HID_T) :: H5T_STD_B32LE
  INTEGER(HID_T) :: H5T_STD_B64BE
  INTEGER(HID_T) :: H5T_STD_B64LE
  INTEGER(HID_T) :: H5T_NATIVE_B8
  INTEGER(HID_T) :: H5T_NATIVE_B16
  INTEGER(HID_T) :: H5T_NATIVE_B32
  INTEGER(HID_T) :: H5T_NATIVE_B64
  INTEGER(HID_T) :: H5T_FORTRAN_S1
  INTEGER(HID_T) :: H5T_C_S1

  INTEGER, PARAMETER :: NUM_NATIVE_INTEGER_KIND = 5
  ! INTEGER*1, INTEGER*2, INTEGER*4, INTEGER*8, INTEGER*16
  INTEGER(HID_T), DIMENSION(1:NUM_NATIVE_INTEGER_KIND) :: H5T_NATIVE_INTEGER_KIND

  INTEGER(HID_T) :: H5T_NATIVE_FLOAT_128 ! NEED IFDEF -MSB-
  !
  ! H5generic flags declaration
  !
  !DEC$if defined(BUILD_HDF5_DLL)
  !DEC$ATTRIBUTES DLLEXPORT :: H5_INDEX_UNKNOWN_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5_INDEX_NAME_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5_INDEX_CRT_ORDER_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5_INDEX_N_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5_ITER_UNKNOWN_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5_ITER_INC_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5_ITER_DEC_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5_ITER_NATIVE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5_ITER_N_F
  !DEC$ATTRIBUTES DLLEXPORT :: HADDR_UNDEF_F
  !DEC$endif
  
  INTEGER :: H5_INDEX_UNKNOWN_F
  INTEGER :: H5_INDEX_NAME_F
  INTEGER :: H5_INDEX_CRT_ORDER_F
  INTEGER :: H5_INDEX_N_F
  INTEGER :: H5_ITER_UNKNOWN_F
  INTEGER :: H5_ITER_INC_F
  INTEGER :: H5_ITER_DEC_F
  INTEGER :: H5_ITER_NATIVE_F
  INTEGER :: H5_ITER_N_F

  INTEGER(HADDR_T) :: HADDR_UNDEF_F
  !
  ! H5F flags declaration
  !
  !DEC$if defined(BUILD_HDF5_DLL)
  !DEC$ATTRIBUTES DLLEXPORT :: H5F_ACC_RDWR_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5F_ACC_RDONLY_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5F_ACC_TRUNC_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5F_ACC_EXCL_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5F_ACC_DEBUG_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5F_SCOPE_LOCAL_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5F_SCOPE_GLOBAL_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5F_CLOSE_DEFAULT_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5F_CLOSE_WEAK_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5F_CLOSE_SEMI_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5F_CLOSE_STRONG_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5F_OBJ_FILE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5F_OBJ_DATASET_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5F_OBJ_GROUP_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5F_OBJ_DATATYPE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5F_OBJ_ALL_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5F_LIBVER_EARLIEST_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5F_LIBVER_LATEST_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5F_UNLIMITED_F
  !DEC$endif
  
  INTEGER :: H5F_ACC_RDWR_F
  INTEGER :: H5F_ACC_RDONLY_F
  INTEGER :: H5F_ACC_TRUNC_F
  INTEGER :: H5F_ACC_EXCL_F
  INTEGER :: H5F_ACC_DEBUG_F
  INTEGER :: H5F_SCOPE_LOCAL_F
  INTEGER :: H5F_SCOPE_GLOBAL_F
  INTEGER :: H5F_CLOSE_DEFAULT_F
  INTEGER :: H5F_CLOSE_WEAK_F
  INTEGER :: H5F_CLOSE_SEMI_F
  INTEGER :: H5F_CLOSE_STRONG_F
  INTEGER :: H5F_OBJ_FILE_F
  INTEGER :: H5F_OBJ_DATASET_F
  INTEGER :: H5F_OBJ_GROUP_F
  INTEGER :: H5F_OBJ_DATATYPE_F
  INTEGER :: H5F_OBJ_ALL_F
  INTEGER :: H5F_LIBVER_EARLIEST_F
  INTEGER :: H5F_LIBVER_LATEST_F
  INTEGER :: H5F_UNLIMITED_F
  !
  ! H5G flags declaration
  !
  !DEC$if defined(BUILD_HDF5_DLL)
  !DEC$ATTRIBUTES DLLEXPORT :: H5G_UNKNOWN_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5G_GROUP_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5G_DATASET_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5G_TYPE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5G_LINK_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5G_UDLINK_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5G_SAME_LOC_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5G_LINK_ERROR_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5G_LINK_HARD_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5G_LINK_SOFT_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5G_STORAGE_TYPE_UNKNOWN_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5G_STORAGE_TYPE_SYMBOL_TABLE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5G_STORAGE_TYPE_COMPACT_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5G_STORAGE_TYPE_DENSE_F
  !DEC$endif
  
  INTEGER :: H5G_UNKNOWN_F
  INTEGER :: H5G_GROUP_F
  INTEGER :: H5G_DATASET_F
  INTEGER :: H5G_TYPE_F
  INTEGER :: H5G_LINK_F
  INTEGER :: H5G_UDLINK_F
  INTEGER :: H5G_SAME_LOC_F
  INTEGER :: H5G_LINK_ERROR_F
  INTEGER :: H5G_LINK_HARD_F
  INTEGER :: H5G_LINK_SOFT_F
  INTEGER :: H5G_STORAGE_TYPE_UNKNOWN_F
  INTEGER :: H5G_STORAGE_TYPE_SYMBOL_TABLE_F
  INTEGER :: H5G_STORAGE_TYPE_COMPACT_F
  INTEGER :: H5G_STORAGE_TYPE_DENSE_F
  !
  ! H5D flags declaration
  !
  !DEC$if defined(BUILD_HDF5_DLL)
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_COMPACT_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_CONTIGUOUS_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_CHUNKED_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_ALLOC_TIME_ERROR_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_ALLOC_TIME_DEFAULT_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_ALLOC_TIME_EARLY_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_ALLOC_TIME_LATE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_ALLOC_TIME_INCR_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_SPACE_STS_ERROR_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_SPACE_STS_NOT_ALLOCATED_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_SPACE_STS_PART_ALLOCATED_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_SPACE_STS_ALLOCATED_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_FILL_TIME_ERROR_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_FILL_TIME_ALLOC_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_FILL_TIME_NEVER_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_FILL_VALUE_ERROR_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_FILL_VALUE_UNDEFINED_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_FILL_VALUE_DEFAULT_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_FILL_VALUE_USER_DEFINED_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_CHUNK_CACHE_NSLOTS_DFLT_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_CHUNK_CACHE_NBYTES_DFLT_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_CHUNK_CACHE_W0_DFLT_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_MPIO_NO_COLLECTIVE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_MPIO_CHUNK_INDEPENDENT_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_MPIO_CHUNK_COLLECTIVE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_MPIO_CHUNK_MIXED_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_MPIO_CONTIG_COLLECTIVE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_VDS_ERROR_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_VDS_FIRST_MISSING_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_VDS_LAST_AVAILABLE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_VIRTUAL_F
  !DEC$endif
  
  INTEGER :: H5D_COMPACT_F
  INTEGER :: H5D_CONTIGUOUS_F
  INTEGER :: H5D_CHUNKED_F

  INTEGER :: H5D_ALLOC_TIME_ERROR_F
  INTEGER :: H5D_ALLOC_TIME_DEFAULT_F
  INTEGER :: H5D_ALLOC_TIME_EARLY_F
  INTEGER :: H5D_ALLOC_TIME_LATE_F
  INTEGER :: H5D_ALLOC_TIME_INCR_F

  INTEGER :: H5D_SPACE_STS_ERROR_F
  INTEGER :: H5D_SPACE_STS_NOT_ALLOCATED_F
  INTEGER :: H5D_SPACE_STS_PART_ALLOCATED_F
  INTEGER :: H5D_SPACE_STS_ALLOCATED_F

  INTEGER :: H5D_FILL_TIME_ERROR_F
  INTEGER :: H5D_FILL_TIME_ALLOC_F
  INTEGER :: H5D_FILL_TIME_NEVER_F

  INTEGER :: H5D_FILL_VALUE_ERROR_F
  INTEGER :: H5D_FILL_VALUE_UNDEFINED_F
  INTEGER :: H5D_FILL_VALUE_DEFAULT_F
  INTEGER :: H5D_FILL_VALUE_USER_DEFINED_F

  ! shortened "_DEFAULT" to "_DFLT" to satisfy the limit of 31
  ! characters for variable names in Fortran.
  ! shortened "_CONTIGUOUS" to "_CONTIG" to satisfy the limit of 31
  ! characters for variable names in Fortran.
  
  INTEGER(SIZE_T) :: H5D_CHUNK_CACHE_NSLOTS_DFLT_F
  INTEGER(SIZE_T) :: H5D_CHUNK_CACHE_NBYTES_DFLT_F

  INTEGER :: H5D_CHUNK_CACHE_W0_DFLT_F
  INTEGER :: H5D_MPIO_NO_COLLECTIVE_F
  INTEGER :: H5D_MPIO_CHUNK_INDEPENDENT_F
  INTEGER :: H5D_MPIO_CHUNK_COLLECTIVE_F
  INTEGER :: H5D_MPIO_CHUNK_MIXED_F
  INTEGER :: H5D_MPIO_CONTIG_COLLECTIVE_F
  INTEGER :: H5D_VDS_ERROR_F
  INTEGER :: H5D_VDS_FIRST_MISSING_F
  INTEGER :: H5D_VDS_LAST_AVAILABLE_F
  INTEGER :: H5D_VIRTUAL_F
  !
  ! H5E flags declaration
  !
  !DEC$if defined(BUILD_HDF5_DLL)
  !DEC$ATTRIBUTES DLLEXPORT :: H5E_DEFAULT_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5E_MAJOR_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5E_MINOR_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5E_WALK_UPWARD_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5E_WALK_DOWNWARD_F
  !DEC$endif

  INTEGER(HID_T) :: H5E_DEFAULT_F
  INTEGER :: H5E_MAJOR_F
  INTEGER :: H5E_MINOR_F
  INTEGER :: H5E_WALK_UPWARD_F
  INTEGER :: H5E_WALK_DOWNWARD_F
  !
  ! H5FD file drivers flags declaration
  !
  !DEC$if defined(BUILD_HDF5_DLL)
  !DEC$ATTRIBUTES DLLEXPORT :: H5FD_MPIO_INDEPENDENT_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5FD_MPIO_COLLECTIVE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5FD_MEM_NOLIST_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5FD_MEM_DEFAULT_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5FD_MEM_SUPER_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5FD_MEM_BTREE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5FD_MEM_DRAW_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5FD_MEM_GHEAP_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5FD_MEM_LHEAP_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5FD_MEM_OHDR_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5FD_MEM_NTYPES_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5FD_CORE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5FD_FAMILY_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5FD_LOG_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5FD_MPIO_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5FD_MULTI_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5FD_SEC2_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5FD_STDIO_F
  !DEC$endif

  INTEGER :: H5FD_MPIO_INDEPENDENT_F
  INTEGER :: H5FD_MPIO_COLLECTIVE_F
  INTEGER :: H5FD_MEM_NOLIST_F
  INTEGER :: H5FD_MEM_DEFAULT_F
  INTEGER :: H5FD_MEM_SUPER_F
  INTEGER :: H5FD_MEM_BTREE_F
  INTEGER :: H5FD_MEM_DRAW_F
  INTEGER :: H5FD_MEM_GHEAP_F
  INTEGER :: H5FD_MEM_LHEAP_F
  INTEGER :: H5FD_MEM_OHDR_F
  INTEGER :: H5FD_MEM_NTYPES_F
  INTEGER(HID_T) :: H5FD_CORE_F
  INTEGER(HID_T) :: H5FD_FAMILY_F
  INTEGER(HID_T) :: H5FD_LOG_F
  INTEGER(HID_T) :: H5FD_MPIO_F
  INTEGER(HID_T) :: H5FD_MULTI_F
  INTEGER(HID_T) :: H5FD_SEC2_F
  INTEGER(HID_T) :: H5FD_STDIO_F
  !
  ! H5I flags declaration
  !
  !DEC$if defined(BUILD_HDF5_DLL)
  !DEC$ATTRIBUTES DLLEXPORT :: H5I_FILE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5I_GROUP_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5I_DATATYPE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5I_DATASPACE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5I_DATASET_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5I_ATTR_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5I_BADID_F
  !DEC$endif

  INTEGER ::  H5I_FILE_F
  INTEGER ::  H5I_GROUP_F
  INTEGER ::  H5I_DATATYPE_F
  INTEGER ::  H5I_DATASPACE_F
  INTEGER ::  H5I_DATASET_F
  INTEGER ::  H5I_ATTR_F
  INTEGER ::  H5I_BADID_F
  !
  ! H5L flags declaration
  !
  !DEC$if defined(BUILD_HDF5_DLL)
  !DEC$ATTRIBUTES DLLEXPORT :: H5L_TYPE_ERROR_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5L_TYPE_HARD_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5L_TYPE_SOFT_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5L_TYPE_EXTERNAL_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5L_SAME_LOC_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5L_LINK_CLASS_T_VERS_F
  !DEC$endif
  
  INTEGER :: H5L_TYPE_ERROR_F
  INTEGER :: H5L_TYPE_HARD_F
  INTEGER :: H5L_TYPE_SOFT_F
  INTEGER :: H5L_TYPE_EXTERNAL_F
  INTEGER :: H5L_SAME_LOC_F
  INTEGER :: H5L_LINK_CLASS_T_VERS_F
  !
  ! H5O flags declaration
  !
  !DEC$if defined(BUILD_HDF5_DLL)
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_COPY_SHALLOW_HIERARCHY_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_COPY_EXPAND_SOFT_LINK_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_COPY_EXPAND_EXT_LINK_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_COPY_EXPAND_REFERENCE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_COPY_WITHOUT_ATTR_FLAG_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_COPY_PRESERVE_NULL_FLAG_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_COPY_ALL_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_SHMESG_NONE_FLAG_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_SHMESG_SDSPACE_FLAG_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_SHMESG_DTYPE_FLAG_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_SHMESG_FILL_FLAG_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_SHMESG_PLINE_FLAG_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_SHMESG_ATTR_FLAG_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_SHMESG_ALL_FLAG_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_HDR_CHUNK0_SIZE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_HDR_ATTR_CRT_ORDER_TRACK_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_HDR_ATTR_CRT_ORDER_INDEX_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_HDR_ATTR_STORE_PHASE_CHA_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_HDR_STORE_TIMES_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_HDR_ALL_FLAGS_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_SHMESG_MAX_NINDEXES_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_SHMESG_MAX_LIST_SIZE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_TYPE_UNKNOWN_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_TYPE_GROUP_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_TYPE_DATASET_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_TYPE_NAMED_DATATYPE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_TYPE_NTYPES_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_INFO_ALL_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_INFO_BASIC_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_INFO_TIME_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_INFO_NUM_ATTRS_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_INFO_HDR_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_INFO_META_SIZE_F
  !
  !DEC$endif
  
  INTEGER :: H5O_COPY_SHALLOW_HIERARCHY_F ! *** THESE VARIABLES DO
  INTEGER :: H5O_COPY_EXPAND_SOFT_LINK_F  ! NOT MATCH THE C VARIABLE
  INTEGER :: H5O_COPY_EXPAND_EXT_LINK_F   ! IN ORDER
  INTEGER :: H5O_COPY_EXPAND_REFERENCE_F  ! TO STAY UNDER THE
  INTEGER :: H5O_COPY_WITHOUT_ATTR_FLAG_F
  INTEGER :: H5O_COPY_PRESERVE_NULL_FLAG_F
  INTEGER :: H5O_COPY_ALL_F
  INTEGER :: H5O_SHMESG_NONE_FLAG_F
  INTEGER :: H5O_SHMESG_SDSPACE_FLAG_F
  INTEGER :: H5O_SHMESG_DTYPE_FLAG_F
  INTEGER :: H5O_SHMESG_FILL_FLAG_F
  INTEGER :: H5O_SHMESG_PLINE_FLAG_F
  INTEGER :: H5O_SHMESG_ATTR_FLAG_F
  INTEGER :: H5O_SHMESG_ALL_FLAG_F
  INTEGER :: H5O_HDR_CHUNK0_SIZE_F
  INTEGER :: H5O_HDR_ATTR_CRT_ORDER_TRACK_F ! 32 CHARACTER
  INTEGER :: H5O_HDR_ATTR_CRT_ORDER_INDEX_F ! VARIABLE
  INTEGER :: H5O_HDR_ATTR_STORE_PHASE_CHA_F ! LENGTH ***
  INTEGER :: H5O_HDR_STORE_TIMES_F
  INTEGER :: H5O_HDR_ALL_FLAGS_F
  INTEGER :: H5O_SHMESG_MAX_NINDEXES_F
  INTEGER :: H5O_SHMESG_MAX_LIST_SIZE_F
  INTEGER :: H5O_TYPE_UNKNOWN_F
  INTEGER :: H5O_TYPE_GROUP_F
  INTEGER :: H5O_TYPE_DATASET_F
  INTEGER :: H5O_TYPE_NAMED_DATATYPE_F
  INTEGER :: H5O_TYPE_NTYPES_F
  INTEGER :: H5O_INFO_ALL_F
  INTEGER :: H5O_INFO_BASIC_F
  INTEGER :: H5O_INFO_TIME_F
  INTEGER :: H5O_INFO_NUM_ATTRS_F
  INTEGER :: H5O_INFO_HDR_F
  INTEGER :: H5O_INFO_META_SIZE_F
  !
  ! H5P flags declaration
  !
  !DEC$if defined(BUILD_HDF5_DLL)
  !DEC$ATTRIBUTES DLLEXPORT :: H5P_FILE_CREATE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5P_FILE_ACCESS_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5P_DATASET_CREATE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5P_DATASET_XFER_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5P_FILE_MOUNT_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5P_DEFAULT_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5P_ROOT_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5P_OBJECT_CREATE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5P_DATASET_ACCESS_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5P_GROUP_CREATE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5P_GROUP_ACCESS_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5P_DATATYPE_CREATE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5P_DATATYPE_ACCESS_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5P_STRING_CREATE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5P_ATTRIBUTE_CREATE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5P_OBJECT_COPY_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5P_LINK_CREATE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5P_LINK_ACCESS_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5P_CRT_ORDER_INDEXED_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5P_CRT_ORDER_TRACKED_F
  !DEC$endif

  INTEGER(HID_T) :: H5P_FILE_CREATE_F
  INTEGER(HID_T) :: H5P_FILE_ACCESS_F
  INTEGER(HID_T) :: H5P_DATASET_CREATE_F
  INTEGER(HID_T) :: H5P_DATASET_XFER_F
  INTEGER(HID_T) :: H5P_FILE_MOUNT_F
  INTEGER(HID_T) :: H5P_DEFAULT_F
  INTEGER(HID_T) :: H5P_ROOT_F
  INTEGER(HID_T) :: H5P_OBJECT_CREATE_F
  INTEGER(HID_T) :: H5P_DATASET_ACCESS_F
  INTEGER(HID_T) :: H5P_GROUP_CREATE_F
  INTEGER(HID_T) :: H5P_GROUP_ACCESS_F
  INTEGER(HID_T) :: H5P_DATATYPE_CREATE_F
  INTEGER(HID_T) :: H5P_DATATYPE_ACCESS_F
  INTEGER(HID_T) :: H5P_STRING_CREATE_F
  INTEGER(HID_T) :: H5P_ATTRIBUTE_CREATE_F
  INTEGER(HID_T) :: H5P_OBJECT_COPY_F
  INTEGER(HID_T) :: H5P_LINK_CREATE_F
  INTEGER(HID_T) :: H5P_LINK_ACCESS_F

  INTEGER :: H5P_CRT_ORDER_INDEXED_F
  INTEGER :: H5P_CRT_ORDER_TRACKED_F
  !
  ! H5R flags declaration
  !
  !DEC$if defined(BUILD_HDF5_DLL)
  !DEC$ATTRIBUTES DLLEXPORT :: H5R_OBJECT_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5R_DATASET_REGION_F
  !DEC$endif

  INTEGER :: H5R_OBJECT_F
  INTEGER :: H5R_DATASET_REGION_F
  !
  ! H5S flags declaration
  !
  !DEC$if defined(BUILD_HDF5_DLL)
  !DEC$ATTRIBUTES DLLEXPORT :: H5S_UNLIMITED_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5S_ALL_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5S_SCALAR_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5S_SIMPLE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5S_NULL_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5S_SELECT_NOOP_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5S_SELECT_SET_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5S_SELECT_OR_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5S_SELECT_AND_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5S_SELECT_XOR_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5S_SELECT_NOTB_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5S_SELECT_NOTA_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5S_SELECT_APPEND_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5S_SELECT_PREPEND_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5S_SELECT_INVALID_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5S_SEL_ERROR_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5S_SEL_NONE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5S_SEL_POINTS_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5S_SEL_HYPERSLABS_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5S_SEL_ALL_F
  !DEC$endif
  
  INTEGER(HSIZE_T) :: H5S_UNLIMITED_F

  INTEGER(HID_T) :: H5S_ALL_F

  INTEGER :: H5S_SCALAR_F
  INTEGER :: H5S_SIMPLE_F
  INTEGER :: H5S_NULL_F

  INTEGER :: H5S_SELECT_NOOP_F
  INTEGER :: H5S_SELECT_SET_F
  INTEGER :: H5S_SELECT_OR_F
  INTEGER :: H5S_SELECT_AND_F
  INTEGER :: H5S_SELECT_XOR_F
  INTEGER :: H5S_SELECT_NOTB_F
  INTEGER :: H5S_SELECT_NOTA_F
  INTEGER :: H5S_SELECT_APPEND_F
  INTEGER :: H5S_SELECT_PREPEND_F
  INTEGER :: H5S_SELECT_INVALID_F

  INTEGER :: H5S_SEL_ERROR_F
  INTEGER :: H5S_SEL_NONE_F
  INTEGER :: H5S_SEL_POINTS_F
  INTEGER :: H5S_SEL_HYPERSLABS_F
  INTEGER :: H5S_SEL_ALL_F
  !
  ! H5T flags declaration
  !
  !DEC$if defined(BUILD_HDF5_DLL)
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_NO_CLASS_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_INTEGER_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_FLOAT_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_TIME_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STRING_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_BITFIELD_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_OPAQUE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_COMPOUND_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_REFERENCE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_ENUM_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_VLEN_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_ARRAY_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_ORDER_LE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_ORDER_BE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_ORDER_VAX_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_ORDER_NONE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_ORDER_MIXED_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_PAD_ZERO_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_PAD_ONE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_PAD_BACKGROUND_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_PAD_ERROR_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_SGN_NONE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_SGN_2_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_SGN_ERROR_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_NORM_IMPLIED_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_NORM_MSBSET_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_NORM_NONE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_CSET_ASCII_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_CSET_UTF8_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STR_NULLTERM_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STR_NULLPAD_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STR_SPACEPAD_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STR_ERROR_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_DIR_ASCEND_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_DIR_DESCEND_F
  !DEC$endif

  INTEGER :: H5T_NO_CLASS_F
  INTEGER :: H5T_INTEGER_F
  INTEGER :: H5T_FLOAT_F
  INTEGER :: H5T_TIME_F
  INTEGER :: H5T_STRING_F
  INTEGER :: H5T_BITFIELD_F
  INTEGER :: H5T_OPAQUE_F
  INTEGER :: H5T_COMPOUND_F
  INTEGER :: H5T_REFERENCE_F
  INTEGER :: H5T_ENUM_F
  INTEGER :: H5T_VLEN_F
  INTEGER :: H5T_ARRAY_F
  INTEGER :: H5T_ORDER_LE_F
  INTEGER :: H5T_ORDER_BE_F
  INTEGER :: H5T_ORDER_VAX_F
  INTEGER :: H5T_ORDER_NONE_F
  INTEGER :: H5T_ORDER_MIXED_F
  INTEGER :: H5T_PAD_ZERO_F
  INTEGER :: H5T_PAD_ONE_F
  INTEGER :: H5T_PAD_BACKGROUND_F
  INTEGER :: H5T_PAD_ERROR_F
  INTEGER :: H5T_SGN_NONE_F
  INTEGER :: H5T_SGN_2_F
  INTEGER :: H5T_SGN_ERROR_F
  INTEGER :: H5T_NORM_IMPLIED_F
  INTEGER :: H5T_NORM_MSBSET_F
  INTEGER :: H5T_NORM_NONE_F
  INTEGER :: H5T_CSET_ASCII_F
  INTEGER :: H5T_CSET_UTF8_F
  INTEGER :: H5T_STR_NULLTERM_F
  INTEGER :: H5T_STR_NULLPAD_F
  INTEGER :: H5T_STR_SPACEPAD_F
  INTEGER :: H5T_STR_ERROR_F
  INTEGER :: H5T_DIR_ASCEND_F
  INTEGER :: H5T_DIR_DESCEND_F
  !
  ! H5Z flags declaration
  !
  !DEC$if defined(BUILD_HDF5_DLL)
  !DEC$ATTRIBUTES DLLEXPORT :: H5Z_FILTER_ERROR_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5Z_FILTER_NONE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5Z_FILTER_ALL_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5Z_FILTER_DEFLATE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5Z_FILTER_SHUFFLE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5Z_FILTER_FLETCHER32_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5Z_FILTER_SZIP_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5Z_ERROR_EDC_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5Z_DISABLE_EDC_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5Z_ENABLE_EDC_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5Z_NO_EDC_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5Z_FLAG_OPTIONAL_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5Z_FILTER_ENCODE_ENABLED_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5Z_FILTER_DECODE_ENABLED_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5Z_FILTER_NBIT_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5Z_FILTER_SCALEOFFSET_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5Z_SO_FLOAT_DSCALE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5Z_SO_FLOAT_ESCALE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5Z_SO_INT_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5Z_SO_INT_MINBITS_DEFAULT_F
  !DEC$endif

  INTEGER :: H5Z_FILTER_ERROR_F
  INTEGER :: H5Z_FILTER_NONE_F
  INTEGER :: H5Z_FILTER_ALL_F
  INTEGER :: H5Z_FILTER_DEFLATE_F
  INTEGER :: H5Z_FILTER_SHUFFLE_F
  INTEGER :: H5Z_FILTER_FLETCHER32_F
  INTEGER :: H5Z_FILTER_SZIP_F
  INTEGER :: H5Z_ERROR_EDC_F
  INTEGER :: H5Z_DISABLE_EDC_F
  INTEGER :: H5Z_ENABLE_EDC_F
  INTEGER :: H5Z_NO_EDC_F
  INTEGER :: H5Z_FLAG_OPTIONAL_F
  INTEGER :: H5Z_FILTER_ENCODE_ENABLED_F
  INTEGER :: H5Z_FILTER_DECODE_ENABLED_F
  INTEGER :: H5Z_FILTER_NBIT_F
  INTEGER :: H5Z_FILTER_SCALEOFFSET_F
  INTEGER :: H5Z_SO_FLOAT_DSCALE_F
  INTEGER :: H5Z_SO_FLOAT_ESCALE_F
  INTEGER :: H5Z_SO_INT_F
  INTEGER :: H5Z_SO_INT_MINBITS_DEFAULT_F
  !
  ! H5 Library flags declaration
  !
  !DEC$if defined(BUILD_HDF5_DLL)
  !DEC$ATTRIBUTES DLLEXPORT :: H5_SZIP_EC_OM_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5_SZIP_NN_OM_F
  !DEC$endif

  INTEGER :: H5_SZIP_EC_OM_F
  INTEGER :: H5_SZIP_NN_OM_F

CONTAINS

  ! Copy a c string to a Fortran string

  SUBROUTINE H5_Fortran_string_c2f(c_string, f_string)

    USE, INTRINSIC :: ISO_C_BINDING
    IMPLICIT NONE
    CHARACTER(KIND=C_CHAR, LEN=*), INTENT(IN) :: c_string
    CHARACTER(LEN=*), INTENT(OUT) :: f_string
    INTEGER(SIZE_T) :: c_len, f_len

    ! Find the length of the C string by locating the null terminator
    c_len = MAX(INDEX(c_string,C_NULL_CHAR, KIND=SIZE_T)-1_SIZE_T,1_SIZE_T)
    ! Find the length of the Fortran string
    f_len = LEN(f_string)

    ! CASE (1): C string is equal to or larger than Fortran character buffer,
    !           so fill the entire Fortran buffer. 
    IF(c_len.GE.f_len)THEN !
       f_string(1:f_len) = c_string(1:f_len)

    ! CASE (2): C string is smaller than Fortran character buffer, 
    !           so copy C string and blank pad remaining characters.
    ELSE
       f_string(1:c_len) = c_string(1:c_len)
       f_string(c_len+1:f_len) =' '
    ENDIF
  END SUBROUTINE H5_Fortran_string_c2f

  SUBROUTINE H5_Fortran_string_f2c(f_string, c_string)

    USE, INTRINSIC :: ISO_C_BINDING
    IMPLICIT NONE
    CHARACTER(LEN=*), INTENT(IN) :: f_string
    CHARACTER(KIND=C_CHAR, LEN=*), INTENT(OUT) :: c_string

    c_string = TRIM(f_string)//C_NULL_CHAR

  END SUBROUTINE H5_Fortran_string_f2c


! Copy Fortran string to C charater array, assuming the C array is one-char
! longer for the terminating null char.
! fstring : the Fortran input string
! cstring : the C output string (with memory already allocated)
!!$subroutine MPIR_Fortran_string_f2c(fstring, cstring)
!!$    implicit none
!!$    character(len=*), intent(in) :: fstring
!!$    character(kind=c_char), intent(out) :: cstring(:)
!!$    integer :: i, j
!!$    logical :: met_non_blank
!!$
!!$    ! Trim the leading and trailing blank characters
!!$    j = 1
!!$    met_non_blank = .false.
!!$    do i = 1, len_trim(fstring)
!!$        if (met_non_blank) then
!!$            cstring(j) = fstring(i:i)
!!$            j = j + 1
!!$        else if (fstring(i:i) /= ' ') then
!!$            met_non_blank = .true.
!!$            cstring(j) = fstring(i:i)
!!$            j = j + 1
!!$        end if
!!$    end do
!!$
!!$    cstring(j) = C_NULL_CHAR
!!$end subroutine MPIR_Fortran_string_f2c
!!$
!!$! Copy C charater array to Fortran string
!!$subroutine MPIR_Fortran_string_c2f(cstring, fstring)
!!$    implicit none
!!$    character(kind=c_char), intent(in) :: cstring(:)
!!$    character(len=*), intent(out) :: fstring
!!$    integer :: i, j, length
!!$
!!$    i = 1
!!$    do while (cstring(i) /= C_NULL_CHAR)
!!$        fstring(i:i) = cstring(i)
!!$        i = i + 1
!!$    end do
!!$
!!$    ! Zero out the trailing characters
!!$    length = len(fstring)
!!$    do j = i, length
!!$        fstring(j:j) = ' '
!!$    end do
!!$end subroutine MPIR_Fortran_string_c2f
  
END MODULE H5GLOBAL

