/** @file

  Copyright (c) 2017 - 2022, Arm Limited. All rights reserved.

  SPDX-License-Identifier: BSD-2-Clause-Patent

  @par Glossary:
    - Cm or CM   - Configuration Manager
    - Obj or OBJ - Object
    - Std or STD - Standard
**/

#ifndef STANDARD_NAMESPACE_OBJECTS_H_
#define STANDARD_NAMESPACE_OBJECTS_H_

#include <AcpiTableGenerator.h>
#include <SmbiosTableGenerator.h>

#pragma pack(1)

/** A macro defining a reserved zero/NULL token value that
    does not identify any object.
*/
#define CM_NULL_TOKEN  0

/** A reference token that the Configuration Manager can use
    to identify a Configuration Manager object.

  This can be used to differentiate between instances of
  objects of the same types. When bit 0 is clear, the
  identification scheme is implementation defined and is
  defined by the Configuration Manager.

  Tokens are also used by table generators to look up
  previously generated tables. To aid with this, the concept
  of an abstract token exists, allowing generators to
  define arbitrary tokens to identify generated tables.
  When bit 1 is set, the token is an abstract token and
  refers to generated tables. The format of an abstract
  token is shown below.

  Typically the token is used to identify a specific instance
  from a set of objects in a call to the GetObject()/SetObject(),
  implemented by the Configuration Manager protocol.

  Note: The token value 0 is reserved for a NULL token and does
        not identify any object.
**/
typedef UINTN CM_OBJECT_TOKEN;

/** The EABSTRACT_TOKEN_ID enum describes the defined namespaces
    for Abstract Tokens.
*/
typedef enum AbstractTokenID {
  ETokenNameSpaceUnknown = 0,
  ETokenNameSpaceSmbios,
  ETokenNameSpaceFdtHwInfo,
  ETokenNameSpaceGenerated,

  ETokenNameSpaceMax
} EABSTRACT_TOKEN_ID;

/** The EFDT_HW_INFO_OBJECT_ID enum describes the defined object
    types for the ETokenNameSpaceFdtHwInfo namespace.
*/
typedef enum FdtHwInfoObjectID {
  EFdtHwInfoIortObject = 0
} EFDT_HW_INFO_OBJECT_ID;

/** Abstract token generated by table generators

  Description of abstract token format
______________________________________________________________________________
|31 |30 |29 |28 || 27 | 26 | 25 | 24 || 23 | 22 | 21 | 20 | 19 | 18 | 17 | 16|
------------------------------------------------------------------------------
| 1   0   1   0 ||   Name Space ID   ||              Object ID               |
______________________________________________________________________________

Bits: [31:28] - Must be 1010 (0xa) for abstract token
Bits: [27:24] - Name Space ID
                0000 - Unknown
                0001 - SMBIOS
Bits: [23:16] - Object ID
_______________________________________________________________________________
|15 |14 |13 |12 || 11 | 10 |  9 |  8 ||  7 |  6 |  5 |  4 ||  3 |  2 |  1 |  0|
-------------------------------------------------------------------------------
|                                  ID                                     | 1 |
_______________________________________________________________________________

Bits: [15:1] - Token ID
               Token ID is arbitrary and is defined by the generator
Bit [0] - Must be 1 for abstract token

Object ID's in the SMBIOS Namespace:
   See ESTD_SMBIOS_TABLE_ID.
*/
#define CM_ABSTRACT_TOKEN_MAKE(ns, obj_id, id)  ((CM_OBJECT_TOKEN)((0xa << 28) | ((ns) << 24) | ((obj_id) << 16) | ((id) << 1) | 1))

/**
  A type representing a SMBIOS structure/table type.

  Types 0 through 127 (7Fh) are reserved for and defined by the
  SMBIOS specification.
  Types 128 through 256 (80h to FFh) are available for system and
  OEM-specific information.

  Note: This Dynamic SMBIOS table generation implementation defines
  TableType FFh as a NULL table which is used by the Dynamic
  SMBIOS table dispatcher to terminate the dependency sequence.
*/
typedef UINT8 SMBIOS_TABLE_TYPE;

/** The ESTD_OBJECT_ID enum describes the Object IDs
    in the Standard Namespace.
*/
typedef enum StdObjectID {
  EStdObjCfgMgrInfo = 0x00000000, ///< 0 - Configuration Manager Info
  EStdObjAcpiTableList,           ///< 1 - ACPI table Info List
  EStdObjSmbiosTableList,         ///< 2 - SMBIOS table Info List
  EStdObjMax
} ESTD_OBJECT_ID;

/** A structure that describes the Configuration Manager Information.
*/
typedef struct CmStdObjConfigurationManagerInfo {
  /// The Configuration Manager Revision.
  UINT32    Revision;

  /** The OEM ID. This information is used to
      populate the ACPI table header information.
  */
  UINT8     OemId[6];
} CM_STD_OBJ_CONFIGURATION_MANAGER_INFO;

/** A structure used to describe the ACPI table generators to be invoked.

  The AcpiTableData member of this structure may be used to directly provide
  the binary ACPI table data which is required by the following standard
  generators:
    - RAW
    - DSDT
    - SSDT

  Providing the ACPI table data is optional and depends on the generator
  that is being invoked. If unused, set AcpiTableData to NULL.
*/
typedef struct CmAStdObjAcpiTableInfo {
  /// The signature of the ACPI Table to be installed
  UINT32                         AcpiTableSignature;

  /// The ACPI table revision
  UINT8                          AcpiTableRevision;

  /// The ACPI Table Generator ID
  ACPI_TABLE_GENERATOR_ID        TableGeneratorId;

  /// Optional pointer to the ACPI table data
  EFI_ACPI_DESCRIPTION_HEADER    *AcpiTableData;

  /// An OEM-supplied string that the OEM uses to identify the particular
  /// data table. This field is particularly useful when defining a definition
  /// block to distinguish definition block functions. The OEM assigns each
  /// dissimilar table a new OEM Table ID.
  /// This field could be constructed using the SIGNATURE_64() macro.
  ///   e.g. SIGNATURE_64 ('A','R','M','H','G','T','D','T')
  /// Note: If this field is not populated (has value of Zero), then the
  /// Generators shall populate this information using part of the
  /// CM_STD_OBJ_CONFIGURATION_MANAGER_INFO.OemId field and the
  /// ACPI table signature.
  UINT64    OemTableId;

  /// An OEM-supplied revision number. Larger numbers are assumed to be
  /// newer revisions.
  /// Note: If this field is not populated (has value of Zero), then the
  /// Generators shall populate this information using the revision of the
  /// Configuration Manager (CM_STD_OBJ_CONFIGURATION_MANAGER_INFO.Revision).
  UINT32    OemRevision;

  /// The minor revision of an ACPI table if required by the table.
  /// Note: If this field is not populated (has value of Zero), then the
  /// Generators shall populate this information based on the latest minor
  /// revision of the table that is supported by the generator.
  /// e.g. This field can be used to specify the minor revision to be set
  /// for the FADT table.
  UINT8     MinorRevision;
} CM_STD_OBJ_ACPI_TABLE_INFO;

/** A structure used to describe the SMBIOS table generators to be invoked.

  The SmbiosTableData member of this structure is used to provide
  the SMBIOS table data which is required by the following standard
  generator(s):
    - RAW

  Providing the SMBIOS table data is optional and depends on the
  generator that is being invoked. If unused, set the SmbiosTableData
  to NULL.
*/
typedef struct CmStdObjSmbiosTableInfo {
  /// SMBIOS Structure/Table Type
  SMBIOS_TABLE_TYPE            TableType;

  /// The SMBIOS Table Generator ID
  SMBIOS_TABLE_GENERATOR_ID    TableGeneratorId;

  /// Optional pointer to the SMBIOS table data
  SMBIOS_STRUCTURE             *SmbiosTableData;
} CM_STD_OBJ_SMBIOS_TABLE_INFO;

#pragma pack()

#endif // STANDARD_NAMESPACE_OBJECTS_H_
