# Copyright © The Debusine Developers
# See the AUTHORS file at the top-level directory of this distribution
#
# This file is part of Debusine. It is subject to the license terms
# in the LICENSE file found in the top-level directory of this
# distribution. No part of Debusine, including this file, may be copied,
# modified, propagated, or distributed except according to the terms
# contained in the LICENSE file.

"""Unit tests for the server-side no-operation task."""

from unittest import TestCase

from debusine.artifacts.models import WorkRequestResults
from debusine.server.tasks import ServerNoop
from debusine.tasks import TaskConfigError


class ServerNoopTests(TestCase):
    """Test the ServerNoop task."""

    def test_execute_result_true(self) -> None:
        """If result=SUCCESS, the task returns SUCCESS."""
        noop = ServerNoop({"result": WorkRequestResults.SUCCESS})
        self.assertEqual(noop._execute(), WorkRequestResults.SUCCESS)

    def test_execute_result_false(self) -> None:
        """If result=FAILURE, the task returns FAILURE."""
        noop = ServerNoop({"result": WorkRequestResults.FAILURE})
        self.assertEqual(noop._execute(), WorkRequestResults.FAILURE)

    def test_execute_exception(self) -> None:
        """If exception=True, the task raises an exception."""
        noop = ServerNoop({"exception": True})
        self.assertRaisesRegex(
            RuntimeError, "Client requested an exception", noop._execute
        )

    def test_no_additional_properties(self) -> None:
        """Assert no additional properties."""
        error_msg = "Extra inputs are not permitted"
        with self.assertRaisesRegex(TaskConfigError, error_msg):
            ServerNoop({"input": {}})

    def test_label(self) -> None:
        """Test get_label."""
        task = ServerNoop({"result": WorkRequestResults.SUCCESS})
        self.assertEqual(task.get_label(), "noop")
