/*
 * ratPGPprog.c --
 *
 *	This file contains compatibility functions.
 *
 * TkRat software and its included text is Copyright 1996-2000 by
 * Martin Forssn
 *
 * The full text of the legal notice is contained in the file called
 * COPYRIGHT, included with this distribution.
 */

#include "ratFolder.h"
#include "ratPGP.h"

/*
 * Maximaum length of pass phrase (plus two)
 */
#define MAXPASSLENGTH 1024

/*
 * Cached pass phrase
 */
static char pgpPass[MAXPASSLENGTH];
static int pgpPassValid = 0;
static Tcl_TimerToken pgpPassToken;


/*
 *----------------------------------------------------------------------
 *
 * ClearPGPPass --
 *
 *      Clear the pgp pass phrase
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The pass phrase is cleared.
 *
 *
 *----------------------------------------------------------------------
 */

void
ClearPGPPass(ClientData unused)
{
    memset(pgpPass, '\0', sizeof(pgpPass));
    pgpPassValid = 0;
}


/*
 *----------------------------------------------------------------------
 *
 * RatPGPPhrase --
 *
 *      Get the pass phrase.
 *
 * Results:
 *	A pointer to a buffer containing the pass phrase or NULL if the
 *	user aborted the operation.
 *
 * Side effects:
 *	It is the callers responsibility to free this buffer.
 *
 *----------------------------------------------------------------------
 */

char*
RatPGPPhrase(Tcl_Interp *interp)
{
    char buf[32], **argv, *result;
    int doCache, timeout, argc;

    Tcl_GetInt(interp,
	       Tcl_GetVar2(interp,"option","cache_pgp_timeout",TCL_GLOBAL_ONLY),
	       &timeout);

    if (pgpPassValid) {
	if (pgpPassToken) {
	    Tcl_DeleteTimerHandler(pgpPassToken);
	}
	if (timeout) {
	    pgpPassToken = Tcl_CreateTimerHandler(timeout*1000, ClearPGPPass,
		    NULL);
	}
	return cpystr(pgpPass);
    }

    RatStrNCpy(buf, "RatGetPGPPassPhrase", sizeof(buf));
    Tcl_Eval(interp, buf);
    Tcl_SplitList(interp, Tcl_GetStringResult(interp), &argc, &argv);
    if (!strcmp("ok", argv[0])) {
	Tcl_GetBoolean(interp,
		   Tcl_GetVar2(interp, "option", "cache_pgp", TCL_GLOBAL_ONLY),
		   &doCache);
	if (doCache) {
	    RatStrNCpy(pgpPass, argv[1], sizeof(pgpPass));
	    pgpPassValid = 1;
	    if (timeout) {
		pgpPassToken = Tcl_CreateTimerHandler(timeout*1000,
			ClearPGPPass, NULL);
	    } else {
		pgpPassToken = NULL;
	    }
	}
	result = cpystr(argv[1]);
	ckfree(argv);
	return result;
    } else {
	ckfree(argv);
	return NULL;
    }
}



/*
 *----------------------------------------------------------------------
 *
 * RatSenderPGPPhrase --
 *
 *      Get the pass phrase. This function may only be called from the
 *	sender process.
 *
 * Results:
 *	A pointer to a static buffer containing the pass phrase. It is up to
 *	the caller to overwrite this buffer with nulls.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

char*
RatSenderPGPPhrase(Tcl_Interp *interp)
{
    static char **argv = NULL;
    int argc;
    char *result = RatSendPGPCommand("PGP getpass");

    if (!strncmp("PHRASE ", result, 7)) {
	if (argv) {
	    ckfree(argv);
	}
	Tcl_SplitList(interp, result, &argc, &argv);
	memset(result, '\0', strlen(result));
	return argv[1];
    } else {
	return NULL;
    }
}


/*
 *----------------------------------------------------------------------
 *
 * RatPGPBodyCheck --
 *
 *      Checks if the given bodypart is either signed or encoded with
 *	pgp. Parts of the bodypart signature are then initialized.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The passed BodyInfo structure is modified.
 *
 *
 *----------------------------------------------------------------------
 */

void
RatPGPBodyCheck(Tcl_Interp *interp, MessageProcInfo *procInfo,
		BodyInfo **bodyInfoPtrPtr)
{
    PARAMETER *parPtr;
    int alg, prot, enc;
    unsigned long length;
    char *text, *start, *end, *middle, *version;

    version = Tcl_GetVar2(interp, "option", "pgp_version", TCL_GLOBAL_ONLY);
    if (!version || !strcmp("0", version)) {
	return;
    }

    /*
     * Check for PGP/MIME messages
     */
    (*bodyInfoPtrPtr)->sigStatus = RAT_UNSIGNED;
    if ((*bodyInfoPtrPtr)->bodyPtr->type == TYPEMULTIPART
	    && !strcasecmp("encrypted", (*bodyInfoPtrPtr)->bodyPtr->subtype)) {
	enc = 0;
	for (parPtr = (*bodyInfoPtrPtr)->bodyPtr->parameter; parPtr;
		parPtr = parPtr->next) {
	    if (!strcasecmp(parPtr->attribute, "protocol")
		    && !strcasecmp(parPtr->value, "application/pgp-encrypted")){
		enc = 1;
		break;
	    }
	}
	if (enc) {
	    RatPGPDecrypt(interp, procInfo, bodyInfoPtrPtr);
	    (*bodyInfoPtrPtr)->encoded = 1;
	}

    } else if ((*bodyInfoPtrPtr)->bodyPtr->type == TYPEMULTIPART
	    && !strcasecmp("signed", (*bodyInfoPtrPtr)->bodyPtr->subtype)) {
	alg = prot = 0;
	for (parPtr = (*bodyInfoPtrPtr)->bodyPtr->parameter; parPtr;
		parPtr = parPtr->next) {
	    if (!strcasecmp(parPtr->attribute, "micalg")
		&& (!strcasecmp(parPtr->value, "pgp-md5")
		    || !strcasecmp(parPtr->value, "pgp-sha1"))) {
		alg = 1;
	    } else if (!strcasecmp(parPtr->attribute, "protocol")
		    && !strcasecmp(parPtr->value, "application/pgp-signature")){
		prot = 1;
	    }
	}
	if (alg && prot) {
	    BodyInfo *bodyInfoPtr;

	    (*procInfo[(*bodyInfoPtrPtr)->type].makeChildrenProc)(interp,
		    *bodyInfoPtrPtr);
	    bodyInfoPtr = *bodyInfoPtrPtr;
	    *bodyInfoPtrPtr = (*bodyInfoPtrPtr)->firstbornPtr;
	    (*bodyInfoPtrPtr)->sigStatus = RAT_UNCHECKED;
	    (*bodyInfoPtrPtr)->secPtr = bodyInfoPtr;
	}
    } else if ((*bodyInfoPtrPtr)->bodyPtr->type == TYPETEXT
	    || ((*bodyInfoPtrPtr)->bodyPtr->type == TYPEAPPLICATION
	    && !strcasecmp("pgp", (*bodyInfoPtrPtr)->bodyPtr->subtype))) {
	text = (*procInfo[(*bodyInfoPtrPtr)->type].fetchBodyProc)
		(*bodyInfoPtrPtr, &length);
	if (text && (((start = RatPGPStrFind(text, length,"BEGIN PGP SIGNED",1))
		&& (middle = RatPGPStrFind(start, length - (start-text),
			"BEGIN PGP SIGNATURE",1))
		&& (end = RatPGPStrFind(middle, length - (middle-text),
			"END PGP",1)))
		|| ((start = RatPGPStrFind(text, length, "BEGIN PGP MESSAGE",1))
		&& (end = RatPGPStrFind(start, length - (start-text),
			"END PGP",1))))) {
	    RatPGPHandleOld(interp, *bodyInfoPtrPtr, text, start, end+1);
	}
    }
}

/*
 *----------------------------------------------------------------------
 *
 * RatPGPCmd --
 *
 *      Handle ratPGP command.
 *
 * Results:
 *	A standard tcl result.
 *
 * Side effects:
 *	Depends on the arguments
 *
 *----------------------------------------------------------------------
 */

int
RatPGPCmd(ClientData clientData, Tcl_Interp *interp, int argc, char **argv)
{
    int length;
    char c;

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		" option ?arg?\"", (char *) NULL);
	return TCL_ERROR;
    }
    c = argv[1][0];
    length = strlen(argv[1]);
    if ((c == 'l') && (strncmp(argv[1], "listkeys", length) == 0)
	    && (length > 1)) {
	if (argc != 3 && argc != 2) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " listkeys [keyring]\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (argc == 3) {
	    return RatPGPListKeys(interp, argv[2]);
	} else {
	    return RatPGPListKeys(interp, NULL);
	}

    } else if ((c == 'e') && (strncmp(argv[1], "extract", length) == 0)
	    && (length > 1)) {
	if (argc != 3 && argc != 4) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " extract id [keyring]\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (argc == 4) {
	    return RatPGPExtractKey(interp, argv[2], argv[3]);
	} else {
	    return RatPGPExtractKey(interp, argv[2], NULL);
	}

    } else if ((c == 'a') && (strncmp(argv[1], "add", length) == 0)
	    && (length > 1)) {
	if (argc != 3 && argc != 4) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " add keys [keyring]\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (argc == 4) {
	    return RatPGPAddKeys(interp, argv[2], argv[3]);
	} else {
	    return RatPGPAddKeys(interp, argv[2], NULL);
	}

    } else {
	Tcl_AppendResult(interp, "bad option \"", argv[1],
		"\": must be listkeys or extract", (char *) NULL);
	return TCL_ERROR;
    }
}


/*
 *----------------------------------------------------------------------
 *
 * RatPGPStrFind --
 *
 *      Find a PGP string in a message
 *
 * Results:
 *	A pointer to the start of the string.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

char*
RatPGPStrFind(char *haystack, long straws, char *needle, int linestart)
{
    long i, j, end;
    int needleSize = strlen(needle);

    end = straws-strlen(needle);

    for (i=0; i<=end; i+= 5) {
	if ('-' == haystack[i]) {
	    for (j=i; j>0 && j>i-5 && '-' == haystack[j]; j--);
	    if ((j >= end-5) || (linestart && j>0 && '\n' != haystack[j])) {
		continue;
	    }
	    if (j > 0) {
		j++;
	    }
	    if (!strncmp("-----", haystack+i, 5-(i-j))
		&& !strncmp(needle, haystack+j+5, needleSize)) {
		return haystack+j;
	    }
	}
    }
    return NULL;
}
