#include <rumba/baseManifold.h>
#include <rumba/manifold.h>
#include <rumba/manifoldFile.h>
#include <rumba/arghandler.h>
#include <rumba/parse.h>
// #include <rumba/util.hpp>
#include <iostream>
#include <cstdlib>
#include <vector>

#include "../matrixutils/normalize.h"

using std::cerr;
using std::endl;
using RUMBA::stream_cast;
using RUMBA::Argument;




void help_message()
{	
	std::cerr << "Usage: normalize-transform\n"
	   	"[-i|--infile] infile [-o|--outfile] outfile\n"
		"[-r|--range] (global|volume|pixel)\n (default: global)\n"
		"[-t|--type] (z|r|m)\n"
		"[[--lower|-l] lower] (default: 0. Only applicable for range scaling)\n"
		"[[--upper|-u] upper]  (default: 100. Only applicable for range scaling)\n" << std::endl;
}

int main(int argc, char** argv)
{
	Argument myArgs [5];
	myArgs[0]=Argument ( "type", RUMBA::ALPHA, 't', "r" );
	myArgs[1]=Argument ( "range", RUMBA::ALPHA, 'r', "global" );
	myArgs[2]=Argument ( "lower", RUMBA::NUMERIC, 'l', 0 );
	myArgs[3]=Argument ("upper", RUMBA::NUMERIC, 'u', 0 );
	myArgs[4]=Argument();




	std::string infile, outfile;
	std::string type,range;
	double upper,lower;
	RUMBA::intPoint origin,dims;
	bool verbose;
	bool debug = true;

	int normalize_type = 0;
	RUMBA::ArgHandler::setRequiredDefaultArg("infile");
	RUMBA::ArgHandler::setRequiredDefaultArg("outfile");

	try 
	{
		RUMBA::ArgHandler argh(argc,argv,myArgs);
		if (argh.arg("help"))
		{
			help_message();
			exit(0);
		}

		verbose = argh.arg("verbose");
		argh.arg("infile", infile );
		argh.arg("outfile", outfile );
		argh.arg("type",type);
		argh.arg("range",range);
		if ( argh.arg("upper"))
			argh.arg("upper",upper);
		if ( argh.arg("lower"))
			argh.arg("lower",lower);

		std::vector<string> v = argh.loose();


		if ( type == "z" )	
			normalize_type = RUMBA::Z_NORMALIZE;
		else if ( type == "r" )
			normalize_type = RUMBA::RANGE_NORMALIZE;
		else if ( type == "m" )
			normalize_type = RUMBA::MEAN_NORMALIZE;
		else 
			throw RUMBA::Exception(
					std::string("bad normalize method ")+type); 

		if ( range != "pixel" && range != "volume" && range != "global" )
			throw RUMBA::Exception (std::string("Invalid range: "+range));



		RUMBA::Manifold<double> M(infile.c_str());

		if ( range == "pixel" )
			RUMBA::pixel_normalize ( &M, normalize_type );
		else if ( range == "volume" )
			RUMBA::volume_normalize ( &M, normalize_type );
		else if ( range == "global" ) 
			RUMBA::global_normalize ( &M, normalize_type );

		M.save ( outfile.c_str() );
		if (verbose)
			M.print(std::cerr);
	
	}
    catch (RUMBA::MissingArgumentException& s)
    {
		cerr << "Error, missing required argument: " << s.error() << endl;
		help_message();
	}
    catch (RUMBA::DuplicateArgumentException& s)
    {
		cerr << "Duplicate argument: " << s.error() << endl;
		help_message();
	}
	catch (RUMBA::ArgHandlerException& s)
	{
		cerr << "Error: " << s.error() << endl;
		help_message();
	}                   
	catch ( RUMBA::Exception & e)
	{
		cerr << "Fatal exception " << e.error() << endl;	
	}

	return 0;
}
