#include <rumba/arghandler.h>
#include <rumba/baseManifold.h>
#include <rumba/manifoldFile.h>

#define EPSILON 1e-8

inline double square(const double& x)
{
	return x*x;
}

double durbinWatson(int skip, int tp, RUMBA::BaseManifold::iterator it, double undefined= 0)
{
	double numerator=0;
	double denominator=0;

	double d;
	double tmp;

	int t;
	for(int t=0;t<tp-1;++t)
	{ 
		denominator += square(static_cast<double>(*it));
		d = - static_cast<double>(*it);   // d = *it + *(it+skip) 
		it += skip; 
		d += static_cast<double>(*it); 
		numerator += d*d; 
	}
	denominator += square(static_cast<double>(*it));
	if (denominator < EPSILON)
		return undefined ;
	else
		return  numerator/denominator;
}

void usage()
{
	std::cerr << "dwmap (--infile|-i) infile (--outfile|o) outfile [--undefined|-u  N]"
		<< std::endl;

}

RUMBA::Argument myArgs [] = 
{
	RUMBA::Argument("undefined",RUMBA::NUMERIC,'u'),
	RUMBA::Argument()
};

int main(int argc, char** argv)
{
	std::string infile,outfile;
	int px,tp;
	RUMBA::ManifoldFile* fin=0;
	RUMBA::BaseManifold* fout=0;

	try 
	{
		RUMBA::ArgHandler argh(argc,argv,myArgs);
		if (argh.arg("help"))
		{
			usage(); 
			return 0;
		}
		argh.arg("infile",infile);
		argh.arg("outfile",outfile);

		fin = RUMBA::ManifoldFile::construct(infile.c_str());
		if (!fin) 
			throw RUMBA::Exception("Fatal: couldn't open file for input");
		RUMBA::intPoint ext = fin->extent(); 
		ext.t()=1;
		fout = RUMBA::ManifoldFile::construct(outfile.c_str(),"float64", ext );
		if (!fout) 
			throw RUMBA::Exception("Fatal: couldn't open file for input");

		px = fin->pixels();
		tp = fin->timepoints();
		fin->setCacheStrategy(RUMBA::CACHE_TIME);
		
		int i = 0;	
		for (RUMBA::BaseManifold::iterator it = fin->begin();
			it!=fin->begin()+px; ++it, ++i)
		{
			fout->setElementDouble(i,durbinWatson( px, tp, it ));
		}

		delete fin;
		delete fout;
	}
	catch ( RUMBA::InvalidArgumentException& s)
	{
		std::cerr << "Invalid argument: " << s.error() << std::endl;
	}
    catch (RUMBA::DuplicateArgumentException& s)
    {
		std::cerr << "Duplicate argument: " << s.error() << std::endl;
	}
	catch (RUMBA::ArgHandlerException& s)
	{
		std::cerr << "Error: " << s.error() << std::endl;
	}
	catch (RUMBA::Exception& s)
	{
		std::cerr << "Exception:" << s.error() << std::endl;
	}

}
