#include <rumba/manifoldmatrix.h>
#include <rumba/arghandler.h>
#include <rumba/matrixio.h>

using RUMBA::invert;

RUMBA::ManifoldMatrix dct_matrix(int size)
{
	static const double pi = 4*atan(1);

	RUMBA::ManifoldMatrix M = RUMBA::makeMatrix(size,size);

	for (int i = 0; i < size; ++i )
	{
		for (int j = 0; j < size; ++j )
			M.element(i,j) = cos((pi*i*j)/size);
	}
	return M;
}

RUMBA::Argument myArgs[] = 
{
	RUMBA::Argument("size",RUMBA::NUMERIC, 's'),
	RUMBA::Argument("components",RUMBA::NUMERIC, 'c'),
	RUMBA::Argument()
};

void usage()
{
	std::cerr << 
		"dct-matrix: produce a matrix M such that Mv is the DCT of v\n"
		"or a discrete cosine transform filter\n"
		"where v is a column vector\n\n"
		"usage: dct-matrix -s|--size <size> [-c|--components <components>]\n"
		"size is the size of the vector v. Components is the number of low\n"
	  	"order components to be zeroed out\n";
}

int main(int argc, char** argv)
{
	int size;
	int components = -1; // number of zeroed-out components
	std::string outfile;

	try
	{
		RUMBA::ArgHandler argh(argc,argv,myArgs);

		if (argh.arg("help"))
		{
			usage();
			exit(0);
		}


		if (!argh.arg("size"))
			throw RUMBA::Exception("Size required");

		argh.arg("size",size);
		if (size < 1)
			throw RUMBA::Exception("Size must be greater than or equal to 1");



		if (argh.arg("components"))
		{
			argh.arg("components",components);
			if (components < 0 || components >= size )
			{
				throw RUMBA::Exception("Size must exceed components");
			}
		}

		RUMBA::ManifoldMatrix M (dct_matrix(size));

		if (components > 0)
		{
			// submatrix
			M = M.subMatrix(0,components,0,M.cols()).transpose();
			M = RUMBA::identityMatrix(M.rows())-
				M*invert(M.transpose()*M)*M.transpose();
		
		}


		if (!argh.arg("outfile"))
		{
			RUMBA::writeManifoldMatrix(M,std::cout);
		}
		else
		{
			argh.arg("outfile",outfile);
			RUMBA::manifoldMatrixWriteHack(M,outfile);
		}

	}
	catch (RUMBA::InvalidArgumentException& e)
	{
		std::cerr << "Invalid argument: " << e.error() << std::endl;
		usage();
		exit(1);
	}

	catch (RUMBA::MissingArgumentException& e)
	{
		std::cerr << "Missing argument: " << e.error() << std::endl;
		usage();
		exit(1);
	}

	catch (RUMBA::Exception& e)
	{
		std::cerr << "Error: " << e.error() << std::endl;
		usage();
		exit(1);
	}

}
