#include <iostream>
#include <numeric>
#include <string>
#include <rumba/fft.h>
#include <rumba/arghandler.h>
#include <rumba/manifoldmatrix.h>
#include <rumba/matrixio.h>

// like octaves Yulewalker function: input is a bunch of autocorrelation 
// coefficients. Gives coefficients for AR(size-1)
// takes a coefficient vector v, and returns a matrix row vector
namespace 
{
	RUMBA::ManifoldMatrix yulewalker 
		(const std::vector<complex<double> >& v, int size)
	{
		RUMBA::ManifoldMatrix M = RUMBA::makeMatrix(size-1, size-1);
		RUMBA::ManifoldMatrix N = RUMBA::makeMatrix(size-1,1);

		for (int i = 0; i < size-1; ++i )
			for (int j = 0; j < size-1; ++j )
				M.element(i,j) = v[abs(i-j)].real();

		for (int i = 1; i < size; ++i )
			N.element(i-1,0) = v[i].real();

		return (invert(M) * N).transpose();
	}


	// ar_matrix: make filter matrix that acts on a time series with size
	// timepoints
	RUMBA::ManifoldMatrix ar_matrix
		(int size, const RUMBA::ManifoldMatrix& result)
	{
		RUMBA::ManifoldMatrix A = RUMBA::makeMatrix ( size, size  );
		std::fill(A.begin(),A.end(),0);
		for (int i = 0; i < result.cols(); ++i )
		{
			for (int j = 0; j < A.cols() - i-1; ++j )
				A.element( j+1+i,j ) = result.element(0,i);
		}
		return A;
	}

}




RUMBA::Argument myArgs [] = 
{
	RUMBA::Argument("arorder", RUMBA::NUMERIC, 'p' ),
	RUMBA::Argument("size", RUMBA::NUMERIC, 's' ),
	RUMBA::Argument()
};


int main(int argc,char** argv)
{
	int tmp;
	int size;
	std::string infile,outfile;
	std::vector<complex<double> > v; double x;

	try 
	{
		RUMBA::ArgHandler argh(argc,argv,myArgs);

		int p = 16;

		if (!argh.arg("infile") || !argh.arg("outfile") )
		{
			std::cerr << "Must supply input and output files" << std::endl;
			exit(1);
		}

		argh.arg("infile", infile);
		argh.arg("outfile", outfile);
		if (argh.arg("arorder")) 
			argh.arg("arorder",p); 
		else 
			throw RUMBA::Exception("Must supply an ar order");

		if (argh.arg("size"))
			argh.arg("size",size);
		else
			throw RUMBA::Exception("Must supply a size");



		std::ifstream in(infile.c_str());
		if (!in)
		{
			std::cerr << "Couldn't open input" << std::endl;
			exit(1);
		}

		while (in>>x)
			v.push_back(x);

	// This needs to be a power of 2 !!! 
	// This is OK -- we should have got the data via a fft in the first place.
		RUMBA::generic_fft( v,v.size() );
		std::cout << "---FFT(spectral density)-----" << std::endl;	

		std::copy (v.begin(),v.end(),ostream_iterator<complex<double> >
			(cout, " ... \n") );
		std::cout << "------------------" << std::endl;	
		
		assert( p > 0 && v.size() > (unsigned int)p );

		RUMBA::ManifoldMatrix result = yulewalker(v,p+1);
		std::cout << "---yulewalker result-----" << std::endl;	

		std::copy (result.begin(),result.end(),ostream_iterator<double >
			(cout, " ... \n") );
		std::cout << "------------------" << std::endl;	
	

		// autoregression band matrix 
		RUMBA::ManifoldMatrix A;

		// I-A
		RUMBA::ManifoldMatrix result_matrix;

		result_matrix = RUMBA::identityMatrix(size)-ar_matrix(size,result);

		RUMBA::manifoldMatrixWriteHack(result_matrix,outfile.c_str());
	
	}
	catch (RUMBA::InvalidArgumentException& e)
	{
		std::cerr << "Invalid argument: " << e.error() << std::endl;
	}	
	catch (RUMBA::Exception& e)
	{
		std::cerr << e.error() << std::endl;
	}
	return 0;
}
