/*
 *
 *   magick_private.c - Ruby interface for ImageMagick
 *
 *   Copyright (C) 2001 Ryuichi Tamura(tam@kais.kyoto-u.ac.jp)
 *
 *   $Date: 2001/07/20 10:07:40 $
 *   $Revision: 1.11 $
 *
 */

#include "magick.h"

VALUE mPrivate;
VALUE cMontageInfo;
VALUE cDrawInfo;
VALUE cQuantizeInfo;
VALUE cGeometry;
VALUE S_FontMetrics;
EXTERN VALUE eImageError;

#define Get_DrawInfo(obj, ptr) {                                \
    Data_Get_Struct(obj, DrawInfo, ptr);                        \
    if (!ptr)                                                   \
        rb_raise(rb_eRuntimeError, "released object");          \
}

/*
 *  Wraps MontageInfo Structure
 */

static VALUE
minfo_s_new(VALUE klass, VALUE image)
{
    MontageInfo *minfo;
    MgkImage *mgk;
    ImageInfo *info;
    ImageInfo *dup_info = (ImageInfo *)NULL;

    IsKindOf(image, cImage);
    Get_MgkImage(image, mgk);
    GetInfoStruct(mgk, info);
    dup_info = CloneImageInfo(info);
    minfo = CloneMontageInfo(dup_info, (MontageInfo *)NULL);

    return Data_Wrap_Struct(klass, 0, DestroyMontageInfo, minfo);
}

static VALUE
minfo_set_filename(VALUE obj, VALUE str)
{
    MontageInfo *minfo;

    Get_MontageInfo(obj, minfo);
    Check_Type(str, T_STRING);
    strcpy(minfo->filename, RSTRING(str)->ptr);

    return obj;
}

static VALUE
minfo_set_geometry(VALUE obj, VALUE str)
{
    MontageInfo *minfo;

    Get_MontageInfo(obj, minfo);
    Check_Type(str, T_STRING);
    CloneString(&minfo->geometry, RSTRING(str)->ptr);

    return obj;
}

static VALUE
minfo_set_tile(VALUE obj, VALUE str)
{
    MontageInfo *minfo;

    Get_MontageInfo(obj, minfo);
    Check_Type(str, T_STRING);
    CloneString(&minfo->tile, RSTRING(str)->ptr);

    return obj;
}

static VALUE
minfo_set_title(VALUE obj, VALUE str)
{
    MontageInfo *minfo;

    Get_MontageInfo(obj, minfo);
    Check_Type(str, T_STRING);
    CloneString(&minfo->title, RSTRING(str)->ptr);

    return obj;
}

static VALUE
minfo_set_frame(VALUE obj, VALUE str)
{
    MontageInfo *minfo;

    Get_MontageInfo(obj, minfo);
    Check_Type(str, T_STRING);
    CloneString(&minfo->frame, RSTRING(str)->ptr);

    return obj;
}

static VALUE
minfo_set_texture(VALUE obj, VALUE str)
{
    MontageInfo *minfo;

    Get_MontageInfo(obj, minfo);
    Check_Type(str, T_STRING);
    CloneString(&minfo->texture, RSTRING(str)->ptr);

    return obj;
}

static VALUE
minfo_set_font(VALUE obj, VALUE str)
{
    MontageInfo *minfo;

    Get_MontageInfo(obj, minfo);
    Check_Type(str, T_STRING);
    CloneString(&minfo->font, RSTRING(str)->ptr);

    return obj;
}

static VALUE
minfo_set_pointsize(VALUE obj, VALUE dbl)
{
    MontageInfo *minfo;

    Get_MontageInfo(obj, minfo);
    minfo->pointsize = NUM2DBL(dbl);

    return obj;
}

static VALUE
minfo_set_gravity(VALUE obj, VALUE gravity_type)
{
    MontageInfo *minfo;

    Get_MontageInfo(obj, minfo);
    minfo->gravity = FIX2INT(gravity_type);

    return obj;
}


static VALUE
minfo_set_border_width(VALUE obj, VALUE border_width)
{
    MontageInfo *minfo;

    Get_MontageInfo(obj, minfo);
    minfo->border_width = FIX2UINT(border_width);

    return obj;
}

static VALUE
minfo_set_shadow(VALUE obj, VALUE shadow)
{
    MontageInfo *minfo;

    Get_MontageInfo(obj, minfo);
    minfo->shadow = RTEST(shadow);

    return obj;
}

static VALUE
minfo_set_composite(VALUE obj, VALUE composite_type)
{
    MontageInfo *minfo;

    Get_MontageInfo(obj, minfo);
    minfo->compose = FIX2INT(composite_type);

    return obj;
}

static VALUE
minfo_set_fill(VALUE obj, VALUE str)
{
    MontageInfo *minfo;
    PixelPacket target;
    char *color = (char *)NULL;

    Check_Type(str, T_STRING);
    Get_MontageInfo(obj, minfo);

    CloneString(&color, RSTRING(str)->ptr);
    QueryColorDatabase(color, &target);

    minfo->fill = target;

    return obj;
}


static VALUE
minfo_set_stroke(VALUE obj, VALUE str)
{
    MontageInfo *minfo;
    PixelPacket target;
    char *color = (char *)NULL;

    Check_Type(str, T_STRING);
    Get_MontageInfo(obj, minfo);

    CloneString(&color, RSTRING(str)->ptr);
    QueryColorDatabase(color, &target);

    minfo->stroke = target;

    return obj;
}


static VALUE
minfo_set_background_color(VALUE obj, VALUE str)
{
    MontageInfo *minfo;
    PixelPacket target;
    char *color = (char *)NULL;

    Check_Type(str, T_STRING);
    Get_MontageInfo(obj, minfo);

    CloneString(&color, RSTRING(str)->ptr);
    QueryColorDatabase(color, &target);

    minfo->background_color = target;

    return obj;
}


static VALUE
minfo_set_border_color(VALUE obj, VALUE str)
{
    MontageInfo *minfo;
    PixelPacket target;
    char *color = (char *)NULL;

    Check_Type(str, T_STRING);
    Get_MontageInfo(obj, minfo);

    CloneString(&color, RSTRING(str)->ptr);
    QueryColorDatabase(color, &target);

    minfo->border_color = target;

    return obj;
}


static VALUE
minfo_set_matte_color(VALUE obj, VALUE str)
{
    MontageInfo *minfo;
    PixelPacket target;
    char *color = (char *)NULL;

    Check_Type(str, T_STRING);
    Get_MontageInfo(obj, minfo);

    CloneString(&color, RSTRING(str)->ptr);
    QueryColorDatabase(color, &target);

    minfo->matte_color = target;

    return obj;
}



/* Wraps DrawInfo structure */


static VALUE
drawinfo_s_new(VALUE klass, VALUE image)
{
    DrawInfo *draw_info;
    MgkImage *mgk;
    ImageInfo *info;

    IsKindOf(image, cImage);
    Get_MgkImage(image, mgk);
    GetInfoStruct(mgk, info);
    draw_info = CloneDrawInfo(info, (DrawInfo *)NULL);

    return Data_Wrap_Struct(klass, 0, DestroyDrawInfo, draw_info);
}

static VALUE
drawinfo_set_primitive(VALUE obj, VALUE str)
{
    DrawInfo *draw_info;

    Check_Type(str, T_STRING);
    Get_DrawInfo(obj, draw_info);

    CloneString(&draw_info->primitive, RSTRING(str)->ptr);

    return obj;
}

static VALUE
drawinfo_set_text(VALUE obj, VALUE str)
{
    DrawInfo *draw_info;

    Check_Type(str, T_STRING);
    Get_DrawInfo(obj, draw_info);

    CloneString(&draw_info->text, RSTRING(str)->ptr);

    return obj;
}

static VALUE
drawinfo_set_geometry(VALUE obj, VALUE str)
{
    DrawInfo *draw_info;

    Check_Type(str, T_STRING);
    Get_DrawInfo(obj, draw_info);

    CloneString(&draw_info->geometry, RSTRING(str)->ptr);

    return obj;
}

#define DegreesToRadians(x) ((x)*3.14159265358979323846/180.0)

static void
update_affine(DrawInfo *draw_info, AffineMatrix affine)
{
    AffineMatrix current;

    current = draw_info->affine;
    draw_info->affine.sx=current.sx*affine.sx+current.ry*affine.rx;
    draw_info->affine.rx=current.rx*affine.sx+current.sy*affine.rx;
    draw_info->affine.ry=current.sx*affine.ry+current.ry*affine.sy;
    draw_info->affine.sy=current.rx*affine.ry+current.sy*affine.sy;
    draw_info->affine.tx=current.sx*affine.tx+current.ry*affine.ty+current.tx;
    draw_info->affine.ty=current.rx*affine.tx+current.sy*affine.ty+current.ty;
}

static VALUE
drawinfo_set_rotate(VALUE obj, VALUE num)
{
    DrawInfo *draw_info;
    AffineMatrix affine;
    double angle;

    IdentityAffine(&affine);
    angle = NUM2DBL(num);
    if (angle == 0.0)
        return obj;
    Get_DrawInfo(obj, draw_info);
    affine.sx = cos(DegreesToRadians(fmod(angle,360.0)));
    affine.rx = sin(DegreesToRadians(fmod(angle,360.0)));
    affine.ry = (-sin(DegreesToRadians(fmod(angle,360.0))));
    affine.sy = cos(DegreesToRadians(fmod(angle,360.0)));
    update_affine(draw_info, affine);

    return obj;
}

static VALUE
drawinfo_set_scale(VALUE obj, VALUE num)
{
    DrawInfo *draw_info;
    AffineMatrix affine;

    IdentityAffine(&affine);
    switch(TYPE(num)) {
      case T_ARRAY:
        if (RARRAY(num)->len != 2)
            rb_raise(rb_eArgError, "need exactly 2 elements");
        affine.sx = NUM2DBL(*(RARRAY(num)->ptr));
        affine.sy = NUM2DBL(*(RARRAY(num)->ptr+1));
        break;
      default:
        affine.sy = affine.sx = NUM2DBL(num);
    }
    Get_DrawInfo(obj, draw_info);
    update_affine(draw_info, affine);
    return obj;
}

static VALUE
drawinfo_set_translate(VALUE obj, VALUE num)
{
    DrawInfo *draw_info;
    AffineMatrix affine;

    IdentityAffine(&affine);
    switch(TYPE(num)) {
      case T_ARRAY:
        if (RARRAY(num)->len != 2)
            rb_raise(rb_eArgError, "need exactly 2 elements");
        affine.tx = NUM2DBL(*(RARRAY(num)->ptr));
        affine.ty = NUM2DBL(*(RARRAY(num)->ptr+1));
        break;
      default:
        affine.ty = affine.tx = NUM2DBL(num);
    }
    Get_DrawInfo(obj, draw_info);
    update_affine(draw_info, affine);
    return obj;
}

static VALUE
drawinfo_set_gravity(VALUE obj, VALUE gravity_type)
{
    DrawInfo *draw_info;
    Get_DrawInfo(obj, draw_info);
    draw_info->gravity = FIX2INT(gravity_type);
    return obj;
}

static VALUE
drawinfo_set_opacity(VALUE obj, VALUE quantum)
{
    DrawInfo *draw_info;
    Get_DrawInfo(obj, draw_info);
    draw_info->opacity = NUM2INT(quantum);
    return obj;
}

static VALUE
drawinfo_set_tile(VALUE obj, VALUE image)
{
    DrawInfo *draw_info;
    MgkImage *mgk;
    ExceptionInfo e;

    IsKindOf(image, cImage);
    Get_MgkImage(image, mgk);
    Get_DrawInfo(obj, draw_info);
    GetExceptionInfo(&e);

    draw_info->tile = CloneImage(mgk->ptr, 0, 0, 1, &e);
    check_exception(draw_info->tile, &e);

    return obj;
}

static VALUE
drawinfo_set_fill(VALUE obj, VALUE str)
{
    DrawInfo *draw_info;

    Check_Type(str, T_STRING);
    Get_DrawInfo(obj, draw_info);
    QueryColorDatabase(RSTRING(str)->ptr, &draw_info->fill);

    return obj;
}

static VALUE
drawinfo_set_stroke(VALUE obj, VALUE str)
{
    DrawInfo *draw_info;
    int status;

    Check_Type(str, T_STRING);
    Get_DrawInfo(obj, draw_info);

    status = QueryColorDatabase(RSTRING(str)->ptr, &draw_info->stroke);
    if (!status)
        rb_warn("Invalid color specification: %s", RSTRING(str)->ptr);

    return obj;
}

static VALUE
drawinfo_set_stroke_antialias(VALUE obj, VALUE boolean)
{
    DrawInfo *draw_info;

    Get_DrawInfo(obj, draw_info);
    draw_info->stroke_antialias = RTEST(boolean);

    return obj;
}

static VALUE
drawinfo_set_stroke_width(VALUE obj, VALUE width)
{
    DrawInfo *draw_info;

    Get_DrawInfo(obj, draw_info);
    draw_info->stroke_width = NUM2DBL(width);

    return obj;
}

static VALUE
drawinfo_set_fill_rule(VALUE obj, VALUE fill_type)
{
    DrawInfo *draw_info;

    Get_DrawInfo(obj, draw_info);
    draw_info->fill_rule = FIX2INT(fill_type);

    return obj;
}

static VALUE
drawinfo_set_line_cap(VALUE obj, VALUE linecap)
{
    DrawInfo *draw_info;

    Get_DrawInfo(obj, draw_info);
    draw_info->linecap = FIX2INT(linecap);

    return obj;
}

static VALUE
drawinfo_set_linejoin(VALUE obj, VALUE linejoin)
{
    DrawInfo *draw_info;

    Get_DrawInfo(obj, draw_info);
    draw_info->linejoin = FIX2INT(linejoin);

    return obj;
}

static VALUE
drawinfo_set_miterlimit(VALUE obj, VALUE limit)
{
    DrawInfo *draw_info;

    Get_DrawInfo(obj, draw_info);
    draw_info->miterlimit = NUM2UINT(limit);

    return obj;
}


static VALUE
drawinfo_set_dash_offset(VALUE obj, VALUE offset)
{
    DrawInfo *draw_info;

    Get_DrawInfo(obj, draw_info);
    draw_info->dash_offset = NUM2UINT(offset);

    return obj;
}

static VALUE
drawinfo_set_decorate(VALUE obj, VALUE decoration_type)
{
    DrawInfo *draw_info;

    Get_DrawInfo(obj, draw_info);
    draw_info->dash_offset = FIX2INT(decoration_type);

    return obj;
}

static VALUE
drawinfo_set_font(VALUE obj, VALUE str)
{
    DrawInfo *draw_info;

    Check_Type(str, T_STRING);
    Get_DrawInfo(obj, draw_info);

    CloneString(&draw_info->font, RSTRING(str)->ptr);

    return obj;
}

static VALUE
drawinfo_set_density(VALUE obj, VALUE str)
{
    DrawInfo *draw_info;
    Check_Type(str, T_STRING);
    Get_DrawInfo(obj, draw_info);

    draw_info->density = AllocateString(RSTRING(str)->ptr);

    return obj;
}

static VALUE
drawinfo_set_text_antialias(VALUE obj, VALUE boolean)
{
    DrawInfo *draw_info;

    Get_DrawInfo(obj, draw_info);
    draw_info->text_antialias = RTEST(boolean);

    return obj;
}

static VALUE
drawinfo_set_pointsize(VALUE obj, VALUE pointsize)
{
    DrawInfo *draw_info;

    Get_DrawInfo(obj, draw_info);
    draw_info->pointsize = NUM2DBL(pointsize);

    return obj;
}


static VALUE
drawinfo_set_box(VALUE obj, VALUE str)
{
    DrawInfo *draw_info;
    int status;

    Check_Type(str, T_STRING);
    Get_DrawInfo(obj, draw_info);

    status = QueryColorDatabase(RSTRING(str)->ptr, &draw_info->box);
    if (!status)
        rb_warn("Invalid color specification: %s", RSTRING(str)->ptr);

    return obj;
}

static VALUE
drawinfo_set_border_color(VALUE obj, VALUE str)
{
    DrawInfo *draw_info;
    int status;

    Check_Type(str, T_STRING);
    Get_DrawInfo(obj, draw_info);

    status = QueryColorDatabase(RSTRING(str)->ptr, &draw_info->border_color);
    if (!status)
        rb_warn("Invalid color specification: %s", RSTRING(str)->ptr);

    return obj;
}


static VALUE
drawinfo_set_compose(VALUE obj, VALUE composition_type)
{
    DrawInfo *draw_info;

    Get_DrawInfo(obj, draw_info);
    draw_info->compose = FIX2INT(composition_type);

    return obj;
}

static VALUE
drawinfo_set_server_name(VALUE obj, VALUE str)
{
    DrawInfo *draw_info;

    Check_Type(str, T_STRING);
    Get_DrawInfo(obj, draw_info);
    CloneString(&draw_info->server_name, RSTRING(str)->ptr);

    return obj;
}

static VALUE
drawinfo_set_dash_pattern(VALUE obj, VALUE pat_ary)
{
    DrawInfo *draw_info;
    int *elts;
    int i;

    Check_Type(pat_ary, T_ARRAY);
    elts = ALLOC_N(unsigned int, RARRAY(pat_ary)->len);

    Get_DrawInfo(obj, draw_info);
    for (i = 0; i < RARRAY(pat_ary)->len; i++) {
        elts[i] = FIX2INT(*(RARRAY(pat_ary)->ptr+i));
        if (elts[i] == 0)
            rb_raise(rb_eArgError, "strict positive value required");
    }
    draw_info->dash_pattern = elts;

    return obj;
}

VALUE
mgk_do_get_font_metrics(VALUE obj, VALUE dinfo)
{
    MgkImage *mgk;
    DrawInfo *draw_info;
    FontMetric font_metrics;
    int status;

    Get_MgkImage(obj, mgk);
    Get_DrawInfo(dinfo, draw_info);
    status = GetFontMetrics(mgk->ptr, draw_info, &font_metrics);
    if (!status)
        rb_warn("Failed to get font metrics");

    return rb_struct_new(S_FontMetrics,
                         rb_float_new((double)font_metrics.pixels_per_em.x),
                         rb_float_new((double)font_metrics.pixels_per_em.y),
                         INT2FIX(font_metrics.ascent),
                         INT2FIX(font_metrics.descent),
                         INT2FIX(font_metrics.width),
                         INT2FIX(font_metrics.height),
                         INT2FIX(font_metrics.max_advance) );
}


/*
 *
 *  Geometry class
 *
 */

struct _Geom {
    int flags;
    RectangleInfo rect;
};

typedef struct _Geom Geometry;

#define Get_Geometry(obj, ptr) {                        \
    Data_Get_Struct(obj, Geometry, ptr);                \
    if (!ptr)                                           \
        rb_raise(rb_eRuntimeError, "released object");  \
}


static VALUE
geom_initialize(VALUE obj, VALUE which, VALUE geom, VALUE w, VALUE h)
{
    Geometry *g;
    int flags;

    Get_Geometry(obj, g);

    if (RTEST(which)){
        flags = ParseGeometry(RSTRING(geom)->ptr, &g->rect.x, &g->rect.y,
                              &g->rect.width, &g->rect.height);
    }
    else {
        g->rect.x = g->rect.y = 0;
        g->rect.width = NUM2INT(w); g->rect.height = NUM2INT(h);
        flags = ParseImageGeometry(RSTRING(geom)->ptr,
                                   &g->rect.x, &g->rect.y,
                                   &g->rect.width, &g->rect.height);
    }
    g->flags = flags;
    rb_iv_set(obj, "@flags", INT2FIX(flags));
    rb_iv_set(obj, "@x", INT2FIX(g->rect.x));
    rb_iv_set(obj, "@y", INT2FIX(g->rect.y));
    rb_iv_set(obj, "@width",INT2FIX( g->rect.width));
    rb_iv_set(obj, "@height", INT2FIX(g->rect.height));

    return obj;
}

static VALUE
geom_s_new(VALUE klass, VALUE which, VALUE str, VALUE w, VALUE h)
{
    Geometry *g;
    VALUE obj, args[4];

    obj = Data_Make_Struct(klass, Geometry, 0, xfree, g);

    rb_attr(klass, rb_intern("flags"), 1, 0, Qtrue);
    rb_attr(klass, rb_intern("x"), 1, 1, Qtrue);
    rb_attr(klass, rb_intern("y"), 1, 1, Qtrue);
    rb_attr(klass, rb_intern("width"), 1, 1, Qtrue);
    rb_attr(klass, rb_intern("height"), 1, 1, Qtrue);

    args[0] = which;
    args[1] = str;
    args[2] = NIL_P(w) ? (VALUE)0 : w;
    args[3] = NIL_P(h) ? (VALUE)0 : h;
    rb_obj_call_init(obj, 4, args);
    return obj;
}

static VALUE
geom_is_valid(VALUE obj)
{
    Geometry *g;

    Get_Geometry(obj, g);
    return (!g->flags & NoValue) ? Qtrue : Qfalse;
}


static VALUE
geom_has_XValue(VALUE obj)
{
    Geometry *g;

    Get_Geometry(obj, g);
    return (g->flags & XValue) ? Qtrue : Qfalse;
}

static VALUE
geom_has_YValue(VALUE obj)
{
    Geometry *g;

    Get_Geometry(obj, g);
    return (g->flags & YValue) ? Qtrue : Qfalse;
}


static VALUE
geom_has_WidthValue(VALUE obj)
{
    Geometry *g;

    Get_Geometry(obj, g);
    return (g->flags & WidthValue) ? Qtrue : Qfalse;
}

static VALUE
geom_has_HeightValue(VALUE obj)
{
    Geometry *g;

    Get_Geometry(obj, g);
    return (g->flags & HeightValue) ? Qtrue : Qfalse;
}

static VALUE
geom_has_AllValues(VALUE obj)
{
    Geometry *g;

    Get_Geometry(obj, g);
    return (g->flags & AllValues) ? Qtrue : Qfalse;
}

static VALUE
geom_has_PercentValue(VALUE obj)
{
    Geometry *g;

    Get_Geometry(obj, g);
    return (g->flags & PercentValue) ? Qtrue : Qfalse;
}


/* wraps QuantizeInfo structure */
static VALUE
qinfo_s_new(VALUE klass)
{
    QuantizeInfo *qinfo;
    qinfo = CloneQuantizeInfo((QuantizeInfo *)NULL);

    return Data_Wrap_Struct(klass, 0, DestroyQuantizeInfo, qinfo);
}

static VALUE
qinfo_set_colorspace(VALUE obj, VALUE colorspace_type)
{
    QuantizeInfo *qinfo;
    Get_QuantizeInfo(obj, qinfo);
    qinfo->colorspace = FIX2INT(colorspace_type);
    return obj;
}

static VALUE
qinfo_set_dither(VALUE obj, VALUE dither)
{
    QuantizeInfo *qinfo;
    Get_QuantizeInfo(obj, qinfo);
    qinfo->dither = RTEST(dither);
    return obj;
}

static VALUE
qinfo_set_measure_error(VALUE obj, VALUE error)
{
    QuantizeInfo *qinfo;
    Get_QuantizeInfo(obj, qinfo);
    qinfo->measure_error = RTEST(error);
    return obj;
}

static VALUE
qinfo_set_number_colors(VALUE obj, VALUE colors)
{
    QuantizeInfo *qinfo;
    Get_QuantizeInfo(obj, qinfo);
    qinfo->number_colors = FIX2INT(colors);
    return obj;
}

static VALUE
qinfo_set_tree_depth(VALUE obj, VALUE depth)
{
    QuantizeInfo *qinfo;
    Get_QuantizeInfo(obj, qinfo);
    qinfo->tree_depth = FIX2INT(depth);
    return obj;
}


void Init_MagickPrivate(void)
{
    mPrivate = rb_define_module_under(mMagick, "Private");

    cMontageInfo = rb_define_class_under(mPrivate, "MontageInfo", rb_cObject);
    rb_define_singleton_method(cMontageInfo, "new", minfo_s_new, 1);
    rb_define_method(cMontageInfo, "filename=", minfo_set_filename, 1);
    rb_define_method(cMontageInfo, "geometry=", minfo_set_geometry, 1);
    rb_define_method(cMontageInfo, "tile=", minfo_set_tile, 1);
    rb_define_method(cMontageInfo, "title=", minfo_set_title, 1);
    rb_define_method(cMontageInfo, "frame=", minfo_set_frame, 1);
    rb_define_method(cMontageInfo, "texture=", minfo_set_texture, 1);
    rb_define_method(cMontageInfo, "font=", minfo_set_font, 1);
    rb_define_method(cMontageInfo, "pointsize=", minfo_set_pointsize, 1);
    rb_define_method(cMontageInfo, "gravity=", minfo_set_gravity, 1);
    rb_define_method(cMontageInfo, "border_width=", minfo_set_border_width, 1);
    rb_define_method(cMontageInfo, "shadow=", minfo_set_shadow, 1);
    rb_define_method(cMontageInfo, "composite=", minfo_set_composite, 1);
    rb_define_method(cMontageInfo, "fill=", minfo_set_fill, 1);
    rb_define_method(cMontageInfo, "stroke=", minfo_set_stroke, 1);
    rb_define_method(cMontageInfo, "background_color=", minfo_set_background_color, 1);
    rb_define_method(cMontageInfo, "border_color=", minfo_set_border_color, 1);
    rb_define_method(cMontageInfo, "matte_color=", minfo_set_matte_color, 1);

    cDrawInfo = rb_define_class_under(mPrivate, "DrawInfo", rb_cObject);
    rb_define_singleton_method(cDrawInfo, "new", drawinfo_s_new, 1);
    rb_define_method(cDrawInfo, "primitive=", drawinfo_set_primitive, 1);
    rb_define_method(cDrawInfo, "text=", drawinfo_set_text, 1);
    rb_define_method(cDrawInfo, "geometry=", drawinfo_set_geometry, 1);
    rb_define_method(cDrawInfo, "gravity=", drawinfo_set_gravity, 1);
    rb_define_method(cDrawInfo, "opacity=", drawinfo_set_opacity, 1);
    rb_define_method(cDrawInfo, "tile=", drawinfo_set_tile, 1);
    rb_define_method(cDrawInfo, "fill=", drawinfo_set_fill, 1);
    rb_define_method(cDrawInfo, "border_color=", drawinfo_set_border_color, 1);
    rb_define_method(cDrawInfo, "stroke=", drawinfo_set_stroke, 1);
    rb_define_method(cDrawInfo, "stroke_antialias=", drawinfo_set_stroke_antialias, 1);
    rb_define_method(cDrawInfo, "stroke_width=", drawinfo_set_stroke_width, 1);
    rb_define_method(cDrawInfo, "fill_rule=", drawinfo_set_fill_rule, 1);
    rb_define_method(cDrawInfo, "linecap=", drawinfo_set_line_cap, 1);
    rb_define_method(cDrawInfo, "linejoin=", drawinfo_set_linejoin, 1);
    rb_define_method(cDrawInfo, "miterlimit=", drawinfo_set_miterlimit, 1);
    rb_define_method(cDrawInfo, "dash_offset=", drawinfo_set_dash_offset, 1);
    rb_define_method(cDrawInfo, "decorate=", drawinfo_set_decorate, 1);
    rb_define_method(cDrawInfo, "font=", drawinfo_set_font, 1);
    rb_define_method(cDrawInfo, "density=", drawinfo_set_density, 1);
    rb_define_method(cDrawInfo, "text_antialias=", drawinfo_set_text_antialias, 1);
    rb_define_method(cDrawInfo, "pointsize=", drawinfo_set_pointsize, 1);
    rb_define_method(cDrawInfo, "box=", drawinfo_set_box, 1);
    rb_define_method(cDrawInfo, "border_color=", drawinfo_set_border_color, 1);
    rb_define_method(cDrawInfo, "compose=", drawinfo_set_compose, 1);
    rb_define_method(cDrawInfo, "server_name=", drawinfo_set_server_name, 1);
    rb_define_method(cDrawInfo, "dash_pattern=", drawinfo_set_dash_pattern, 1);
    rb_define_method(cDrawInfo, "translate=",drawinfo_set_translate, 1);
    rb_define_method(cDrawInfo, "scale=", drawinfo_set_scale, 1);
    rb_define_method(cDrawInfo, "rotate=", drawinfo_set_rotate, 1);

    cGeometry = rb_define_class_under(mPrivate, "Geometry", rb_cObject);
    rb_define_singleton_method(cGeometry, "new", geom_s_new, 4);
    rb_define_method(cGeometry, "initialize", geom_initialize, 4);
    rb_define_method(cGeometry, "is_valid?", geom_is_valid, 0);
    rb_define_method(cGeometry, "has_x?", geom_has_XValue, 0);
    rb_define_method(cGeometry, "has_y?", geom_has_YValue, 0);
    rb_define_method(cGeometry, "has_width?", geom_has_WidthValue, 0);
    rb_define_method(cGeometry, "has_height?", geom_has_HeightValue, 0);
    rb_define_method(cGeometry, "has_allvalues?", geom_has_AllValues, 0);
    rb_define_method(cGeometry, "has_percent?", geom_has_PercentValue, 0);

    cQuantizeInfo = rb_define_class_under(mPrivate, "QuantizeInfo", rb_cObject);
    rb_define_singleton_method(cQuantizeInfo, "new", qinfo_s_new, 0);
    rb_define_method(cQuantizeInfo, "colorspace=", qinfo_set_colorspace,1);
    rb_define_method(cQuantizeInfo, "dither=", qinfo_set_dither, 1);
    rb_define_method(cQuantizeInfo, "measure_error=", qinfo_set_measure_error, 1);
    rb_define_method(cQuantizeInfo, "colors=", qinfo_set_number_colors, 1);
    rb_define_method(cQuantizeInfo, "tree_depth=", qinfo_set_tree_depth, 1);

    S_FontMetrics = rb_struct_define("FontMetrics","charcter_width", "charcter_height",
                                     "ascent", "descent", "text_width",
                                     "text_height", "max_advance", 0);
}
