#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>
#include <unistd.h>
#include <sys/mman.h>
#include <fcntl.h>
#include "fasldef.h"
#include <rscheme/smemory.h>
#include <rscheme/scheme.h>
#include <rscheme/api.h>
#include <rscheme/bcextend.h>
#include <rscheme/rlseconf.h>
#include <rscheme/heapi.h>

#ifdef HAVE_MACH_H
#include <mach/mach.h>
#include <mach/exception.h>
#include <mach/mach_error.h>
#define USE_MACH_API (1)
#else
#define USE_MACH_API (0)
#endif

/* #define DEBUG_ENABLE */

#ifdef PLATFORM_LINUX
#if PLATFORM_ARCH_PPC
/* presume its MkLinux... */
#define FASL_FIXED_ADDR     (0x30000000)
#define FASL_FIXED_ADDR_ALT (0x38000000)
#else
#if PLATFORM_ARCH_ALPHA
#define FASL_FIXED_ADDR     (0x6100000000UL)
#define FASL_FIXED_ADDR_ALT (0x6200000000UL)
#else
#if PLATFORM_ARCH_S390
#define FASL_FIXED_ADDR     (0x60000000)
#define FASL_FIXED_ADDR_ALT (0x68000000)
#else
/* presume its i386 */
#define FASL_FIXED_ADDR     (0xA0000000)
#define FASL_FIXED_ADDR_ALT (0xA8000000)
#endif
#endif
#endif
#endif

#ifdef PLATFORM_RHAPSODY
#define FASL_FIXED_ADDR     (0xA8000000)
#define FASL_FIXED_ADDR_ALT (0xAC000000)
#endif

void *fasl_loaded_at = NULL;

#define _const const

void fasl_error( _const char *msg, _const char *info )
{
  fprintf( stderr, "FASL error: %s (%s)\n", msg, info );
  exit(1);
}

static size_t max_fasl_size = 30*1024*1024;
static struct FASL_Header *hdr;
static void *rgn_ptr, *rgn_limit;

static void *heap_reserve( int fd, void *at );
static void heap_write( const char *path, int fd );
static int fasl_open_file( const char *path );

void *fasl_alloc( size_t len )
{
  void *p = rgn_ptr;
  len = ((len - 1) | 3) + 1;
  rgn_ptr = ((char *)rgn_ptr) + len;
  if (rgn_ptr > rgn_limit)
    {
      fprintf( stderr, 
	       "image exceeds max size (%ldMb)...\n",
	       (long)(max_fasl_size / (1024 * 1024)) );
      exit(1);
    }
  return p;
}

static gc_obj_addr xlate_ptr( gc_obj_addr ptr )
{
  IRC_Header *p = ptr-1;
  assert( p->flagBits & IRC_MASK_CLIENTBYTE );
  return p->prev;
}



static IRC_Header *my_alloc_big( struct IRC_Heap *heap, UINT_32 size )
{
  return (IRC_Header *)fasl_alloc( size + sizeof(IRC_Header) );
}

static void my_alloc_chunk( struct IRC_Heap *heap )
{
  void *new;

  new = fasl_alloc( ALLOCATION_CHUNK_SIZE );
  if (new == (void *)(((char *)heap->moreSpacePtr + heap->spaceLeft)))
    {
      /* extend the old rgn instead of replacing it, thereby reducing
         external fragmentation */
       heap->spaceLeft += ALLOCATION_CHUNK_SIZE;
    }
  else
    {
      heap->spaceLeft = ALLOCATION_CHUNK_SIZE;
      heap->moreSpacePtr = new;
    }
}


IRC_Heap *build_fasl_heap( gc_obj_addr *roots, unsigned num_roots );

static int fasl_count_size( void *info, void *ptr )
{
  unsigned use;
  IRC_SizeClass *sc = IRCH(ptr)->sizeClass;

  if (sc->isLargeObject) {
    use = SIZEOF_PTR( GCPTR_TO_PTR( ptr ) )
      + sizeof( POBHeader )
      + sizeof( IRC_Header );
  } else {
    use = sc->itemSize;
  }

  (*(unsigned long*)info) += use;
  return 0;
}


static unsigned long fasl_est_size( void )
{
  unsigned long fasl_size = 0;

  gc_for_each( fasl_count_size, &fasl_size );
  printf( "fasl heap size estimate = %lu bytes\n", fasl_size );
  return fasl_size;
}

void fasl_save( const char *path,
	        gc_obj_addr *roots,
	        unsigned num_roots,
	        const char *comment_str )
{
  void *rgn;
  int fd;
  unsigned i;
  void *at;

  /* Round estimated size up to 64K and toss in an extra MB */

  max_fasl_size = (fasl_est_size() | 0xFFFF) + 1 + 1024*1024;
  printf( "fasl heap max size = %luK\n", max_fasl_size / 1024 );

  fd = fasl_open_file( path );
  if (fd < 0) {
    scheme_error( "save-fasl-image: could not create ~s", 
                  1, make_string(path) );
  }

#ifdef FASL_FIXED_ADDR
  at = (void *)FASL_FIXED_ADDR;
  if (at == (void *)fasl_loaded_at) {
    /*  we have already loaded an image at the primary address;
     *  try somewhere else
     */
    at = (void *)FASL_FIXED_ADDR_ALT;
  }
#else
  at = NULL;
#endif

  rgn = heap_reserve( fd, at );
  if (!rgn)
    {
      close(fd);
      fasl_error( "couldn't map file", path );
    }
  printf( "fasl load address: %p\n", rgn );

  rgn_ptr = rgn;
  rgn_limit = ((char *)rgn + max_fasl_size);

  hdr = FASL_ALLOC( struct FASL_Header );

  hdr->pre_loaded_at = (void *)rgn;
  memset( hdr->skip, ' ', sizeof hdr->skip );
  hdr->skip[0] = '\n';

  hdr->image_magic = FASL_MAGIC;
  hdr->fasl_version = FASL_VERSION;

  hdr->for_arch = FASL_ARCH;
  hdr->spare = 0;
  hdr->build_date = time(NULL);
  hdr->first_alloc_area = NULL;

  /* build the output image */

  hdr->heap = build_fasl_heap( roots, num_roots );

#ifdef DEBUG_ENABLE
  if (fasl_verbose >= 2)
  {
    struct IRC_Gen *g = &hdr->heap->theGenerations[0];
    int i;

    printf( "loaded generation (#%u):\n", g->genNum );
    for (i=0; i<NUM_PHYSICAL_SIZE_CLASSES; i++)
      {
	IRC_SizeClass *sc = &g->theSizeClasses[i];
	printf( "----  size class #%d (%u max) ----\n", i, sc->itemSize );
	printSizeClass( sc );
      }
  }
#endif


  hdr->root_list = FASL_ALLOCN( void *, (1+num_roots) );
  for (i=0; i<num_roots; i++)
  {
    hdr->root_list[i] = roots[i] ? xlate_ptr( roots[i] ) : NULL;
  }

  if (comment_str)
    {
      int len = (strlen(comment_str) | 15)+1;

      hdr->root_list[num_roots] = FASL_ALLOCN( char, len );
      printf( "root[%d] => %p => '%s'\n",
	      num_roots, 
	      hdr->root_list[num_roots], 
	      comment_str );
      strcpy( (char *)hdr->root_list[num_roots], comment_str );
    }
  else
    hdr->root_list[num_roots] = NULL;
  
  /* finish up */

  hdr->total_size = ((char *)rgn_ptr) - ((char *)hdr->pre_loaded_at);

  /* write out the data */

  if (fasl_verbose >= 1)
    printf( "loaded at: %p -- %lu bytes\n", 
	   hdr->pre_loaded_at, (unsigned long)hdr->total_size );

  heap_write( path, fd );
}

static IRC_Header *first, *last;

static struct IRC_PtrList all_dst;
static struct IRC_PtrList all_templates;

static void initPtrList( struct IRC_PtrList *ptrlist )
{
    ptrlist->first = NULL;
    ptrlist->last = NULL;
}

static void add_to_list( IRC_Header *item, int where )
{
  if (where < 0)  /* at the end */
    {
      if (first)
	{
	  last->next = item;
	}
      else
	{
	  first = item;
	}
      item->next = NULL;
    }
  else
    {
      IRC_Header *prev = NULL, *p = first;
      int i;

      for (i=0; p && i<where; i++)
	{
	  prev = p;
	  p = p->next;
	}
      if (prev)
	{
	  prev->next = item;
	  item->next = prev->next;
	}
      else
	{
	  item->next = first;
	  first = item;
	}
    }
}

void enq_src_obj( gc_obj_addr item )
{
  IRC_Header *cell = IRCH(item);

  if (!(cell->flagBits & IRC_MASK_CLIENTBYTE))
    {
      obj x = GCPTR_TO_PTR(item);
      unsigned im;

      im = class_image_mode( CLASSOF_PTR(x) );

      if (im == 4) /* image mode = 4 iff (instance? x <template>) */
	{
	  /* make sure its been unstubbed */

	  struct function_descr *fn;

	  fn = (struct function_descr *)OBJ_TO_RAW_PTR(gvec_ref(x,SLOT(1)));
	  if (fn->in_part->tag >= STUB_PART_TAG)
	    {
	      template_unstub(x);
	    }

	  /* if it's a template, put it at the end */
	  
	  if (first)
	    {
	      last->next = cell;
	    }
	  else
	    {
	      first = cell;
	    }
	  cell->next = NULL;
	  last = cell;
	}
      else
	{
	  IRC_Header *nxt;

	  /* otherwise, put it a little back from the front */
	  nxt = first;
	  if (first)
	    {
	      /* 0 back from the front == in the very front */
	      first = cell;
	    }
	  else
	    {
	      first = cell;
	    }
	  cell->next = nxt;
	  if (!nxt)
	    last = cell;
	}
    }
  cell->flagBits |= IRC_MASK_CLIENTBYTE;
}

void flush_src_queue( struct IRC_Heap *dheap )
{
  IRC_Header *cell;

  while (first)
    {
      UINT_32 spc;
      void *dst;
      POBHeader *pob;
      obj thing;
      unsigned im;

      cell = first;
      first = first->next;
      spc = cell->sizeClass->itemSize - sizeof( IRC_Header );
      if (fasl_verbose >= 3)
	printf( "copying object: %p (sc: %lu)", cell, (unsigned long)spc );

      thing = GCPTR_TO_PTR((cell+1));
      pob = PTR_TO_HDRPTR(thing);

      if (cell->sizeClass->isLargeObject)
	{
	  spc = pob->pob_size + sizeof( POBHeader );
	  if (fasl_verbose >= 3)
	    {
	      printf( " (large)" );
	    }
	}

      dst = IRC_alloc( dheap, spc );
      IRCH(dst)->flagBits |= IRC_MASK_MAPPED;

      IRC_ptrListAdd( &all_dst, dst );

      /* store the new pointer in the "prev" slot */
      cell->prev = dst;

      if (fasl_verbose >= 3)
	  printf( " ==> %p", dst );

      /* interpret it's class... */

      im = class_image_mode( pob->pob_class );
      if (fasl_verbose >= 3)
	{
	  printf( " (%s, %lu bytes, mode %u)\n", 
		 symbol_text( class_name(pob->pob_class) ),
		 (unsigned long)pob->pob_size,
		 im );
	  fflush( stdout );
	}
      
      memcpy( dst, cell+1, spc );
      if (im == 4)
	{
	  IRC_ptrListAdd( &all_templates, dst );
	}
      else if (im == 8)
	{
	  /* it's an <allocation-area>, push it onto the list
	     of them using the allocfn slot as the link ptr */

	  AllocArea *aa = (AllocArea *)((char *)dst + sizeof(POBHeader));

	  printf( "alloc area at: %p\n", aa );
	  aa->allocfn = (allocator_fn *)(hdr->first_alloc_area);
	  hdr->first_alloc_area = aa;
	}
      
      /* traverse it... */

      if (EQ(gvec_read(pob->pob_class,SLOT(1)),ZERO))
	{
	  unsigned i;
	  if (fasl_verbose >= 4)
	    printf( "  gvec (%lu slots)\n", 
		    (unsigned long)(SIZEOF_PTR(thing)/SLOT(1)) );

	  for (i=0; i<SIZEOF_PTR(thing); i+=SLOT(1))
	    {
	      obj slotv = gvec_read( thing, i );

	      if (fasl_verbose >= 5)
		printf( "  [%lu]: %08lx", 
			(unsigned long)(i/SLOT(1)), 
			(unsigned long)VAL(slotv) );
	      if (OBJ_ISA_PTR(slotv))
		{
		  if (fasl_verbose >= 5)
		    printf( " *\n" );
		  enq_src_obj( PTR_TO_GCPTR(slotv) );
		}
	      else
		{
		  if (fasl_verbose >= 5)
		    printf( "\n" );
		}
	    }
	}
      /* heap type 5 is mixvec(2) */
      else if (EQ(gvec_read(pob->pob_class,SLOT(1)),int2fx(5)))
	{
	  unsigned i;
	  if (fasl_verbose >= 4)
	    printf( "  mixvec(2) (%lu bytes)\n", 
		    (unsigned long)SIZEOF_PTR(thing) );

	  for (i=0; i<SLOT(2); i+=SLOT(1))
	    {
	      obj slotv = gvec_read( thing, i );

	      if (fasl_verbose >= 5)
		printf( "  [%lu]: %#lx", 
			(unsigned long)(i/SLOT(1)), 
			(unsigned long)VAL(slotv) );
	      if (OBJ_ISA_PTR(slotv))
		{
		  if (fasl_verbose >= 5)
		    printf( " *\n" );
		  enq_src_obj( PTR_TO_GCPTR(slotv) );
		}
	      else
		{
		  if (fasl_verbose >= 5)
		    printf( "\n" );
		}
	    }
	  
	}
    }
}

obj xlate_pob( obj item )
{
    return GCPTR_TO_PTR( xlate_ptr( PTR_TO_GCPTR(item) ) );
}



void translate_dst( void )
{
  struct IRC_PtrBucket *b;
  struct IRC_Header **p, **lim;

  for (b=all_dst.first; b; b=b->next)
    {
      lim = b->ptr;
      for (p=b->contents; p<lim; p++)
	{
	  obj thing;
	  POBHeader *pob;
	  
	  if (fasl_verbose >= 3)
	    printf( "translating object: %p", *p );
	  thing = GCPTR_TO_PTR((*p));
	  
	  pob = PTR_TO_HDRPTR(thing);
	  if (fasl_verbose >= 3)
	    {
	      printf( " (%s, %lu bytes, mode %u)\n", 
		     symbol_text( class_name(pob->pob_class) ),
		     (unsigned long)pob->pob_size,
		     class_image_mode( pob->pob_class ) );
	      fflush(stdout);
	    }
	  
	  /* traverse it... */
	  pob->pob_class = xlate_pob( pob->pob_class );

	  if (EQ(gvec_read(pob->pob_class,SLOT(1)),ZERO))
	    {
	      unsigned i;
	      obj *body = PTR_TO_DATAPTR(thing);

	      if (fasl_verbose >= 4)
		printf( "  gvec (%lu slots)\n", 
			(unsigned long)(SIZEOF_PTR(thing)/SLOT(1)) );
	      
	      for (i=0; i<SIZEOF_PTR(thing); i+=SLOT(1), body++)
		{
		  if (fasl_verbose >= 5)
		    printf( "  [%lu]: %#lx", 
			    (unsigned long)(i/SLOT(1)), 
			    (unsigned long)VAL(*body) );
		  if (OBJ_ISA_PTR(*body))
		    {
		      *body = xlate_pob( *body );
		      if (fasl_verbose >= 5)
			printf( " => %#lx\n", (unsigned long)VAL(*body) );
		    }
		  else
		    {
		      if (fasl_verbose >= 5)
			printf( "\n" );
		    }
		}
	    }
	  else if (EQ(gvec_read(pob->pob_class,SLOT(1)),int2fx(5)))
	    {
	      unsigned long i;
	      obj *body = PTR_TO_DATAPTR(thing);

	      if (fasl_verbose >= 4)
		printf( "  mixvec(2) (%lu slots)\n", 
			(unsigned long)SIZEOF_PTR(thing) );
	      
	      for (i=0; i<SLOT(2); i+=SLOT(1), body++)
		{
		  if (fasl_verbose >= 5)
		    printf( "  [%lu]: %#lx", 
			    (unsigned long)(i/SLOT(1)), 
			    (unsigned long)VAL(*body) );
		  if (OBJ_ISA_PTR(*body))
		    {
		      *body = xlate_pob( *body );
		      if (fasl_verbose >= 5)
			printf( " => %#lx\n", (unsigned long)VAL(*body) );
		    }
		  else
		    {
		      if (fasl_verbose >= 5)
			printf( "\n" );
		    }
		}
	    }
	}
    }
}

struct IRC_Heap *build_fasl_heap( gc_obj_addr *roots, unsigned num_roots )
{
  struct IRC_Heap *vheap = FASL_ALLOC(struct IRC_Heap);
  unsigned i;
  
  irc_init_heap( vheap );

  vheap->alloc_chunk_meth = my_alloc_chunk;
  vheap->alloc_big_meth = my_alloc_big;
  
  /* destructively and recursively copy the source image */

  initPtrList( &all_dst );
  initPtrList( &all_templates );

  first = NULL;
  last = NULL;
  for (i=0; i<num_roots; i++)
    {
      if (roots[i])
	enq_src_obj( roots[i] );
    }
  flush_src_queue( vheap );
  translate_dst();
  /* fasl_verbose = 9; */
  gen_code_ptrs( hdr, &all_templates );
  
  return vheap;
}


void fasl_save_vec( const char *path, obj vec, const char *comment )
{
  gc_obj_addr *rv;
  unsigned i, n = SIZEOF_PTR(vec) / SLOT(1);

  rv = malloc( n * sizeof(gc_obj_addr) );

  for (i=0; i<n; i++)
    {
      obj ent = gvec_ref( vec, SLOT(i) );

      if (OBJ_ISA_PTR(ent))
	rv[i] = PTR_TO_GCPTR(ent);
      else if (EQ(ent,FALSE_OBJ))
	rv[i] = NULL;
      else
	scheme_error( "save-fasl-image: root vector entry ~s is invalid",
		      1, ent );
    }

  fasl_save( path, rv, n, comment );
  free( rv );
}

/**/


#ifdef PLATFORM_AIX
#define DO_MMAP_XFLAGS ( MAP_VARIABLE )
#else
#define DO_MMAP_XFLAGS 0
#endif

#if defined(PLATFORM_SUNOS) || defined(__FreeBSD__) || \
    defined(PLATFORM_IRIX) || defined(PLATFORM_BSDI) || \
    defined(PLATFORM_BSD) || defined(PLATFORM_LINUX) || \
    defined(PLATFORM_AIX) || defined(PLATFORM_MACOSX)
#define DO_MMAP_SHARING MAP_SHARED
#else
#define DO_MMAP_SHARING MAP_PRIVATE
#endif

#if HAVE_MACH_H

#if defined(PLATFORM_NEXT)
#define mach_task_self()  task_self()
#endif

static void *do_mmap( int fd, void *try_at )
{
  kern_return_t rc;
  vm_address_t addr;

  addr = (vm_address_t)try_at;

  rc = vm_allocate( mach_task_self(), &addr, 
                    max_fasl_size, 
                    /* anywhere */ try_at ? FALSE : TRUE );

  if (rc != KERN_SUCCESS) {
    return NULL;
  }
  return (void *)addr;
}
#else /* HAVE_MACH_H */

static void *do_mmap( int fd, void *try_at )
{
  caddr_t try, at;
  int mode, flags, mm_fd;
  int ret;

  flags = DO_MMAP_SHARING|DO_MMAP_XFLAGS;

  if (try_at) {
    flags |= MAP_FIXED;
  }
  try = (caddr_t)try_at;

  at = mmap( try, max_fasl_size, PROT_READ|PROT_WRITE, flags, fd, 0 );

  if (at == MAP_FAILED) {
    return NULL;
  }
  return (void *)at;
}
#endif

static void *heap_reserve( int fd, void *try_at )
{
  void *p;
  
  if (try_at) {
    /* try at the recommended address */
    p = do_mmap( fd, try_at );
    if (p) {
      return p;
    }
  }

  /* fallback -- try at any old place */
  return do_mmap( fd, NULL );
}


#if FASL_NSHARE || USE_MACH_API
#define FASL_NEED_WRITEBACK (1)
#else
#define FASL_NEED_WRITEBACK (0)
#endif

static void heap_write( const char *path, int fd )
{
  unsigned long actual_size = hdr->total_size;

  printf( "total heap size: %lu (had estimated %lu)\n", 
          actual_size,
          max_fasl_size );

#if FASL_NEED_WRITEBACK
  {
    int n;
    n = write( fd, hdr, hdr->total_size );
    if (n != hdr->total_size)
      {
	perror( "writing fasl image" );
	exit(1);
      }
  }
#endif

#if USE_MACH_API
  if (vm_deallocate( task_self(), (vm_address_t)hdr, max_fasl_size )
      != KERN_SUCCESS) {
    mach_error( "vm_deallocate", rc );
  }
#else
  if (munmap( (caddr_t)hdr, max_fasl_size ) < 0) {
    perror( "munmap" );
  }
#endif
  hdr = NULL;

  ftruncate( fd, actual_size );
  close( fd );
}

#define FASL_PERM (S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP|S_IROTH|S_IWOTH)

#ifdef PLATFORM_AIX
#define FASL_NSHARE O_NSHARE
#else
#define FASL_NSHARE 0
#endif

#define FASL_NEED_TRUNCATE (1)

static int fasl_open_file( const char *path )
{
  int fd;

  fd = open( path, O_RDWR|O_CREAT|O_TRUNC|FASL_NSHARE, FASL_PERM );

  if (fd < 0) {
    fasl_error( "couldn't create", path );
  }

#if FASL_NEED_TRUNCATE
  if (ftruncate( fd, max_fasl_size ) < 0) {
    fasl_error( "couldn't allocate space", path );
  }
#endif
  return fd;
}

