
(define-class <ps-device> (<graphics-device>)
  (ps-port type: <output-port>)
  (ctm type: <transform>)
  (owner init-value: #f))

(define (close-ps-device (dev <ps-device>))
  (if (not (get-property dev 'eps? #f))
      (format (ps-port dev) "showpage\n"))
  (format (ps-port dev) "%%EOF\n")
  (close-output-port (ps-port dev)))

(define (open-ps-device (file <string>))
  (let* ((port (open-output-file file))
	 (dev (make <ps-device>
		    ps-port: port
		    ctm: (make-affine-transform))))
    (format port "%!PS-Adobe-3.0\n")
    (format port "%%Title: ~a\n" file)
    (format port "%%Creator: ~a (dv ~a)\n" 
	    (or (getenv "USER") "unknown user")
	    $dv-version)
    (format port "%%EndComments\n")
    dev))

(define (open-eps-device (file <string>) bbox)
  (let* ((port (open-output-file file))
	 (dev (make <ps-device>
		    ps-port: port
		    ctm: (make-affine-transform))))
    (set-property! dev 'eps? #t)
    (format port "%!PS-Adobe-3.0 EPSF-3.0\n")
    (let ((ll (lower-left bbox))
	  (ur (upper-right bbox)))
      (format port "%%BoundingBox: ~d ~d ~d ~d\n"
	      (inexact->exact (ceiling (psq (x ll))))
	      (inexact->exact (ceiling (psq (y ll))))
	      (inexact->exact (ceiling (psq (x ur))))
	      (inexact->exact (ceiling (psq (y ur))))))
    (format port "%%Title: ~a\n" file)
    (format port "%%Creator: ~a (dv ~a)\n" 
	    (or (getenv "USER") "unknown user")
	    $dv-version)
    (format port "%%EndComments\n")
    dev))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;   PostScript
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define-method psq ((self <fixnum>)) self)
(define-method psq ((self <number>)) 
  (if (exact? self)
      (exact->inexact self)
      self))

(define-method moveto ((self <ps-device>) (pt <point>))
  (format (ps-port self) "~a ~a moveto\n" (psq (x pt)) (psq (y pt)))
  (values))

(define-method lineto ((self <ps-device>) (pt <point>))
  (format (ps-port self) "~a ~a lineto\n" (psq (x pt)) (psq (y pt)))
  (values))

(define-method curveto ((self <ps-device>) (h1 <point>)
					   (h2 <point>) 
					   (pt <point>))
  (format (ps-port self)
	  "~d ~d  ~d ~d  ~d ~d curveto\n"
	  (psq (x h1)) (psq (y h1))
	  (psq (x h2)) (psq (y h2))
	  (psq (x pt)) (psq (y pt)))
  (values))

(define-method closepath ((self <ps-device>))
  (format (ps-port self) "closepath\n")
  (values))

(define-method stroke ((self <ps-device>))
  (format (ps-port self) "stroke\n")
  (values))

(define-method fill ((self <ps-device>))
  (format (ps-port self) "fill\n")
  (values))

(define (write-ps-string port (str <string>))
  (write-char #\( port)
  (let loop ((i 0))
    (if (eq? i (string-length str))
	(write-char #\) port)
	(let* ((ch (string-ref str i))
	       (esc (assq ch '((#\( "\\(")
			       (#\) "\\)")
			       (#\nl "\\n")
			       (#\cr "\\r")
			       (#\tab "\\t")
			       (#\bs "\\b")
			       (#\ff "\\f")
			       (#\\ "\\\\")))))
	  (if esc
	      (write-string port (cadr esc))
	      (if (and (>= (char->integer ch) 32)
		       (< (char->integer ch) 127))
		  (write-char ch port)
		  (format port "\\~03o" (char->integer ch))))
	  (loop (+ i 1))))))
  
(define-method show ((self <ps-device>) (str <string>))
  (write-ps-string (ps-port self) str)
  (format (ps-port self) " show\n")
  (values))

(define-method setfont ((self <ps-device>) (font <text-font>))
  (format (ps-port self) "/~a findfont ~d scalefont setfont\n"
	  (get-ps-font-name font)
	  (psq (font-size font))))

(define-method setlinewidth ((self <ps-device>) width)
  (format (ps-port self) "~d setlinewidth\n" (psq width)))

(define-method setcolor ((self <ps-device>) colspec)
  (case colspec
    ((black) (format (ps-port self) "0 setgray\n"))
    ((white) (format (ps-port self) "1 setgray\n"))
    (else
     (case (car colspec)
       ((rgb) (format (ps-port self) "~d ~d ~d setrgbcolor\n"
		      (cadr colspec)
		      (caddr colspec)
		      (cadddr colspec)))
       ((gray) (format (ps-port self) "~d setgray\n" (cadr colspec)))))))

(define-method device-color ((dev <ps-device>) colorspec)
  ;; identity xform -- let `setcolor' deal with it
  colorspec)

(define-method translate ((self <ps-device>) (delta <point>))
  (format (ps-port self) "~d ~d translate\n" (psq (x delta)) (psq (y delta))))

(define-method concat ((self <ps-device>) tm)
  (let ((v (vector-map psq (matrix tm))))
    (format (ps-port self)
	    "[~d ~d ~d ~d ~d ~d] concat\n"
	    (vector-ref v 0)
	    (vector-ref v 1)
	    (vector-ref v 2)
	    (vector-ref v 3)
	    (vector-ref v 4)
	    (vector-ref v 5))
    (values)))

;;;

(define-method with-gstate-saved ((self <ps-device>) thunk)
  (format (ps-port self) "gsave\n")
  (thunk)
  (format (ps-port self) "grestore\n")
  (values))
