/*				       	-*- c-file-style: "bsd" -*-
 * rproxy -- dynamic caching and delta update in HTTP
 * $Id: error.c,v 1.10 2000/08/24 02:51:47 mbp Exp $
 * 
 * Copyright (C) 2000 by Martin Pool <mbp@linuxcare.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* TODO: Put a short message about what went wrong into the status
 * line?  Is this allowed? */


#include "config.h"

#include "sysheaders.h"

#include <sys/socket.h>

#include "util.h"
#include "rproxy.h"
#include "msgpage.h"
#include "trace.h"
#include "request.h"
#include "error.h"

/*
 * If possible, send MSG_BUF to the client as an error message.
 */
static void
rp_send_error(request_t * req, char const *msg_buf)
{
    if (req && req_can_send_client_error_p(req)) {
	msg_error(req, HTTP_BAD_GATEWAY, msg_buf);
    }
}


/*
 * Drop the upstream connection immediately, because there's nothing
 * more we can say to them.
 */
static void
_rp_drop_upstream(request_t *req)
{
    if (req->f_upstream) {
        int fd = fileno(req->f_upstream);
        int const shutdown_both = 2;
        if (shutdown(fd, shutdown_both) == -1) {
            rp_log(LOGAREA_NET, LOG_ERR,
                         "error shutting down upstream socket fd %d: %s",
                         fd, strerror(errno));
            /* continue, we're going to die soon anyhow */
        }
        req->f_upstream = NULL;
    }
}


static void
_rp_close_downstream(request_t *req)
{
    if (req->f_from_client) {
        int fd = fileno(req->f_from_client);
        int const shutdown_recv = 0;
        if (shutdown(fd, shutdown_recv) == -1) {
            rp_log(LOGAREA_NET, LOG_ERR,
                         "error shutting down receive from downstream fd %d: %s",
                         fd, strerror(errno));
            /* continue, we're going to die soon anyhow */
        }
        req->f_from_client = NULL;
    }

    if (req->f_to_client) {
	/* There's no point sending anything else, that'd just confuse them.
	   */
	if (fclose(req->f_to_client) == EOF) {
            rp_log(LOGAREA_NET, LOG_ERR,
                         "error closing socket to downstream fd %d: %s",
                         fileno(req->f_to_client), strerror(errno));
            /* continue, we're going to die soon anyhow */
        }
	req->f_to_client = NULL;
    }
}


/*
 * Report an error in a request.  If you don't have a request yet,
 * then REQ may be null, but please avoid this if possible.
 *
 * We can't trust that any particular field of REQ will be
 * non-null.
 *
 * XXX: Perhaps it would be best if we just exit(1) at the end of this
 * routine?  It'd save a lot of trouble trying to unwind things on the
 * way back, and I'm sure we don't catch everything at the moment.
 *
 * TODO: Linger until all the data is acked by downstream.
 */
void
rp_request_failed(request_t * req, int code, char const *fmt, ...)
{
    va_list         ap;
    int             ret;
    char            msg_buf[1000];

    if (!code)
	code = HTTP_BAD_GATEWAY;

    req->status = code;

    va_start(ap, fmt);
    ret = vslprintf(msg_buf, sizeof(msg_buf) - 1, fmt, ap);
    msg_buf[sizeof(msg_buf) - 1] = 0;
    va_end(ap);
    
    rp_log(LOGAREA_HTTP, LOG_ERR, "%s", msg_buf);

    rp_send_error(req, msg_buf);

    _rp_drop_upstream(req);
    _rp_close_downstream(req);
    
    exit(EXIT_REQ_ERR);
}
