#* 
#* ------------------------------------------------------------------
#* Role PlayingDB V2.0 by Deepwoods Software
#* ------------------------------------------------------------------
#* RPGEdMap.tcl - Map editor
#* Created by Robert Heller on Mon Aug 24 14:31:22 1998
#* ------------------------------------------------------------------
#* Modification History: 
#* $Log: RPGEdMap.tcl,v $
#* Revision 1.6  1999/07/13 01:29:16  heller
#* Fix documentation: spelling, punctuation, etc.
#*
#* Revision 1.5  1999/04/19 21:36:01  heller
#* Update HelpTopics to match help topic links.
#*
#* Revision 1.4  1999/03/28 06:20:44  heller
#* Update on-line help.
#*
#* Revision 1.3  1999/01/02 00:07:53  heller
#* Small error in WriteMap.
#*
#* Revision 1.2  1999/01/01 18:00:12  heller
#* Remove Load button and add Help button
#*
#* Revision 1.1  1998/12/29 22:13:30  heller
#* Initial revision
#*
#* ------------------------------------------------------------------
#* Contents:
#* ------------------------------------------------------------------
#*  
#*     Role Playing DB -- A database package that creates and maintains
#* 		       a database of RPG characters, monsters, treasures,
#* 		       spells, and playing environments.
#* 
#*     Copyright (C) 1995,1998  Robert Heller D/B/A Deepwoods Software
#* 			51 Locke Hill Road
#* 			Wendell, MA 01379-9728
#* 
#*     This program is free software; you can redistribute it and/or modify
#*     it under the terms of the GNU General Public License as published by
#*     the Free Software Foundation; either version 2 of the License, or
#*     (at your option) any later version.
#* 
#*     This program is distributed in the hope that it will be useful,
#*     but WITHOUT ANY WARRANTY; without even the implied warranty of
#*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#*     GNU General Public License for more details.
#* 
#*     You should have received a copy of the GNU General Public License
#*     along with this program; if not, write to the Free Software
#*     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#* 
#*  
#* 

#@Chapter:RPGEdMap.tcl -- Maps of the adventure
#@Label:RPGEdMap.tcl
#$Id: RPGEdMap.tcl,v 1.6 1999/07/13 01:29:16 heller Rel1 $
# This file implements the GUIs that create and edit maps.


proc RPGEdMap {{filename {}}} {
# This procedure edits a map file.  A map contains a series of levels, each of
# which contain one or more spaces.
# <in> filename -- the name of the map file to edit.
# [index] RPGEdMap!procedure

  global tk_version

  set toplevel [GenerateToplevelName rpgEdMap]

  RPGToplevel .$toplevel {Role Playing V2 Map Editor} Map
  wm withdraw .$toplevel
  global .$toplevel
  upvar #0 .$toplevel data
  set data(filename) "$filename"
  set data(filetype) "map"
  if {[string length "$filename"] > 0} {
    if {[ReadMap .$toplevel "$filename"] == 0} {
      CloseWindow .$toplevel
      tkerror "File does not exist or is not readable: $filename"
      return
    }
  } elseif {[CreateNewMap .$toplevel] == 0} {
    CloseWindow .$toplevel
    return
  }

  # build widget .$toplevel.upper
  frame .$toplevel.upper \
    -borderwidth {2} \
    -relief {ridge}

  # build widget .$toplevel.upper.canvasMF
  frame .$toplevel.upper.canvasMF

  # build widget .$toplevel.upper.canvasMF.canvasVscroll
  frame .$toplevel.upper.canvasMF.canvasVscroll \
    -borderwidth {2}

  # build widget .$toplevel.upper.canvasMF.canvasVscroll.mapCanvas
  canvas .$toplevel.upper.canvasMF.canvasVscroll.mapCanvas \
    -borderwidth {2} \
    -confine {0} \
    -height {150} \
    -relief {sunken} \
    -width {150} \
    -scrollregion "$data(scrollregion)" \
    -xscrollcommand ".$toplevel.upper.canvasMF.hScrollFill.hscroll set" \
    -yscrollcommand ".$toplevel.upper.canvasMF.canvasVscroll.vScroll set"
  bind .$toplevel.upper.canvasMF.canvasVscroll.mapCanvas <1> \
		[list UnselectLoadedSpace ".$toplevel" %x %y]

  # build widget .$toplevel.upper.canvasMF.canvasVscroll.vScroll
  scrollbar .$toplevel.upper.canvasMF.canvasVscroll.vScroll \
    -command ".$toplevel.upper.canvasMF.canvasVscroll.mapCanvas yview"
  # build widget .$toplevel.upper.canvasMF.hScrollFill
  frame .$toplevel.upper.canvasMF.hScrollFill \
    -borderwidth {2}

  # build widget .$toplevel.upper.canvasMF.hScrollFill.hscroll
  scrollbar .$toplevel.upper.canvasMF.hScrollFill.hscroll \
    -command ".$toplevel.upper.canvasMF.canvasVscroll.mapCanvas xview" \
    -orient {horizontal}

  # build widget .$toplevel.upper.canvasMF.hScrollFill.frame22
  frame .$toplevel.upper.canvasMF.hScrollFill.frame22 \
    -borderwidth {2} \
    -height {17} \
    -width {24}

  # build widget .$toplevel.upper.info
  frame .$toplevel.upper.info \
    -borderwidth {2}

  # build widget .$toplevel.upper.info.name
  frame .$toplevel.upper.info.name

  # build widget .$toplevel.upper.info.name.label9
  label .$toplevel.upper.info.name.label9 \
    -text {Name:}

  # build widget .$toplevel.upper.info.name.value
  entry .$toplevel.upper.info.name.value \
    -width {0} \
    -textvariable ".[set toplevel](name)"

  # build widget .$toplevel.upper.info.level
  frame .$toplevel.upper.info.level

  # build widget .$toplevel.upper.info.level.label9
  label .$toplevel.upper.info.level.label9 \
    -text {Current Level:}

  # build widget .$toplevel.upper.info.level.value
  scale .$toplevel.upper.info.level.value \
    -length {13} \
    -orient {horizontal} \
    -sliderlength {4} \
    -from $data(highestLevel) \
    -to $data(deepestLevel) \
    -width {4} \
    -command "SelectLevel .$toplevel"

  # build widget .$toplevel.upper.info.description
  frame .$toplevel.upper.info.description \
    -relief {raised}

  # build widget .$toplevel.upper.info.description.scrollbar1
  scrollbar .$toplevel.upper.info.description.scrollbar1 \
    -command ".$toplevel.upper.info.description.value yview"

  # build widget .$toplevel.upper.info.description.value
  text .$toplevel.upper.info.description.value \
    -height {9} \
    -width {28} \
    -wrap {word} \
    -yscrollcommand ".$toplevel.upper.info.description.scrollbar1 set"
  bindtags .$toplevel.upper.info.description.value \
	[list .$toplevel.upper.info.description.value Text .$toplevel all UpdDescription]

  # build widget .$toplevel.controls
  frame .$toplevel.controls \
    -borderwidth {2} \
    -relief {ridge}

  # build widget .$toplevel.controls.spaceEdit
  frame .$toplevel.controls.spaceEdit \
    -borderwidth {2}

  # build widget .$toplevel.controls.spaceEdit.label18
  label .$toplevel.controls.spaceEdit.label18 \
    -borderwidth {4} \
    -relief {ridge} \
    -text {Space:}

  # build widget .$toplevel.controls.spaceEdit.button19
  button .$toplevel.controls.spaceEdit.button19 \
    -text {New} \
    -command "AddNewSpace .$toplevel"

  # build widget .$toplevel.controls.spaceEdit.button20
  button .$toplevel.controls.spaceEdit.button20 \
    -text {Delete} \
    -command "DeleteSpace .$toplevel" \
    -state disabled

  # build widget .$toplevel.controls.spaceEdit.name
  label .$toplevel.controls.spaceEdit.name \
    -relief {sunken} \
    -textvariable ".[set toplevel](selectedSpaceName)"

  # build widget .$toplevel.buttons
  frame .$toplevel.buttons \
    -borderwidth {2}

  # build widget .$toplevel.buttons.button24
  button .$toplevel.buttons.button24 \
    -text {Save} \
    -command "SaveMap .$toplevel"

  # pack master .$toplevel.upper
  pack configure .$toplevel.upper.canvasMF \
    -expand 1 \
    -fill both \
    -side left
  pack configure .$toplevel.upper.info \
    -expand 1 \
    -fill both \
    -side right

  # pack master .$toplevel.upper.canvasMF
  pack configure .$toplevel.upper.canvasMF.canvasVscroll \
    -expand 1 \
    -fill both
  pack configure .$toplevel.upper.canvasMF.hScrollFill \
    -fill x \
    -side bottom

  # pack master .$toplevel.upper.canvasMF.canvasVscroll
  pack configure .$toplevel.upper.canvasMF.canvasVscroll.mapCanvas \
    -expand 1 \
    -fill both \
    -side left
  pack configure .$toplevel.upper.canvasMF.canvasVscroll.vScroll \
    -fill y \
    -side right

  # pack master .$toplevel.upper.canvasMF.hScrollFill
  pack configure .$toplevel.upper.canvasMF.hScrollFill.hscroll \
    -expand 1 \
    -fill x \
    -side left
  pack configure .$toplevel.upper.canvasMF.hScrollFill.frame22 \
    -side right

  # pack master .$toplevel.upper.info
  pack configure .$toplevel.upper.info.name \
    -expand 1 \
    -fill x
  pack configure .$toplevel.upper.info.level \
    -expand 1 \
    -fill x
  pack configure .$toplevel.upper.info.description \
    -expand 1 \
    -fill both

  # pack master .$toplevel.upper.info.name
  pack configure .$toplevel.upper.info.name.label9 \
    -side left
  pack configure .$toplevel.upper.info.name.value \
    -expand 1 \
    -fill x \
    -side right

  # pack master .$toplevel.upper.info.level
  pack configure .$toplevel.upper.info.level.label9 \
    -side left
  pack configure .$toplevel.upper.info.level.value \
    -expand 1 \
    -fill x \
    -side right

  # pack master .$toplevel.upper.info.description
  pack configure .$toplevel.upper.info.description.scrollbar1 \
    -fill y \
    -side right
  pack configure .$toplevel.upper.info.description.value \
    -expand 1 \
    -fill both

  # pack master .$toplevel.controls
  pack configure .$toplevel.controls.spaceEdit \
    -expand 1 \
    -fill x

  # pack master .$toplevel.controls.spaceEdit
  pack configure .$toplevel.controls.spaceEdit.label18 \
    -side left
  pack configure .$toplevel.controls.spaceEdit.button19 \
    -side left
  pack configure .$toplevel.controls.spaceEdit.button20 \
    -side left
  pack configure .$toplevel.controls.spaceEdit.name \
    -expand 1 \
    -fill x \
    -side right

  # pack master .$toplevel.buttons
  pack configure .$toplevel.buttons.button24 \
    -expand 1 \
    -side left

  # pack master .$toplevel
  pack configure .$toplevel.upper \
    -expand 1 \
    -fill both
  pack configure .$toplevel.controls \
    -expand 1 \
    -fill x
  pack configure .$toplevel.buttons \
    -expand 1 \
    -fill x

  # build canvas items .$toplevel.upper.canvasMF.canvasVscroll.mapCanvas
  .$toplevel.upper.info.description.value insert end "$data(description)"
  SelectLevel .$toplevel [.$toplevel.upper.info.level.value get]
  set data(selectedSpaceId) {}
  set data(selectedSpaceIndexString) {}
  set data(selectedSpaceName) {}
# end of widget tree

  wm deiconify .$toplevel

}

proc SelectLevel {tl level} {
# This procedure selects the level to display.  It is bound to the level
# scale widget.
# <in> tl -- the toplevel.
# <in> level -- the new level.
# [index] SelectLevel!procedure

  upvar #0 $tl data
  set level [expr int($level)]
  set data(level)  $level
  $tl.upper.canvasMF.canvasVscroll.mapCanvas delete all
  set data(selectedSpaceId) {}
  set data(selectedSpaceIndexString) {}
  set data(selectedSpaceName) {}
  ReDrawSpaces $tl
}

proc ReadCompleteList {fp bufferVar} {
# Helper function to read in a complete in the Tcl sense -- balanced parens,
# braces, brackets, and quote marks.
# <in> fp -- file pointer object.
# <name> bufferVar -- the name of the buffer variable to collect the input in.
# [index] ReadCompleteList!procedure

  upvar 1 $bufferVar buffer
  set buffer {}
  set result -1
  while {1} {
    if {[gets $fp line] < 0} {
      return $result
    }
    append buffer "$line"
    if {$result < 0} {
      set result [string length "$buffer"]
    } else {
      incr result [string length "$buffer"]
    }
    if {[string length "$buffer"] > 0 && \
	[info complete "$buffer"] == 1} {return result}
  }
}


proc ReadMap {tl filename} {
# Procedure to read a map into a GUI toplevel. A map file contains a set of 
# name value items that map to the data array used to represent the map while
# in memory.  The saved fields include: shape, name, description, deepestLevel,
# highestLevel, maxX, maxY, minX, minY, and all of the space file names 
# (Spaces,l,x,y). 
# <in> tl -- the name of the toplevel.
# <in> filename -- the name of the file to read.
# [index] ReadMap!procedure

  upvar #0 $tl data

  if {[catch [list open "$filename" r] rfp]} {
    bgerror "Could not open map file: $filename for reading: $rfp"
    return 0
  }

  while {[ReadCompleteList $rfp list] >= 0} {
    set s [lindex $list 0]
    set v "[lindex $list 1]"
    set data($s) "$v"
    if {[regexp {^Spaces,([^,]+),([^,]+),(.+)$} "$s" whole l x y] > 0} {
      set buffer [Record]
      if {[catch [list $buffer ReadRecord "$v"] err]} {
        bgerror "Missing or unreadable space file: $v, skipped: $err"
	unset data($s)
	rename $buffer {}
	continue
      }
      if {[string compare {*Space} "[lindex [$buffer ReturnRecord] 0]"] != 0} {
	bgerror "Not a space file: $v"
	unset data($s)
	rename $buffer {}
	continue
      }
      set data(SpaceDirty,$l,$x,$y) 0
      set data(Spaces,$l,$x,$y) [Space]
      set data(SpaceFiles,$l,$x,$y) "$v"
      $data(Spaces,$l,$x,$y) UpdateFromRecord $buffer
      rename $buffer {}
    }
  }

  set data(cleanupFun) CleanUpMap
  set data(scrollregion) [list [expr $data(minX) * 10] \
			       [expr $data(minY) * 10] \
			       [expr $data(maxX) * 10] \
			       [expr $data(maxY) * 10]]
  set data(dirty) 0

  return 1
}

proc CreateNewMap {tl} {
# This procedure creates a new map.
# <in> tl -- the toplevel.
# [index] CreateNewMap!procedure

  upvar #0 $tl data

  # build widget .createMapDialog
  if {"[info procs XFEdit]" != ""} {
    catch "XFDestroy .createMapDialog"
  } {
    catch "destroy .createMapDialog"
  }
  toplevel .createMapDialog 

  # Window manager configurations
  wm positionfrom .createMapDialog ""
  wm sizefrom .createMapDialog ""
  wm maxsize .createMapDialog 1000 1000
  wm minsize .createMapDialog 10 10
  wm title .createMapDialog {Create New Map}
  wm transient .createMapDialog .


  # build widget .createMapDialog.name
  frame .createMapDialog.name \
    -borderwidth {2} \
    -relief {ridge}

  # build widget .createMapDialog.name.label4
  label .createMapDialog.name.label4 \
    -text {Name:}

  # build widget .createMapDialog.name.value
  entry .createMapDialog.name.value \
    -width {0} \
    -textvariable "[set tl](name)"

  # build widget .createMapDialog.sqhex
  frame .createMapDialog.sqhex \
    -borderwidth {2} \
    -relief {ridge}

  # build widget .createMapDialog.sqhex.radiobutton6
  radiobutton .createMapDialog.sqhex.radiobutton6 \
    -relief {raised} \
    -text {Squares} \
    -value {Space::Square} \
    -variable "[set tl](shape)"

  # build widget .createMapDialog.sqhex.radiobutton7
  radiobutton .createMapDialog.sqhex.radiobutton7 \
    -relief {raised} \
    -text {Hexes} \
    -value {Space::Hexagon} \
    -variable "[set tl](shape)"

  # build widget .createMapDialog.description
  frame .createMapDialog.description \
    -borderwidth {2} \
    -relief {ridge}

  # build widget .createMapDialog.description.scrollbar1
  scrollbar .createMapDialog.description.scrollbar1 \
    -command {.createMapDialog.description.value yview} \
    -relief {raised}

  # build widget .createMapDialog.description.value
  text .createMapDialog.description.value \
    -height {10} \
    -width {70} \
    -wrap {word} \
    -yscrollcommand {.createMapDialog.description.scrollbar1 set}

  # build widget .createMapDialog.buttons
  frame .createMapDialog.buttons \
    -borderwidth {2}

  # build widget .createMapDialog.buttons.button8
  button .createMapDialog.buttons.button8 \
    -text {Create} \
    -command "CreateNewMapCreateButton $tl .createMapDialog"

  # build widget .createMapDialog.buttons.button9
  button .createMapDialog.buttons.button9 \
    -text {Dismiss} \
    -command "CreateNewMapDismisButton $tl .createMapDialog"

  # build widget .createMapDialog.buttons.button10
  button .createMapDialog.buttons.button10 \
    -text {Help} \
    -command {HelpTopic {Create Map Dialog}}

  # build widget .createMapDialog.extents
  frame .createMapDialog.extents \
    -borderwidth {2} \
    -relief {ridge}

  # build widget .createMapDialog.extents.label2
  label .createMapDialog.extents.label2 \
    -text {Min X:}

  # build widget .createMapDialog.extents.minX
  entry .createMapDialog.extents.minX \
    -width {0} \
    -textvariable "[set tl](minX)"
  bindtags .createMapDialog.extents.minX \
	[list .createMapDialog.extents.minX Entry .createMapDialog all IntEntry]

  # build widget .createMapDialog.extents.label4
  label .createMapDialog.extents.label4 \
    -text {Min Y:}

  # build widget .createMapDialog.extents.minY
  entry .createMapDialog.extents.minY \
    -width {0} \
    -textvariable "[set tl](minY)"
  bindtags .createMapDialog.extents.minY \
	[list .createMapDialog.extents.minY Entry .createMapDialog all IntEntry]

  # build widget .createMapDialog.extents.label6
  label .createMapDialog.extents.label6 \
    -text {Max X:}

  # build widget .createMapDialog.extents.maxX
  entry .createMapDialog.extents.maxX \
    -width {0} \
    -textvariable "[set tl](maxX)"
  bindtags .createMapDialog.extents.maxX \
	[list .createMapDialog.extents.maxX Entry .createMapDialog all IntEntry]

  # build widget .createMapDialog.extents.label8
  label .createMapDialog.extents.label8 \
    -text {Max Y:}

  # build widget .createMapDialog.extents.maxY
  entry .createMapDialog.extents.maxY \
    -width {0} \
    -textvariable "[set tl](maxY)"
  bindtags .createMapDialog.extents.maxY \
	[list .createMapDialog.extents.maxY Entry .createMapDialog all IntEntry]

  # build widget .createMapDialog.extents.label10
  label .createMapDialog.extents.label10 \
    -text {Max Depth:}

  # build widget .createMapDialog.extents.deepestLevel
  entry .createMapDialog.extents.deepestLevel \
    -width {0} \
    -textvariable "[set tl](deepestLevel)"
  bindtags .createMapDialog.extents.deepestLevel \
	[list .createMapDialog.extents.deepestLevel Entry .createMapDialog all IntEntry]

  # build widget .createMapDialog.extents.label12
  label .createMapDialog.extents.label12 \
    -text {Max Height:}

  # build widget .createMapDialog.extents.highestLevel
  entry .createMapDialog.extents.highestLevel \
    -width {0} \
    -textvariable "[set tl](highestLevel)"
  bindtags .createMapDialog.extents.highestLevel \
	[list .createMapDialog.extents.minX Entry .createMapDialog all IntEntry]

  # pack master .createMapDialog.name
  pack configure .createMapDialog.name.label4 \
    -side left
  pack configure .createMapDialog.name.value \
    -expand 1 \
    -fill x \
    -side right

  # pack master .createMapDialog.sqhex
  pack configure .createMapDialog.sqhex.radiobutton6 \
    -expand 1 \
    -side left
  pack configure .createMapDialog.sqhex.radiobutton7 \
    -expand 1 \
    -side right

  # pack master .createMapDialog.description
  pack configure .createMapDialog.description.scrollbar1 \
    -fill y \
    -side right
  pack configure .createMapDialog.description.value \
    -expand 1 \
    -fill both

  # pack master .createMapDialog.buttons
  pack configure .createMapDialog.buttons.button8 \
    -expand 1 \
    -side left
  pack configure .createMapDialog.buttons.button9 \
    -expand 1 \
    -side left
  pack configure .createMapDialog.buttons.button10 \
    -expand 1 \
    -side right

  # pack master .createMapDialog.extents
  pack configure .createMapDialog.extents.label2 \
    -side left
  pack configure .createMapDialog.extents.minX \
    -expand 1 \
    -fill x \
    -side left
  pack configure .createMapDialog.extents.label4 \
    -side left
  pack configure .createMapDialog.extents.minY \
    -expand 1 \
    -fill x \
    -side left
  pack configure .createMapDialog.extents.label6 \
    -side left
  pack configure .createMapDialog.extents.maxX \
    -expand 1 \
    -side left
  pack configure .createMapDialog.extents.label8 \
    -side left
  pack configure .createMapDialog.extents.maxY \
    -expand 1 \
    -fill x \
    -side left
  pack configure .createMapDialog.extents.label10 \
    -side left
  pack configure .createMapDialog.extents.deepestLevel \
    -expand 1 \
    -fill x \
    -side left
  pack configure .createMapDialog.extents.label12 \
    -side left
  pack configure .createMapDialog.extents.highestLevel \
    -expand 1 \
    -fill x \
    -side left

  # pack master .createMapDialog
  pack configure .createMapDialog.name \
    -expand 1 \
    -fill x
  pack configure .createMapDialog.sqhex \
    -expand 1 \
    -fill x
  pack configure .createMapDialog.description \
    -fill both
  pack configure .createMapDialog.extents \
    -expand 1 \
    -fill x
  pack configure .createMapDialog.buttons \
    -expand 1 \
    -fill x

  set data(name) {}
  .createMapDialog.description.value insert end {}
  set data(minX) -75
  set data(minY) -75
  set data(maxX) 75
  set data(maxY) 75
  set data(deepestLevel) 5
  set data(highestLevel) 1
  set data(shape) {Space::Square}
  set data(cleanupFun) CleanUpMap
  set data(CreateResult) 0
  set data(dirty) 0
# end of widget tree

  set oldFocus [focus]
  set oldGrab [grab current .createMapDialog]
  if {$oldGrab != ""} {
    set grabStatus [grab status $oldGrab]
  }

  focus .createMapDialog.name.value
  grab .createMapDialog
  tkwait window .createMapDialog

  catch {focus $oldFocus}
  if {$oldGrab != ""} {
        if {$grabStatus == "global"} {
            grab -global $oldGrab
        } else {
            grab $oldGrab
        }
  }
  return $data(CreateResult)
}

proc CheckWriteDirtyRecordMap {tl} {
# This procedure is called when the toplevel window is being closed and the
# data is ``dirty'' (modified).  It takes care of saving the data if the user
# chooses to save it.
# <in> tl -- the toplevel.
# [index] CheckWriteDirtyRecordMap!procedure

  upvar #0 $tl data
  global $tl

#  get Map base file name

  set filename "$data(filename)"
  set filetype "$data(filetype)"
  set saveP [tk_dialog .askDirty "Save map?" "Save modified Map data?" \
	questhead 0 "Yes" "No"]
  if {$saveP == 1} {
    CleanUpMap $tl
    return
  }
  if {[string length "$filename"] == 0} {
    set filename [tk_getSaveFile -defaultextension ".$filetype" \
				 -initialfile "new.$filetype" \
				 -initialdir "[pwd]" \
				 -filetypes [list [list "Map files" \
							  "*.$filetype"]]\
				 -parent . \
				 -title {File to save map data in}]
    if {[string length "$filename"] == 0} {
      CleanUpMap $tl
      return
    }
    if {[string length "[file extension $filename]"] == 0} {
      set filename "$filename.$filetype"
    }
  }
  WriteMap $tl $filename 1

  CleanUpMap $tl
}

proc SaveMap {tl} {
# Procedure to save a map object.  Bound to the ``Save'' menu item on the 
# ``File'' menu.
# <in> tl -- the toplevel.
# [index] SaveMap!procedure

  upvar #0 $tl data
  set filename "$data(filename)"
  set filetype "$data(filetype)"
  if {[string length "$filename"] == 0} {
    set filename [tk_getSaveFile -defaultextension ".$filetype" \
				 -initialfile "new.$filetype" \
				 -initialdir "[pwd]" \
				 -filetypes [list [list "Map files" \
							  "*.$filetype"]]\
				 -parent . \
				 -title {File to save map data in}]
    if {[string length "$filename"] == 0} {
      return
    }
    if {[string length "[file extension $filename]"] == 0} {
      set filename "$filename.$filetype"
    }
    set data(filename) "$filename"
  }
  WriteMap $tl $filename 1
  set data(dirty) 0
  foreach s [array names data SpaceDirty,*] {
    set data($s) 0
  }
}

proc SaveAsMap {tl} {
# This procedure saves a map in an alternative file.  Bound to the ``SaveAs...''
# menu item of the ``File'' menu.
# <in> tl -- the toplevel to save.
# [index] SaveAsMap!procedure

  upvar #0 $tl data
  set filename "$data(filename)"
  set filetype "$data(filetype)"
  if {[string length "$filename"] == 0} {set filename "new.$filetype"}
  set filename [tk_getSaveFile -defaultextension ".$filetype" \
                                 -initialfile "$filename" \
                                 -initialdir "[pwd]" \
                                 -filetypes [list [list "Map files" \
                                                          "*.$filetype"]]\
                                 -parent . \
                                 -title {File to save map data in}]
  if {[string length "$filename"] == 0} {
      return
  }
  if {[string length "[file extension $filename]"] == 0} {
      set filename "$filename.$filetype"
  }
  set data(filename) "$filename"
  WriteMap $tl $filename 0 1
  set data(dirty) 0
  foreach s [array names data SpaceDirty,*] {
    set data($s) 0
  }
}

proc WriteMap {tl filename {IfDirty 0} {forceName 0}} {
# Procedure to write out a map object to a file.  The data is written out as 
# a series of Tcl 2 element lists: slot name slot value.  The slots saved are:
# shape, name, description, deepestLevel, highestLevel, maxX, maxY, minX,
# minY, and all of the space files.  Each (dirty) space is also written out.
# The space file names are formed from the base map file name with the spaces
# l, x, and y coordinates encoded into the filenames
# <in> tl -- the toplevel to write.
# <in> filename -- the filename to write to.
# <in> IfDirty -- write out only dirty spaces.
# <in> forceName -- compute new filenames for all spaces.
# [index] WriteMap!procedure

  upvar #0 $tl data

# write base map file
#data(shape)                    = Space::Hexagon
#data(name)                     = Test
#data(description)              = 
#data(deepestLevel)             = 5
#data(highestLevel)             = 1
#data(maxX)                     = 75
#data(maxY)                     = 75
#data(minX)                     = -75
#data(minY)                     = -75

  if {[catch [list open "$filename" w] wfp]} {
    bgerror "Could not open map file: $filename for writing: $wfp"
    return 0
  }

  foreach s {shape name description deepestLevel highestLevel maxX maxY 
             minX minY} {
    puts $wfp [list $s "$data($s)"]
  }

  set fBase [file rootname "$filename"]

  foreach spI [array names data Spaces,*] {
    set space $data($spI)
    if {[string compare "[info command $space]" {}] == 0} {
      set space [Space -this $space]
    }
    regsub -all {Spaces} "$spI" {SpaceFiles} spFileI
    regsub -all {Spaces} "$spI" {SpaceDirty} spDI
    set spFile "$data($spFileI)"
    regexp {^Spaces,([^,]+),([^,]+),(.+)$} "$spI" whole l x y
    if {[regsub -all {\.} "$x" {_} xx] > 0} {set x "$xx"}
    if {[regsub -all {\.} "$y" {_} yy] > 0} {set y "$yy"}
    if {$data($spDI) > 0 || $IfDirty == 0 || $forceName == 1 || \
	[string length "$spFile"] == 0} {
      # Write Space file
#data(SpaceFiles,1,0.0,0.0)     = 
      if {[string length "$spFile"] == 0 || $forceName == 1} {
	set spFile "${fBase}SpaceL${l}X${x}Y${y}.space"
      }
      set buffer [Record -this [$space RawData]]
      $buffer WriteRecord "$spFile"
      rename $buffer {}
    }
    # write space info
    puts $wfp [list $spI "$spFile"]
  }  

  close $wfp
  return 1
}

proc CleanUpMap {tl} {
# Clean up procedure.  Free up all of the space objects and then free up the
# map array itself.
# <in> tl -- the toplevel to clean up.
# [index] CleanUpMap!procedure

  upvar #0 $tl data
  global $tl

  foreach spI [array names data Spaces,*] {
    set space $data($spI)
    if {[string compare "[info command $space]" {}] == 0} {
      set space [Space -this $space]
    }
    rename $space {}
  }  
  unset $tl
}


proc CreateNewMapCreateButton {tl dialog} {
# Procedure to handle the Create button on the create new map dialog.
# <in> tl -- the toplevel to attach the new map to.
# <in> dialog -- the dialog used to set up the creation parameters.
# [index] CreateNewMapCreateButton!procedure

  upvar #0 $tl data
  if {[string length "set data(name)"] == 0} {
    return
  }
  set data(CreateResult) 1
  set data(description) "[.createMapDialog.description.value get 1.0 end]"
  set data(scrollregion) [list [expr $data(minX) * 10] \
			       [expr $data(minY) * 10] \
			       [expr $data(maxX) * 10] \
			       [expr $data(maxY) * 10]]
  set data(dirty) 1
  if {"[info procs XFEdit]" != ""} {
    catch "XFDestroy $dialog"
  } {
    catch "destroy $dialog"
  }
}

proc CreateNewMapDismisButton {tl dialog} {
# Procedure to handle the Dismiss button on the create new map dialog.
# <in> tl -- the toplevel to attach the new map to.
# <in> dialog -- the dialog used to set up the creation parameters.
# [index] CreateNewMapDismisButton!procedure

  upvar #0 $tl data
  set data(CreateResult) 0
  if {"[info procs XFEdit]" != ""} {
    catch "XFDestroy $dialog"
  } {
    catch "destroy $dialog"
  }
}  

proc AddNewSpace {tl} {
# Procedure to add a new space to the current level.
# <in> tl -- the toplevel.
# [index] AddNewSpace!procedure

  upvar #0 $tl data
  set newSpace "[CreateNewSpace $tl]"
  if {[string length "$newSpace"] > 0} {
    if {[catch [list set data(Spaces,$data(level),[$newSpace CenterX],[$newSpace CenterY])] old] == 0} {
      rename $old {}
    }
    if {[catch [list set data(SpaceFiles,$data(level),[$newSpace CenterX],[$newSpace CenterY])] old] == 0} {
      file delete -force "$old"
    }
    set data(Spaces,$data(level),[$newSpace CenterX],[$newSpace CenterY]) $newSpace
    set data(SpaceFiles,$data(level),[$newSpace CenterX],[$newSpace CenterY]) {}
    set data(SpaceDirty,$data(level),[$newSpace CenterX],[$newSpace CenterY]) 1
    set data(dirty) 1
    ReDrawSpaces $tl
  }
}

proc ReDrawSpaces {tl} {
# Procedure to re-draw the spaces in the current level.
# <in> tl -- the toplevel.
# [index] ReDrawSpaces!procedure

  upvar #0 $tl data
  set canvas $tl.upper.canvasMF.canvasVscroll.mapCanvas
  $canvas delete all
  set data(selectedSpaceId) {}
  set spaceElts [array names data "Spaces,$data(level),*,*"]
  foreach spelt $spaceElts {
    set space $data($spelt)
    set drawcmd "[$space MakeGraphicCommannd .1]"
    set id [eval [concat $canvas create $drawcmd]]
    $canvas bind $id <Double-Button-1> [list EditLoadedSpace \
			       $tl \
			       "$data(SpaceFiles,$data(level),[$space CenterX],[$space CenterY])" \
			       $space \
			       $data(level)]
    
    $canvas bind $id <1> [list SelectLoadedSpace \
			       $tl \
			       $space \
			       $id \
			       "$data(level),[$space CenterX],[$space CenterY]"]
    if {[string compare "$data(selectedSpaceIndexString)" \
			"$data(level),[$space CenterX],[$space CenterY]"] == 0} {
      SelectLoadedSpace $tl $space $id \
			"$data(level),[$space CenterX],[$space CenterY]"
    }    
  }
}

proc SelectLoadedSpace {tl space id indexString} {
# Procedure bound to button 1 -- Select.  This procedure marks the space as
# selected.  The space's border is changed from black to read (highlighted)
# and the space's access info is saved in the data array for use elsewhere.
# <in> tl -- the toplevel.
# <in> space -- the Space object.
# <in> id -- the space's canvas id.
# <in> indexString -- the space's index string.
# [index] SelectLoadedSpace!procedure

  upvar #0 $tl data
  set canvas $tl.upper.canvasMF.canvasVscroll.mapCanvas
  if {[string length "$data(selectedSpaceId)"] > 0} {
    $canvas itemconfigure $data(selectedSpaceId) -outline black
  }
  $canvas itemconfigure $id -outline red
  set data(selectedSpaceId) $id
  set data(selectedSpaceIndexString) "$indexString"
  set data(selectedSpaceName) "[$space Name]"
  $tl.controls.spaceEdit.button20 configure -state normal
}

proc UnselectLoadedSpace {tl Mx My} {
# Procedure bound to the base canvas's button 1 event. This procedure is used
# to deselect the currently selected space when the user clicks on the canvas 
# background.
# <in> tl -- the toplevel.
# <in> Mx -- mouse X coordinate.
# <in> My -- mouse Y coordinate.
# [index] UnselectLoadedSpace!procedure

  upvar #0 $tl data
  set canvas $tl.upper.canvasMF.canvasVscroll.mapCanvas
  set x [$canvas canvasx $Mx]
  set y [$canvas canvasy $My]
  if {[string length "$data(selectedSpaceId)"] > 0} {
    set newitem [$canvas find overlapping $x $y [expr $x + 1.0] [expr $y + 1.0]]
    set olditem $data(selectedSpaceId)
    if {[string compare "$newitem" "$olditem"] == 0} {return}
    $canvas itemconfigure $olditem -outline black
  }
  set data(selectedSpaceId) {}
  set data(selectedSpaceIndexString) {}
  set data(selectedSpaceName) {}
  $tl.controls.spaceEdit.button20 configure -state disabled
}

proc DeleteSpace {tl} {
# Procedure to delete the selected space.
# <in> tl -- the toplevel.
# [index] DeleteSpace!procedure

  upvar #0 $tl data
  $tl.controls.spaceEdit.button20 configure -state disabled
  if {[string compare "$data(selectedSpaceId)" {}] == 0} {
    return
  }
  set canvas $tl.upper.canvasMF.canvasVscroll.mapCanvas
  $canvas delete $data(selectedSpaceId)
  set sp $data(Spaces,$data(selectedSpaceIndexString))
  set spFile $data(SpaceFiles,$data(selectedSpaceIndexString))
  unset data(SpaceDirty,$data(selectedSpaceIndexString))
  rename $sp {}
  if {[string length "$spFile"] > 0} {
    catch [list file delete -force "$spFile"]
  }
  unset data(SpaceFiles,$data(selectedSpaceIndexString))
  unset data(Spaces,$data(selectedSpaceIndexString))
  set data(selectedSpaceId) {}
  set data(selectedSpaceIndexString) {}
  set data(selectedSpaceName) {}
}  
    
proc OpenMap {tl} {
# Procedure to open a new Map GUI toplevel and load a map file into it.
# <in> tl -- the current toplevel.
# [index] OpenMap!procedure

  if {"$tl" == {.}} {
    set data(filename) {}
    set data(filetype) map
    set data(class) Map
  } else {
    upvar #0 $tl data
  }
  set filename "$data(filename)"
  set filetype "$data(filetype)"
  set initdir "[file dirname $filename]"
  if {[string compare "$initdir" {.}] == 0} {set initdir "[pwd]"}
  if {[string compare "[string index "$initdir" 0]" {/}] != 0} {
    set initdir [file join "[pwd]" "$initdir"]
  }
  set filename [tk_getOpenFile -defaultextension ".$data(filetype)" \
			   -filetypes [list [list "$data(class) files" \
					"*.$filetype"]]\
			   -parent $tl \
			   -initialfile "$filename" \
			   -initialdir "$initdir" \
			   -title "File to load $data(class) data from"]
  if {[string length "$filename"] == 0} {return}
  RPGEd$data(class) "$filename"
}


package provide RPGEdMap 1.0
