/****h* ROBODoc/Configuration
 * FUNCTION
 *   Functions to access the ROBODoc configuration and configuration
 *   file (robodoc.rc) or the file specified with the --rc option.
 *
 *   The robodoc.rc file consists of a number of blocks.  Each
 *   block starts with a line of the form 
 *
 *   <block name>:
 *
 *   This is followed by a number of lines of data.  Each line starts
 *   with at least one space followed by the actual data.
 *
 *   This module parses this data and stores it in the global
 *   configuration.
 *
 * NOTES
 *   Is missing a lot of documentation.
 *
 ******
 * $Id: roboconfig.c,v 1.23 2003/12/30 17:39:36 gumpu Exp $
 */

#include <stdio.h>
#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "headertypes.h"
#include "util.h"
#include "roboconfig.h"

#ifdef DMALLOC
#include <dmalloc.h>
#endif


/****v* Configuration/item_names
 * NAME
 *   item_names -- default item names
 * SYNOPSIS
 *   char *default_item_names[]
 * FUNCTION
 *   Defines the names of items that ROBODoc recognized as
 *   items by default if none are specified in the
 *   robodoc.rc file.
 * SOURCE
 */

char               *default_item_names[] = {
    "SOURCE",                   /* source code inclusion */
    "NAME",                     /* Item name + short description */
    "COPYRIGHT",                /* who own the copyright : "(c) <year>-<year> by <company/person>" */
    "SYNOPSIS", "USAGE",        /* how to use it */
    "FUNCTION", "DESCRIPTION", "PURPOSE",       /* what does it */
    "AUTHOR",                   /* who wrote it */
    "CREATION DATE",            /* when did the work start */
    "MODIFICATION HISTORY", "HISTORY",  /* who done what changes when */
    "INPUTS", "ARGUMENTS", "OPTIONS", "PARAMETERS", "SWITCHES", /* what can we feed into it */
    "OUTPUT", "SIDE EFFECTS",   /* what output will be made */
    "RESULT", "RETURN VALUE",   /* what do we get returned */
    "EXAMPLE",                  /* a clear example of the items use */
    "NOTES",                    /* any annotations */
    "DIAGNOSTICS",              /* diagnostical output */
    "WARNINGS", "ERRORS",       /* warning & error-messages */
    "BUGS",                     /* known bugs */
    "TODO", "IDEAS",            /* what to implement next & ideas */
    "PORTABILITY",              /* where does it come from, where will it work */
    "SEE ALSO",                 /* references */
    "METHODS", "NEW METHODS",   /* oop methods */
    "ATTRIBUTES", "NEW ATTRIBUTES",     /* oop attributes */
    "TAGS",                     /* tagitem description */
    "COMMANDS",                 /* command description */
    "DERIVED FROM",             /* oop super class */
    "DERIVED BY",               /* oop sub class */
    "USES", "CHILDREN",         /* what modules are used by this one */
    "USED BY", "PARENTS",       /* which modules do use this */
    NULL,                       /* don't delete, so we can count how many there are... */
};

/***********/

/* Maximum length of a line in the configuration file */
#define BUFFER_LENGTH 2048


/****v* Configuration/configuration
 * FUNCTION
 *   This global stores all the configuration parameters specified on
 *   the command line and in the robodoc.rc file.
 * SOURCE
 */

struct RB_Configuration configuration;

/*****/

static void         RB_AllocItemNames( void );
static void         RB_AllocOptions( unsigned int argc, char **argv );
static T_Block_Kind RB_BlockKind( char *line );
static T_Line_Kind  RB_ConfigLineKind( char *line );
static void         RB_FirstScan( FILE * f );
static void         RB_SecondScan( FILE * f );
static void         RB_AllocIgnoreItems( void );
static void         RB_Alloc_Parameters( struct Parameters* parameters, unsigned int size );
static void         RB_AddParameter( char* name, struct Parameters* parameters );
static void         RB_GetParameters( char* line, struct Parameters* parameters );
static void         RB_Install_Custom_HeaderTypes( void );


/****f* Configuration/RB_ReadConfiguration
 * FUNCTION
 *   Read the robodoc configuration file, and create
 *   a RB_Configuration structure.
 *
 *   The configuration is scanned twice. The first scan
 *   is to determine the number of elements in each section,
 *   in the second scan the elements are read and stored.
 *
 * SYNOPSIS
 *   void RB_ReadConfiguration( unsigned int argc, char **argv, char* filename )
 * INPUTS
 *   o argc -- the arg count as received by main()
 *   o argv -- the arg valules as received by main()
 *   o filename -- an optional filename.  If none is given,
 *               "robodoc.rc" is used.
 * RESULT
 *   An initialized configuration.
 * SOURCE
 */

void
RB_ReadConfiguration( unsigned int argc, char **argv, char* filename )
{
    FILE               *f;

    if ( filename ) 
    {
        f = fopen( filename, "r" );
        if ( !f ) 
        {
            RB_Panic( "Can't open %s\n", filename );
        }
    }
    else
    {
        f = fopen( "robodoc.rc", "r" );
        /* TODO look at other locations. */
    }

    if ( f )
    {
        RB_FirstScan( f );
    }

    RB_AllocItemNames();
    RB_AllocOptions( argc, argv );
    RB_AllocIgnoreItems();
    RB_Alloc_Parameters( &( configuration.custom_headertypes ), 10 );
    RB_Alloc_Parameters( &( configuration.ignore_files ), 10 );
    if ( f )
    {
        rewind( f );
        RB_SecondScan( f );
        fclose( f );
    }
    RB_Install_Custom_HeaderTypes();
}

/******/

/****if* Config/RB_ConfigLineKind
 * FUNCTION
 *   Deterimine the kind of line we a currently processing.
 * SYNOPSIS
 *   static T_Line_Kind RB_ConfigLineKind( char* line )
 * INPUTS
 *   line -- the current line.
 * RETURN
 *   The kind of line.
 * SOURCE
 */

static T_Line_Kind
RB_ConfigLineKind( char *line )
{
    T_Line_Kind         kind = CFL_UNKNOWN;

    if ( *line == '\0' )
    {
        kind = CFL_EMPTYLINE;
    }
    else if ( *line == '#' )
    {
        kind = CFL_REMARK;
    }
    else if ( isspace( *line ) )
    {
        char               *cur_char = line;

        for ( ; *cur_char && isspace( *cur_char ); ++cur_char )
        {
            /* Empty */
        }
        if ( *cur_char == '\0' )
        {
            kind = CFL_EMPTYLINE;
        }
        else
        {
            /* There is atleast one non-space character */
            kind = CFL_PARAMETER;
        }
    }
    else
    {
        kind = CFL_SECTION;
    }
    return kind;
}

/********/


static T_Block_Kind
RB_BlockKind( char *line )
{
    T_Block_Kind      section_kind = SK_UNKNOWN;

    if ( strcmp( line, "items:" ) == 0 )
    {
        section_kind = SK_ITEMS;
    }
    else if ( strcmp( line, "options:" ) == 0 )
    {
        section_kind = SK_OPTIONS;
    }
    else if ( strcmp( line, "extensions:" ) == 0 )
    {
        section_kind = SK_EXTENSIONS;
        printf("Warning:  the 'extensions:' section is obsolete, use 'ignore files:' instead\n");
    }
    else if ( strcmp( line, "ignore items:" ) == 0 )
    {
        section_kind = SK_IGNOREITEMS;
    }
    else if ( strcmp( line, "headertypes:" ) == 0 )
    {
        section_kind = SK_HEADERTYPES;
    }
    else if ( strcmp( line, "ignore files:" ) == 0 )
    {
        section_kind = SK_IGNORE_FILES;
    }
    else
    {
        RB_Panic( "unknown section kind \"%s\"\n", line );
    }
    return section_kind;
}


static void RB_Install_Custom_HeaderTypes( void )
{
    unsigned int i;
    for ( i = 0; i < configuration.custom_headertypes.number; i += 3 )
    {
        char*  typeCharString;
        char*  indexName; 
        char*  fileName;
        typeCharString = configuration.custom_headertypes.names[ i ];
        indexName = configuration.custom_headertypes.names[ i + 1 ];
        fileName = configuration.custom_headertypes.names[ i + 2 ];
        if ( strlen( typeCharString ) > 1 )
        {
            RB_Panic( "Trying to add a new headertype with parameters"
                      "%s %s %s.\nHowever the typecharacter %s is more "
                      "than one character long\n",
                      typeCharString, indexName, fileName,
                      typeCharString );
        }
        RB_AddHeaderType( typeCharString[ 0 ], indexName, fileName );
    }
}



static void RB_Alloc_Parameters( struct Parameters* parameters, unsigned int size )
{
    parameters->size = size;
    parameters->number = 0;
    parameters->names = calloc( size, sizeof( char* ) );
}


/* TODO Documentation */
static void RB_AddParameter( char* name, struct Parameters* parameters )
{
    parameters->names[ parameters->number ] = RB_StrDup( name );
    parameters->number++;
    if ( parameters->number >= parameters->size )
    {
        parameters->size *= 2;
        parameters->names = realloc( parameters->names, parameters->size * sizeof( char* ) );
    }
}


/* TODO Documentation */
static void RB_GetParameters( char* line, struct Parameters* parameters )
{
    int                 i;
    int                 n = strlen( line );

    /* Remove any spaces at the end of the line */
    for ( i = n - 1; i >= 0 && isspace( line[i] ); --i )
    {
        line[i] = '\0';
    }

    assert( i > 0 );            /* If i <= 0 then the line was empty
                                   and that cannot be, because this 
                                   is supposed to be a parameter */

    /* Skip any white space at the begin of the line. */
    n = strlen( line );
    for ( i = 0; i < n && isspace( line[i] ); ++i )
    {
        /* Empty */
    }
    line += i;

    n = strlen( line );
    for ( i = 0;  i < n; /* empty */)
    {
        char* name = line;
        if ( line[ i ] == '"' ) 
        {
            /* It is quoted string, fetch everything until
             * the next quote */
            ++name; /* skip the double quote */
            for ( ++i; ( i < n ) && ( line[ i ] != '"') ; ++i )
            {
                /* empty */
            }
            if ( i == n ) 
            {
                RB_Panic( "Missing quote in your .rc file in line:\n  %s\n", line );
            }
            else
            {
#if defined(__APPLE__)
                /* hacked because of error when compiling on Mac OS X */ 
                assert ( line[ i ] == 34 );
#else
                assert ( line[ i ] == '"' );
#endif
                line[ i ] = '\0';
                RB_AddParameter( name, parameters );
            }
        }
        else
        {
            /* a single word, find the next space */
            for ( ; ( i < n ) && !isspace( line[ i ] ); ++i )
            {
                /* empty */
            }
            if ( i < n )
            {
                line[ i ] = '\0';
            }
            RB_AddParameter( name, parameters );
        }
        /* Is there anything left? */
        if ( i < n )
        {
            /* skip any spaces until the next parameter */
            ++i; /* first skip the nul character */
            line += i;
            n = strlen( line );
            for ( i = 0 ; ( i < n ) && isspace( line[ i ] ); ++i )
            {
                /* empty */
            }
            line += i;
            n = strlen( line );
            i = 0;
        }
    }
}



char               *
RB_GetParameter( char *line )
{
    int                 i;
    int                 n = strlen( line );

    /* Remove any spaces at the end of the line */
    for ( i = n - 1; i >= 0 && isspace( line[i] ); --i )
    {
        line[i] = '\0';
    }

    assert( i > 0 );            /* If i <= 0 then the line was empty
                                   and that cannot be, because this 
                                   is supposed to be a parameter */

    /* Skip any white space at the begin of the line. */
    n = strlen( line );
    for ( i = 0; i < n && isspace( line[i] ); ++i )
    {
        /* Empty */
    }
    line += i;


    return RB_StrDup( line );
}


void
RB_StripCR( char *line )
{
    char               *c;

    for ( c = line; *c; ++c )
    {
        if ( *c == '\n' )
        {
            *c = '\0';
        }
    }
}

void
RB_Free_Configuration( void  )
{
    unsigned int        i;

    for ( i = 0; i < configuration.no_items; ++i )
    {
        free( configuration.item_names[i] );
    }
    free( configuration.item_names );

    for ( i = 0; i < configuration.no_ignore_items; ++i )
    {
        free( configuration.ignore_items[i] );
    }
    free( configuration.ignore_items );
    /* TODO  Deallocate custom_headertypes */
}



/* TODO Documentation */
static void
RB_AllocIgnoreItems( void )
{
    if ( configuration.no_ignore_items )
    {
        configuration.ignore_items =
            ( char ** ) calloc( configuration.no_ignore_items,
                                sizeof( char * ) );
    }
    else
    {
        configuration.ignore_items = NULL;
    }
}


/* TODO Documentation */
static void
RB_AllocOptions( unsigned int argc, char **argv )
{
    unsigned int   i;
    RB_Alloc_Parameters( &( configuration.options ), argc );
    for ( i = 0; i < argc; ++i )
    {
        RB_AddParameter( argv[i], &( configuration.options ) );
    }
}




/* TODO Documentation */
static void
RB_AllocItemNames( void )
{
    if ( configuration.no_items )
    {
        configuration.no_items++;
        configuration.item_names =
            ( char ** ) calloc( configuration.no_items, sizeof( char * ) );
        assert( configuration.item_names );
        /* The SOURCE item is always included */
        configuration.item_names[0] = RB_StrDup( "SOURCE" );
    }
    else
    {
        /* No item names were defined, so we use the default ones */
        unsigned int                 i;

        for ( ; default_item_names[configuration.no_items];
              ++configuration.no_items )
        {
            /* empty */
        }
        configuration.item_names =
            ( char ** ) calloc( configuration.no_items, sizeof( char * ) );
        for ( i = 0; i < configuration.no_items; ++i )
        {
            configuration.item_names[i] = RB_StrDup( default_item_names[i] );
        }
    }
    RB_Say( "There are %d item names\n", configuration.no_items );
}



/* Count number of paramters and options */
static void
RB_FirstScan( FILE * f )
{
    char                line_buffer[BUFFER_LENGTH];
    T_Block_Kind      section_kind = SK_UNKNOWN;
    T_Line_Kind         line_kind = CFL_UNKNOWN;

    while ( fgets( line_buffer, BUFFER_LENGTH, f ) )
    {
        RB_StripCR( line_buffer );
        line_kind = RB_ConfigLineKind( line_buffer );
        switch ( line_kind )
        {
        case CFL_REMARK:
        case CFL_EMPTYLINE:    /* fall through */
            break;
        case CFL_SECTION:
            section_kind = RB_BlockKind( line_buffer );
            break;
        case CFL_PARAMETER:
            {
                switch ( section_kind )
                {
                case SK_ITEMS:
                    ++configuration.no_items;
                    break;
                case SK_EXTENSIONS:
                    /* Obsolete */
                    break;
                case SK_IGNOREITEMS:
                    ++configuration.no_ignore_items;
                    break;
                case SK_OPTIONS:      /* fall through */
                case SK_HEADERTYPES: 
                case SK_IGNORE_FILES:
                    break;
                case SK_UNKNOWN:
                    assert( 0 );
                }
            }
            break;
        case CFL_UNKNOWN:
        default:
            assert( 0 );
        }
    }
}


/* TODO Documentation */
static void
RB_SecondScan( FILE * f )
{
    char                line_buffer[BUFFER_LENGTH];
    T_Block_Kind      section_kind = SK_UNKNOWN;
    T_Line_Kind         line_kind = CFL_UNKNOWN;
    int                 itemname_nr = 1;        /* 0 if for the SOURCE_ITEM */
    int                 ignore_nr = 0;

    while ( fgets( line_buffer, BUFFER_LENGTH, f ) )
    {
        RB_StripCR( line_buffer );
        line_kind = RB_ConfigLineKind( line_buffer );
        switch ( line_kind )
        {
        case CFL_REMARK:
        case CFL_EMPTYLINE:    /* fall through */
            /* Do nothing */
            break;
        case CFL_SECTION:
            section_kind = RB_BlockKind( line_buffer );
            break;
        case CFL_PARAMETER:
            {
                switch ( section_kind )
                {
                case SK_ITEMS:
                    configuration.item_names[itemname_nr] =
                        RB_GetParameter( line_buffer );
                    ++itemname_nr;
                    break;
                case SK_OPTIONS:
                    RB_GetParameters( line_buffer, &( configuration.options ) );
                    break;
                case SK_EXTENSIONS:
                    /* Obsolete */
                    break;
                case SK_IGNOREITEMS:
                    configuration.ignore_items[ignore_nr] =
                        RB_GetParameter( line_buffer );
                    ++ignore_nr;
                    break;
                case SK_HEADERTYPES:
                    RB_GetParameters( line_buffer, &( configuration.custom_headertypes ) );
                    break;
                case SK_IGNORE_FILES:
                    RB_GetParameters( line_buffer, &( configuration.ignore_files ) );
                    break;
                case SK_UNKNOWN:
                    break;
                default:
                    assert( 0 );
                }
            }
            break;
        case CFL_UNKNOWN:
        default:
            assert( 0 );
        }
    }
}


