///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
// direct solver interface
//
#include "rheolef/solver.h"
#include "rheolef/eye.h"
#include "solver_wrapper.h"
#include "solver_ldlt_builtin.h"
#include "solver_cholmod.h"
#include "solver_umfpack.h"
#include "solver_mumps.h"
#include "solver_trilinos_ifpack.h"
#include "solver_no_trilinos_ifpack.h"
#include "solver_pastix.h"
#include <typeinfo>

namespace rheolef {

template <class T, class M>
void
solver_abstract_rep<T,M>::set_preconditionner (const solver_basic<T,M>&)
{
  fatal_macro ("undefined set_preconditionner for this solver");
}
template <class T, class M>
void
solver_rep<T,M>::set_preconditionner (const solver_basic<T,M>& sa)
{
  check_macro (_ptr != 0, "solver::set_preconditionner(): uninitialized solver");
  _ptr->set_preconditionner (sa);
}

template <class T, class M>
typename solver_abstract_rep<T,M>::determinant_type
solver_abstract_rep<T,M>::det() const
{
  //std::string type = pretty_typename_of(*this);
  std::string type = typeid(*this).name();
  warning_macro ("undefined computation of the determinant for this solver="<<type);
  return determinant_type();
}
template <class T, class M>
typename solver_rep<T,M>::determinant_type
solver_rep<T,M>::det() const
{
  check_macro (_ptr != 0, "solver::det(): uninitialized solver");
  return _ptr->det();
}

template<class T, class M>
solver_rep<T,M>::solver_rep ()
 : _ptr(0)
{
}
template<class T, class M>
solver_rep<T,M>::solver_rep (const csr<T,M>& a, const solver_option& opt)
 : _ptr(0)
{
  typedef solver_wrapper_rep<T,M> rep;
  _ptr = new_macro (rep(a,opt));
}
template<class T, class M>
const solver_option&
solver_rep<T,M>::option() const
{
  check_macro (_ptr != 0, "solver::option(): uninitialized solver");
  return _ptr -> option();
}
template<class T, class M>
void
solver_rep<T,M>::build_eye ()
{
  typedef eye_rep<T,M> rep;
  if (_ptr) delete_macro (_ptr);
  _ptr = new_macro (rep());
}
template<class T, class M>
void
solver_rep<T,M>::build_lu (const csr<T,M>& a, const solver_option& opt)
{
#ifdef _RHEOLEF_HAVE_MUMPS
  // note: for both sym & unsym matrix, mumps appears to be faster than umfpack and cholmod
  //bool prefer_cholmod = (a.dis_ext_nnz() == 0) && a.is_symmetric() && a.is_definite_positive(); // when sdp and diagonal per process
  //bool prefer_umfpack = opt.force_seq || (a.dis_ext_nnz() == 0); // when diagonal per process
  bool prefer_cholmod = false; // prefer mumps that seems faster than cholmod, even in seq case
  bool prefer_umfpack = opt.force_seq // idem, but when building schur complement, mumps is still buggy
                      || ((a.dis_ext_nnz() == 0) && 
			  (opt.prefered_library == "umfpack"	
# if !(defined( _RHEOLEF_HAVE_MUMPS_WITH_METIS) || \
       defined( _RHEOLEF_HAVE_MUMPS_WITH_PARMETIS) || \
       defined( _RHEOLEF_HAVE_MUMPS_WITH_SCOTCH) || \
       defined( _RHEOLEF_HAVE_MUMPS_WITH_PTSCOTCH))
			   // when nproc==1, prefer umfpack to mumps when mumps is compiled with poor ordering
                           || opt.prefered_library == ""
#endif // prefer umfpack when mumps has poor ordering
                          ));
#else  // !_RHEOLEF_HAVE_MUMPS
  bool prefer_cholmod = (a.dis_ext_nnz() == 0) && a.is_symmetric() && a.is_definite_positive(); // when sdp and diagonal per process
  bool prefer_umfpack = (a.dis_ext_nnz() == 0)  // when (unsym or (sym & undef)) & diag per proc
                      || (opt.prefered_library == "umfpack"); // explicitely prefer (more robust on ill-conditioned systems)
#endif // ! _RHEOLEF_HAVE_MUMPS

#ifdef _RHEOLEF_HAVE_CHOLMOD
  if (prefer_cholmod) {
    // diagonal bloc per process => use cholmod
    typedef solver_cholmod_rep<T,M> rep;
    if (_ptr) delete_macro (_ptr);
    _ptr = new_macro (rep(a,opt));
    return;
  }
#endif // _RHEOLEF_HAVE_CHOLMOD

#ifdef _RHEOLEF_HAVE_UMFPACK
  if (prefer_umfpack) {
    trace_macro ("use umfpack");
    // diagonal bloc per process => use umfpack
    typedef solver_umfpack_rep<T,M> rep;
    if (_ptr) delete_macro (_ptr);
    _ptr = new_macro (rep(a,opt));
    return;
  }
#endif // _RHEOLEF_HAVE_UMFPACK

#ifdef _RHEOLEF_HAVE_MUMPS
  trace_macro ("use mumps");
  typedef solver_mumps_rep<T,M> rep;
#elif defined(_RHEOLEF_HAVE_PASTIX)
  typedef solver_pastix_rep<T,M> rep;
  opt.iterative = 0;
#else // ! _RHEOLEF_HAVE_TRILINOS && !_RHEOLEF_HAVE_PASTIX
  typedef solver_ldlt_builtin_rep<T,M> rep;
#endif // ! _RHEOLEF_HAVE_MUMPS

  if (_ptr) delete_macro (_ptr);
  _ptr = new_macro (rep(a,opt));
}
template<class T, class M>
void
solver_rep<T,M>::build_ilu (const csr<T,M>& a, const solver_option& opt)
{
#ifdef _RHEOLEF_HAVE_TRILINOS
  typedef solver_trilinos_ifpack_rep<T,M> rep;
#elif defined(_RHEOLEF_HAVE_PASTIX)
  typedef solver_pastix_rep<T,M> rep;
  opt.iterative = 1;
#else // ! _RHEOLEF_HAVE_TRILINOS && !_RHEOLEF_HAVE_PASTIX
  typedef solver_no_trilinos_ifpack_rep<T,M> rep;
#endif
  if (_ptr) delete_macro (_ptr);
  _ptr = new_macro (rep(a,opt));
}
template<class T, class M>
solver_rep<T,M>::~solver_rep ()
{
  if (_ptr) delete_macro (_ptr);
  _ptr = 0;
}
template<class T, class M>
void 
solver_rep<T,M>::update_values (const csr<T,M>& a)
{
  _ptr->update_values (a);
}
template<class T, class M>
vec<T,M>
solver_rep<T,M>::solve (const vec<T,M>& b) const
{
  return _ptr->solve (b);
}
template<class T, class M>
vec<T,M>
solver_rep<T,M>::trans_solve (const vec<T,M>& b) const
{
  return _ptr->trans_solve (b);
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
#define _RHEOLEF_instanciation(T,M)            	\
template class solver_abstract_rep<T,M>;	\
template class solver_rep<T,M>;

_RHEOLEF_instanciation(Float,sequential)
#ifdef _RHEOLEF_HAVE_MPI
_RHEOLEF_instanciation(Float,distributed)
#endif // _RHEOLEF_HAVE_MPI
#undef _RHEOLEF_instanciation

} // namespace rheolef
