///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// i/o for cemagref surface mesh
// = bidimensional mesh with a cote z(x,y)
//
// author: Pierre.Saramito@imag.fr
//
#include "rheolef/georep.h"
#include "rheolef/field.h"
#include "rheolef/rheostream.h" // i/o utility
#include "geo-connectivity.h"
using namespace rheolef;
using namespace std;

// =============================================================
// cemagref boundary condition identifiers
// and rheolef associated naming convention
// =============================================================


static 
const char* 
domain_name [] = {

        "output", 			// id = 1
	"reflexion", 			// id = 2
	"input", 			// id = 3
	"constrained_output"		// id = 4

	// others will be labeled "unnamed5" "unnamed6" ...
};
static
const georep::size_type 
max_cemagref_id = sizeof(domain_name)/sizeof(char*);

static
int
get_cemagref_id (const string& domname)
{
    for (int k = 0; k < 4; k++)
	if (domname == domain_name[k]) return k+1;
    return 0;
}
static
inline
georep::size_type
add1(georep::size_type i)
{
   return (i == numeric_limits<georep::size_type>::max() ? 0 : i+1);
}
static
inline
georep::size_type
sub1(georep::size_type i)
{
   return (i == 0 ? numeric_limits<georep::size_type>::max() : i-1);
}
static
inline
void
decr(georep::size_type& i)
{
   i = sub1(i);
}
// =============================================================
// input in cemagref format 
// =============================================================

istream& 
georep::get_cemagref (istream& in, field& zh, bool load_z)
{
   size_type undef = numeric_limits<size_type>::max();

   // ----------------------------------------------------------
   // load connectivity
   // ----------------------------------------------------------
   if (!in.good()) error_macro("bad input stream for cemagref mesh.");
   _dim = 2;
   _version = 2; // have edge numbering
   reset_counters ();

   size_type n_elt;
   in >> n_elt;
   resize(n_elt);

   size_type idx = 0;
   georep::iterator last_elt = end();
   for (georep::iterator q = begin(); q != last_elt && in.good(); q++, idx++) {
      geo_element& K = *q;

      // -------------------------------------------------------
      // get edge infos
      // -------------------------------------------------------
      size_type n_edge;
      in >> n_edge;
      check_macro (n_edge == 3 || n_edge == 4, "unexpected element with " << n_edge << " edges");
      size_type n_vert = n_edge; // in 2d
      K.set_variant (n_vert, 2);
      for (size_type j = 0; j < n_edge; j++) {
         size_type edge_j;
	 in >> edge_j;
	 K.set_edge (j, sub1(edge_j));
      }
      // -------------------------------------------------------
      // get vertice infos
      // -------------------------------------------------------
      in >> n_vert;
      check_macro (n_vert == 3 || n_vert == 4, "unexpected element with " << n_vert << " vertices");
      for (size_type j = 0 ; j < n_vert; j++) {
	 in >> K[j];
	 decr(K[j]);
      }
      K.set_index (idx);
      _count_geo     [K.dimension()]++;
      _count_element [K.variant()]++;

      // -------------------------------------------------------
      // skip neighbour infos
      // -------------------------------------------------------
      size_type n_dummy, dummy;
      in >> n_dummy;
      for (size_type j = 0; j < n_dummy; j++)  in >> dummy;
   }
   // ----------------------------------------------------------
   // get edges and domain id
   // ----------------------------------------------------------
   size_type n_edg;
   in >> n_edg;
   _count_geo     [1] = n_edg;
   _count_element [reference_element::e] = n_edg;
   vector<vector<pair<size_type,size_type> > > domain_table;
   for (size_type i = 0; i < n_edg; i++) {

      // -------------------------------------------------------
      // get vertices
      // -------------------------------------------------------
      size_type ns, s[2];
      in >> ns;
      check_macro (ns == 2, "unexpected edge with " << ns << " vertices");
      for (size_type j = 0; j < 2; j++) {
	 in >> s[j];
 	 decr(s[j]);
      }
      // -------------------------------------------------------
      // neighbours
      // -------------------------------------------------------
      size_type nv, v[2];
      in >> nv;
      check_macro (nv == 2, "unexpected edge with " << ns << " neighbours");
      for (size_type j = 0; j < 2; j++) {
	  in >> v[j];
	  decr(v[j]);
      }
      // -------------------------------------------------------
      // get domain id
      // -------------------------------------------------------
      size_type cond;
      in >> cond;
      if (cond == 0) {
	// internal edge
	check_macro (v[0] != undef && v[1] != undef, "boundary edge with cond == 0");
	continue; 
      }
      size_type k = cond-1;
      if (k >= domain_table.size()) {
	 domain_table.resize(k+1);
      }
      if (v[0] != undef) {
          check_macro (v[1] == undef, "internal edge with cond=" << cond << " != 0");
          domain_table[k].push_back (make_pair(s[0],s[1]));
      } else if (v[1] != undef) {
          // reorient the boundary edge in the positive boundary circulation 
	  // i.e. outside at the right of the edge
	  trace_macro ("boundary edge " << i << " = ("<<s[0]<<" "<<s[1]<<"): vertices swapped");
          domain_table[k].push_back (make_pair(s[1],s[0]));
      } else {
	  error_macro ("isolated edge with cond=" << cond << " != 0");
      }
   }
   if (!in.good()) {
       error_macro ("a problem occurs while loading a cemagref mesh");
   }
   // ----------------------------------------------------------
   // load edge domains
   // ----------------------------------------------------------
   // count the number of domains
   size_type n_dom = 0;
   georep::domlist_type::iterator q = _domlist.begin();
   for (size_type k = 0; k < domain_table.size(); k++, q++) {
      if (domain_table[k].size() == 0) continue;
      n_dom++;
   }
   _domlist.resize(n_dom);

   // copy domain_table into domlist
   q = _domlist.begin();
   for (size_type k = 0; k < domain_table.size(); k++) {

      if (domain_table[k].size() == 0) {
	  continue;
      }
      string name = (k < 4 ? domain_name [k] : "unnamed"+itos(k+1));
      (*q).set(domain_table[k].begin(), domain_table[k].size(), name);
      q++;
   }
   // ----------------------------------------------------------
   // load coordinates
   // ----------------------------------------------------------
   _xmin[0] = _xmin[1] =  numeric_limits<Float>::max();
   _xmax[0] = _xmax[1] = -numeric_limits<Float>::max();
   _xmin[2] = _xmax[2] = 0;
   size_type n_vert;
   in >> n_vert;
   _x.resize(n_vert);
   _count_geo     [0] = n_vert;
   _count_element [reference_element::p] = n_vert;
   Float z_dummy;
   vector<Float> z;
   if (load_z) z.resize(n_vert);
   vec<Float>::iterator qz = z.begin();
   georep::iterator_node last_q = _x.end();
   for (georep::iterator_node q = _x.begin(); q != last_q; q++) {
      georep::nodelist_type::reference p = (*q);
      p.get(in, 2);
      if (load_z) in >> *qz++; else in >> z_dummy;
      _xmin[0] = ::min(p[0], _xmin[0]);
      _xmax[0] = ::max(p[0], _xmax[0]);
      _xmin[1] = ::min(p[1], _xmin[1]);
      _xmax[1] = ::max(p[1], _xmax[1]);
   }
   // ----------------------------------------------------------
   // set edge indexes in domains
   // ----------------------------------------------------------
   vector<set<size_type> > ball (n_node());
   build_point_to_element_sets (begin(), end(), ball.begin());
   for (georep::domlist_type::iterator i = _domlist.begin(); i != _domlist.end(); i++) {
        propagate_subgeo_numbering ((*i).begin(), (*i).end(), begin(), ball.begin());
   }
   // ----------------------------------------------------------
   // build z field
   // ----------------------------------------------------------
   if (!load_z) return in;  
   if (zh.size() != n_vert || zh.get_space().get_approx() != "P1") {
#ifdef TODO
       space V(*this, "P1");
       zh = field(V);
#else
       fatal_macro ("get_cemagref no more supported");
#endif // TODO
   }
   for (size_type i = 0; i < z.size(); i++) {
       zh.at(i) = z[i];
   }
   return in;  
}
istream& 
georep::get_cemagref (istream& in)
{
    field z_dummy;
    return get_cemagref (in, z_dummy, false);
}
// =============================================================
// utilities for output in cemagref format 
// =============================================================
static
inline
georep::size_type
get_first (const set<georep::size_type>& x)
{
    return *(x.begin());
}
static
inline
georep::size_type
get_second (const set<georep::size_type>& x)
{
    set<georep::size_type>::const_iterator p = x.begin();
    return *(++p);
}
// =============================================================
// output in cemagref format 
// =============================================================
ostream& 
georep::put_cemagref (ostream& out, const field& z, bool store_z) const
{
   // fortran likes a decimal-point character in floating-point
   std::ios::fmtflags have_showpoint = (out.flags() & std::ios::showpoint);
   out.setf(std::ios::showpoint);

   size_type undef = numeric_limits<size_type>::max();
   may_have_version_2();
   if (store_z) {
      check_macro (z.get_space().get_approx() == "P1",
	"P1 approximation expected, `"<< z.get_space().get_approx()
	<< "' founded");
   }
   // ----------------------------------------------------------
   // output elements
   // ----------------------------------------------------------
   vector<set<size_type> > ball (n_node());
   build_point_to_element_sets (begin(), end(), ball.begin());
   out << size () << endl;
   for (const_iterator i = begin(); i != end(); i++) {

        const geo_element& K = *i;
        size_type idx = K.index();
        // ----------------------------------------------------------
        // output edges and vertices indexes
        // ----------------------------------------------------------
        out << K.n_edge() << endl;
        for (size_type j = 0; j < K.n_edge(); j++) {
           out << add1(K.edge(j)) << " ";
        }
        out << endl
            << K.size() << endl;
        for (size_type j = 0; j < K.size(); j++) {
           out << add1(K[j]) << " ";
        }
        // ----------------------------------------------------------
        // compute neighbours at vertices
        // ----------------------------------------------------------
        vector<size_type> neighbour;

        // neighbour accross the j-th edge [v0,v1]
        for (size_type j0 = 0; j0 < K.size(); j0++) {
	    size_type j1 = (j0+1) % K.size();
	    size_type v0 = K[j0];
	    size_type v1 = K[j1];

	    // c = all elements containing [v0,v1]
	    set<size_type> c;
            set_intersection (ball[v0].begin(), ball[v0].end(), ball[v1].begin(), ball[v1].end(), 
		insert_iterator<set<size_type> > (c, c.begin()));

	    switch (c.size()) {
	      case 1: 
		// edge [v0,v1] is on the boundary
		neighbour.push_back (undef);
	        break;
	      case 2:
		// edge [v0,v1] is internal: between K and the neighbour k1 accross [v0,v1]
		if (get_first(c) != idx) neighbour.push_back(get_first(c));
		else                     neighbour.push_back(get_second(c));
	        break;
	      default:
		error_macro ("edge contained by " << c.size() << " elements");
	    }
	}
	// d = all neighbours arround vertice v0, and that are
        // not neighbours accross an edge [v0,v1] or [vp,v0]
	for (size_type jp = 0; jp < K.size(); jp++) {

	    size_type j0 = (jp+1) % K.size(); // next vertice
	    size_type j1 = (j0+1) % K.size();
	    size_type v0 = K[j0];
	    size_type v1 = K[j1];
	    size_type vp = K[jp];

	    set<size_type> c1;
            set_intersection (ball[v0].begin(), ball[v0].end(), ball[v1].begin(), ball[v1].end(), 
		insert_iterator<set<size_type> > (c1, c1.begin()));

	    set<size_type> cp;
            set_intersection (ball[v0].begin(), ball[v0].end(), ball[vp].begin(), ball[vp].end(), 
		insert_iterator<set<size_type> > (cp, cp.begin()));

	    set<size_type> tmp;
            set_difference (ball[v0].begin(), ball[v0].end(), c1.begin(), c1.end(),
		insert_iterator<set<size_type> > (tmp, tmp.begin()));

	    set<size_type> d;
            set_difference (tmp.begin(), tmp.end(), cp.begin(), cp.end(),
		insert_iterator<set<size_type> > (d, d.begin()));

	    if (d.size() == 0) {
		neighbour.push_back (undef);
	    } else {
		for (set<size_type>::iterator p = d.begin(); p != d.end(); ++p) {
		    neighbour.push_back (*p);
	        }
	    }
        }
        // ----------------------------------------------------------
        // output neighbours at vertices
        // ----------------------------------------------------------
        out << endl
            << neighbour.size() << endl;
        for (size_type j = 0; j < neighbour.size(); j++) {
           out << add1(neighbour[j]) << " ";
        }
        out << endl;
    }
    // ----------------------------------------------------------
    // compute edges 
    // ----------------------------------------------------------
    vector<pair<size_type,size_type> > edge          (n_edge(), make_pair(undef,undef));
    vector<pair<size_type,size_type> > edge_neighbour(n_edge(), make_pair(undef,undef));
    for (const_iterator iter_elt = begin(); iter_elt != end(); iter_elt++) {
        const geo_element& K = *iter_elt;
        for (size_type i = 0; i < K.n_edge(); i++) {
            size_type ie = K.edge (i);
            size_type v0 = K.subgeo_vertex (1, i, 0);
            size_type v1 = K.subgeo_vertex (1, i, 1);
	    if (v0 < v1) {
                edge [ie].first  = v0;
                edge [ie].second = v1;
		edge_neighbour [ie].first = K.index();
	    } else {
                edge [ie].first  = v1;
                edge [ie].second = v0;
		edge_neighbour [ie].second = K.index();
	    }
        }
    }
    // ----------------------------------------------------------
    // compute edge condition identifier (0 : when internal) 
    // ----------------------------------------------------------
    vector<size_type> edge_domain_id (n_edge(), 0);
    size_type new_id = max_cemagref_id+1;
    for (georep::domlist_type::const_iterator i = _domlist.begin(); i != _domlist.end(); i++) {
        const domain& d = *i;
        size_type id = get_cemagref_id (d.name());
	if (id == 0) id = new_id++;
        for (domain::const_iterator j = d.begin(); j != d.end(); j++) {
	    const geo_element& e = *j;
	    edge_domain_id [e.index()] = id;
        }
    }
    // ----------------------------------------------------------
    // output edges 
    // ----------------------------------------------------------
    out << endl
        << n_edge() << endl;
    for (size_type i = 0 ; i < n_edge(); i++) {
       out << "2" << endl
           << add1(edge[i].first) << " " << add1(edge[i].second) << endl
           << "2" << endl
           << add1(edge_neighbour[i].first) << " " << add1(edge_neighbour[i].second) << endl
           << edge_domain_id[i] << endl;
    }
    // ----------------------------------------------------------
    // output vertices and optional cote 
    // ----------------------------------------------------------
    out << endl
        << n_node() << endl;
    size_type idx = 0;
    for (georep::const_iterator_node q = begin_node(); q != end_node(); q++, idx++) {
        const point& p = *q;
	out << p[0] << " " << p[1] << " ";
        if (!store_z) {
	   out << "0" << endl;
        } else {
	   out << z.at(idx) << endl;
        }
    }
    if (!have_showpoint) {
	// restaure context
        out.unsetf(std::ios::showpoint);
    }
    return out;
}
ostream& 
georep::put_cemagref (ostream& out) const
{
    field z_dummy;
    return put_cemagref (out, z_dummy, false);
}

