#ifndef _RHEO_CHARACTERISTIC_H
#define _RHEO_CHARACTERISTIC_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
#include "rheolef/field.h"
namespace rheolef { 

/*Class:characteristic
NAME: @code{characteristic} - discrete mesh advection by a field: gh(x)=fh(x-dt*uh(x))
@cindex  mesh
@cindex  advection
@cindex  method of characteristic
@cindex  quadrature formulae
@findex  riesz_representer
@clindex geomap
@clindex space
@clindex geo

SYNOPSIS:
 The class characteristic is a class implementing the
 Lagrange-Galerkin method. It is the extension of the method
 of characteristic in the finite element context.
 This is an expreimental implementation: please use the "geomap"
 one for practical usage.
EXAMPLE:
@noindent
 The following code compute the Riesz representant, denoted
 by "mgh" of gh(x)=fh(x-dt*uh(x)).
 @example
  geo omega_h;
  field uh = ...;
  field fh = ...;
  characteristic X (omega_h, -dt*uh);
  field mgh = riesz_representer(Vh, compose(fh, X));
 @end example
 The Riesz representer is the "mgh" vector of values:
 @example
	mgh(i) = integrate fh(x-dt*uh(x)) phi_i(x) dx
 @end example
 where phi_i is the i-th basis function in Vh
 and the integral is evaluated by using a quadrature formulae.
 By default the quadrature formule is Gauss-Lobatto with
 the order equal to the polynomial order of Vh.
 This quadrature formulae guaranties inconditional stability
 at any polynomial order (order 1: trapeze, order 2: simpson).
 Extension will accept in the future alternative quadrature 
 formulae.
End: */

//<characteristic:
class characteristic {
public:
	characteristic(const geo& bg_omega, const field& ah);
	const geo& get_background_geo() const;
	const field& get_advection() const;
protected:
	geo   _bg_omega;
	field _ah;
};
class field_o_characteristic {
public:
	field_o_characteristic(const field& fh, const characteristic& X);
	friend field_o_characteristic compose(
		const field& fh, const characteristic& X);
	Float operator() (const point& x) const;
	point vector_evaluate (const point& x) const;
	tensor tensor_evaluate (const point& x) const;
	const field& get_field() const;
	const geo& get_background_geo() const;
	const field& get_advection() const;
protected:
	field          _fh;
	characteristic _X;
	point advect (const point& x0) const;
};
//>characteristic:
inline
characteristic::characteristic (const geo& bg_omega, const field& ah)
 : _bg_omega (bg_omega), _ah (ah) {}

inline
const geo&
characteristic::get_background_geo() const
{
  return _bg_omega;
}
inline
const field&
characteristic::get_advection() const
{
  return _ah;
}
inline
field_o_characteristic::field_o_characteristic(const field& fh, const characteristic& X) : _fh(fh), _X(X) {}

inline
field_o_characteristic
compose(const field& fh, const characteristic& X)
{
  return field_o_characteristic(fh,X);
}
inline
const field&
field_o_characteristic::get_field() const
{
  return _fh;
}
inline
const geo&
field_o_characteristic::get_background_geo() const
{
  return _X.get_background_geo();
}
inline
const field&
field_o_characteristic::get_advection() const
{
  return _X.get_advection();
}
}// namespace rheolef
#endif // _RHEO_CHARACTERISTIC_H
