#ifndef _RHEO_BASIS_H
#define _RHEO_BASIS_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
#include "rheolef/basis_rep.h"
#include "rheolef/smart_pointer.h"

namespace rheolef { 

/*Class:basis
NAME: @code{basis} - polynomial basis
@cindex  polynomial basis
@clindex basis
@cindex  reference element
@clindex reference_element
SYNOPSIS:
  @noindent
  The @code{basis} class defines functions that evaluates a polynomial
  basis and its derivatives on a point. The polynomial basis
  is designated by a string, e.g. "P0", "P1", "P2", "bubble",...
  indicating the basis. The basis depends also of the reference element:
  triangle, square, tetrahedron (@pxref{reference_element iclass}).
  For instance, on a square, the "P1"
  string designates the common Q1 four-nodes basis on the reference square.

  @noindent
  The nodes associated to the Lagrange polynomial basis
  are also available by its associated accessor.

IMPLEMENTATION NOTE:
  @noindent
  The @code{basis} class 
  is a @pxref{smart_pointer iclass}) class on a @code{basis_rep} class that is a pure virtual base class 
  for effective bases, e.g. basis_P1, basis_P1, etc.
AUTHORS:
    LMC-IMAG, 38041 Grenoble cedex 9, France
   | Pierre.Saramito@imag.fr
DATE:   7 january 2004
End:
*/

//<basis:
class basis : public smart_pointer<basis_rep> {
public:

// typedefs:

    typedef size_t size_type;
    typedef basis_rep::dof_family_type dof_family_type;

// allocators:

    basis (std::string name = "");

// accessors:

    std::string name() const;
    size_type   degree() const;
    size_type   size (reference_element hat_K, dof_family_type family=element_constant::dof_family_max) const;
    dof_family_type family() const;

    dof_family_type dof_family(
        reference_element hat_K,
        size_type         i_dof_local) const;

    Float eval(
        reference_element hat_K,
        size_type         i_dof_local,
        const point&      hat_x) const;
    
    point grad_eval(
        reference_element hat_K,
        size_type         i_dof_local,
        const point&      hat_x) const;

    tensor hessian_eval(
        reference_element hat_K,
        size_type         i_dof_local,
        const point&      hat_x) const;

    void eval(
        reference_element    hat_K,
        const point&         hat_x,
	std::vector<Float>&  values) const;
    
    void grad_eval(
        reference_element    hat_K,
        const point&         hat_x,
	std::vector<point>&  values) const;

    void hessian_eval(
        reference_element    hat_K,
        const point&         hat_x,
	std::vector<tensor >&  values) const;

    void hat_node(
        reference_element    hat_K,
	std::vector<point>&  hat_node) const;

    void dump(std::ostream& out = std::cerr) const;
};
//>basis:
// -----------------------------------------------------------
// inlined
// -----------------------------------------------------------
inline
basis::basis(std::string name)
 : smart_pointer<basis_rep> (basis_rep::make_ptr(name))
{
}
inline
std::string
basis::name() const
{
    return data().name();
}
inline
basis::dof_family_type
basis::family() const
{
    return data().family();
}
inline
basis::size_type 
basis::degree() const
{
    return data().degree();
}
inline
basis::size_type
basis::size (reference_element hat_K, dof_family_type family) const
{
    return data().size (hat_K, family);
}
inline
basis::dof_family_type 
basis::dof_family(reference_element hat_K, size_type i_dof_local) const
{
    return data().dof_family(hat_K, i_dof_local);
}
inline
Float 
basis::eval(
        reference_element hat_K,
        size_type         i_dof_local,
        const point&      hat_x) const
{
    return data().eval (hat_K, i_dof_local, hat_x);
}
inline
point 
basis::grad_eval(
        reference_element hat_K,
        size_type         i_dof_local,
        const point&      hat_x) const
{
    return data().grad_eval (hat_K, i_dof_local, hat_x);
}
inline
tensor  
basis::hessian_eval(
        reference_element hat_K,
        size_type         i_dof_local,
        const point&      hat_x) const
{
    return data().hessian_eval (hat_K, i_dof_local, hat_x);
}
inline
void
basis::eval(
        reference_element    hat_K,
        const point&         hat_x,
	std::vector<Float>&  values) const
{
    return data().eval (hat_K, hat_x, values);
}
inline
void
basis::grad_eval(
        reference_element    hat_K,
        const point&         hat_x,
	std::vector<point>&  values) const
{
    return data().grad_eval (hat_K, hat_x, values);
}inline
void
basis::hessian_eval(
        reference_element    hat_K,
        const point&         hat_x,
	std::vector<tensor >&  values) const
{
    return data().hessian_eval (hat_K, hat_x, values);
}
inline
void
basis::hat_node(
        reference_element    hat_K,
	std::vector<point>&  hat_node) const
{
    return data().hat_node (hat_K, hat_node);
}
inline
void
basis::dump(std::ostream& out) const
{
    out << "basis " << name() << std::endl;
}
}// namespace rheolef
#endif // _RHEO_BASIS_H
