///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "rheolef/basis.h"
#include "arma2csr.icc"

namespace rheolef {
using namespace std;

// =========================================================================
// basis members
// =========================================================================
// naming scheme for standard FEM families
template <class T>
std::string
basis_rep<T>::standard_naming (std::string family_name, size_t degree, const basis_option& sopt)
{
  return family_name
       + itos(degree)
       + (sopt.is_continuous() ? "" : "d")
       + sopt.stamp();
}
template <class T>
std::string
basis_rep<T>::name() const
{
  // this is a default behavior for general classes with degree as parameter
  // this function is virtual and redefined by "bubble" and all specific classes
  // without degree as a parameter
  return standard_naming (family_name(), degree(), option());
}
// scalar-valued
template <class T>
void
basis_rep<T>::eval(
    reference_element                    hat_K,
    const std::vector<point_basic<T> >&  node,
    arma::Mat<T>&                        value) const
{
  // value(i,j) := psi_j (node(i))
  arma::Col<T>& phi = _phi [hat_K.variant()];
  if (phi.size() == 0) {
    phi.resize (size(hat_K));
  }
  value.resize (node.size(), phi.size());
  for (size_type loc_inod = 0, loc_nnod = node.size(); loc_inod < loc_nnod; ++loc_inod) {
    eval (hat_K, node[loc_inod], phi);
    value (loc_inod,arma::span::all) = trans(phi); // TODO: how to avoid copies here ? template & virtual eval...
  }
}
template <class T>
void
basis_rep<T>::eval(
    reference_element                    hat_K,
    const std::vector<point_basic<T> >&  node,
    arma::Col<T>&                        value) const
{
  // value(i*ndof+j) := psi_j (node(i))
  arma::Col<T>& phi = _phi [hat_K.variant()];
  if (phi.size() == 0) {
    phi.resize (size(hat_K));
  }
  value.resize (node.size()*phi.size());
  for (size_type loc_inod = 0, loc_nnod = node.size(); loc_inod < loc_nnod; ++loc_inod) {
    eval (hat_K, node[loc_inod], phi);
#ifdef TODO
    value (arma::span(loc_inod*phi.size(), (loc_inod+1)*phi.size()-1)) = phi; // TODO: how to avoid copies here ? template & virtual eval...
#endif // TODO
    for (size_type loc_idof = 0, loc_ndof = phi.size(); loc_idof < loc_ndof; ++loc_idof) {
      size_type iloc = loc_inod*loc_ndof+loc_idof;
      value [iloc] = phi[loc_idof]; // TODO: how to avoid copies here ? template & virtual eval for one node...
    }
  }
}
// vector-valued
template <class T>
void
basis_rep<T>::eval(
    reference_element                    hat_K,
    const std::vector<point_basic<T> >&  node,
    ublas::matrix<point_basic<T> >&      value) const
{
  // value(i,j) := psi_j (node(i))
  std::vector<point_basic<T> >& phi = _v_phi [hat_K.variant()];
  if (phi.size() == 0) {
    phi.resize (size(hat_K));
  }
  value.resize (node.size(), phi.size());
  for (size_type loc_inod = 0, loc_nnod = node.size(); loc_inod < loc_nnod; ++loc_inod) {
    eval (hat_K, node[loc_inod], phi);
    for (size_type loc_idof = 0, loc_ndof = phi.size(); loc_idof < loc_ndof; ++loc_idof) {
      value (loc_inod,loc_idof) = phi[loc_idof]; // TODO: how to avoid copies here ? template & virtual eval for one node...
    }
  }
}
template <class T>
void
basis_rep<T>::grad_eval(
    reference_element                    hat_K,
    const std::vector<point_basic<T> >&  node,
    ublas::matrix<point_basic<T> >&      value) const  // scalar-valued grad
{
  // value(i,j) := grad_psi_j (node(i))
  std::vector<point_basic<T> >& grad_phi = _v_phi [hat_K.variant()];
  if (grad_phi.size() == 0) {
    grad_phi.resize (size(hat_K));
  }
  value.resize (node.size(), grad_phi.size());
  for (size_type loc_inod = 0, loc_nnod = node.size(); loc_inod < loc_nnod; ++loc_inod) {
    grad_eval (hat_K, node[loc_inod], grad_phi);
    for (size_type loc_idof = 0, loc_ndof = grad_phi.size(); loc_idof < loc_ndof; ++loc_idof) {
      value (loc_inod,loc_idof) = grad_phi[loc_idof]; // TODO: how to avoid copies here ? template & virtual eval for one node...
    }
  }
}
template <class T>
void
basis_rep<T>::grad_eval(
    reference_element                    hat_K,
    const std::vector<point_basic<T> >&  node,
    std::vector<point_basic<T> >&        value) const  // scalar-valued grad
{
  // value(i*ndof+j) := grad_psi_j (node(i))
  std::vector<point_basic<T> >& grad_phi = _v_phi [hat_K.variant()];
  if (grad_phi.size() == 0) {
    grad_phi.resize (size(hat_K));
  }
  value.resize (node.size()*grad_phi.size());
  for (size_type loc_inod = 0, loc_nnod = node.size(); loc_inod < loc_nnod; ++loc_inod) {
    grad_eval (hat_K, node[loc_inod], grad_phi);
    for (size_type loc_idof = 0, loc_ndof = grad_phi.size(); loc_idof < loc_ndof; ++loc_idof) {
      size_type iloc = loc_inod*loc_ndof+loc_idof;
      value [iloc] = grad_phi[loc_idof]; // TODO: how to avoid copies here ? template & virtual eval for one node...
    }
  }
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
#define _RHEOLEF_instanciation(T)                                             	\
template std::string basis_rep<T>::name() const;				\
template void basis_rep<T>::eval (						\
    reference_element, 								\
    const std::vector<point_basic<T> >&, 					\
    arma::Mat<T>&) const;							\
template void basis_rep<T>::eval(						\
    reference_element                    hat_K,					\
    const std::vector<point_basic<T> >&  node,					\
    arma::Col<T>&                        value) const;				\
template void basis_rep<T>::eval(						\
    reference_element,								\
    const std::vector<point_basic<T> >&,					\
    ublas::matrix<point_basic<T> >&) const;					\
template void basis_rep<T>::grad_eval(						\
    reference_element                    hat_K,					\
    const std::vector<point_basic<T> >&  node,					\
    std::vector<point_basic<T> >&        value) const;				\
template void basis_rep<T>::grad_eval(						\
    reference_element,								\
    const std::vector<point_basic<T> >&,					\
    ublas::matrix<point_basic<T> >&) const;					\
template std::string basis_rep<T>::standard_naming (std::string family_name, 	\
	size_t degree, const basis_option& sopt);

_RHEOLEF_instanciation(Float)

}// namespace rheolef
