///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "basis_fem_Pk_bernstein.h"
#include "rheolef/rheostream.h"
#include "equispaced.icc"
#include "warburton.icc"

namespace rheolef {
using namespace std;

// =========================================================================
// basis members
// =========================================================================
template<class T>
basis_fem_Pk_bernstein<T>::~basis_fem_Pk_bernstein()
{
}
template<class T>
basis_fem_Pk_bernstein<T>::basis_fem_Pk_bernstein (
  std::string              name, 
  const basis_option& sopt) 
  : basis_rep<T> (name,sopt),
    _raw_basis(name)
{
  // Bezier nodes are equispaced => forced
  base::_sopt.set_node(basis_option::equispaced);
}
template<class T>
void
basis_fem_Pk_bernstein<T>::_initialize (reference_element hat_K) const
{
#ifdef TO_CLEAN
  bool inv_ok = build_vdm (
    _raw_basis,
    hat_K,
    base::option(),
    base::_hat_node[hat_K.variant()],
    base::_raw_value[hat_K.variant()],
    base::_vdm[hat_K.variant()],
    base::_inv_vdm[hat_K.variant()]);
  check_macro (inv_ok, "unisolvence failed for "
	<< base::name() <<"(" << hat_K.name() << ") basis");
#endif // TO_CLEAN

  // initialization is similar to Pk-Lagrange
  size_type k = degree();
  size_type variant = hat_K.variant();

  // first_idof, first_inod:
  base::_first_idof [variant].fill (0);
  if (k == 0) {
    base::_first_idof [variant][hat_K.dimension()+1] = 1;
  } else {
    switch (variant) {
      case reference_element::p:
        base::_first_idof [variant][1] = 1;
        break;
      case reference_element::e:
        base::_first_idof [variant][1] = 2;
        base::_first_idof [variant][2] = k+1;
        break;
      case reference_element::t:
        base::_first_idof [variant][1] = 3;
        base::_first_idof [variant][2] = 3*k;
        base::_first_idof [variant][3] = (k+1)*(k+2)/2;
        break;
      case reference_element::q:
        base::_first_idof [variant][1] = 4;
        base::_first_idof [variant][2] = 4*k;
        base::_first_idof [variant][3] = (k+1)*(k+1);
        break;
      case reference_element::T:
        base::_first_idof [variant][1] = 4;
        base::_first_idof [variant][2] = 6*(k-1) + 4;
        base::_first_idof [variant][3] = 4*(k-1)*(k-2)/2 + base::_first_idof [variant][2];
        base::_first_idof [variant][4] = (k+1)*(k+2)*(k+3)/6;
        break;
      case reference_element::P:
        base::_first_idof [variant][1] = 6;
        base::_first_idof [variant][2] = 9*(k-1) + 6;
        base::_first_idof [variant][3] = 2*(k-1)*(k-2)/2 + 3*(k-1)*(k-1) + base::_first_idof [variant][2];
        base::_first_idof [variant][4] = (k+1)*(k+1)*(k+2)/2;
        break;
      case reference_element::H:
        base::_first_idof [variant][1] = 8;
        base::_first_idof [variant][2] = 12*(k-1) + 8;
        base::_first_idof [variant][3] = 6*(k-1)*(k-1) +  base::_first_idof [variant][2];
        base::_first_idof [variant][4] = (k+1)*(k+1)*(k+1);
        break;
      default: error_macro ("unexpected element variant `"<<variant<<"'");
    }
  }
  base::_first_inod [variant] = base::_first_idof [variant];

  // nodes:
  switch (base::_sopt.get_node()) {
    case basis_option::equispaced:
          pointset_lagrange_equispaced (hat_K, k, base::_hat_node[variant]);
          break;
    case basis_option::warburton:
          pointset_lagrange_warburton  (hat_K, k, base::_hat_node[variant]); break;
    default: error_macro ("unsupported node set: "<<base::_sopt.get_node_name());
  }
  // vdm:
  _raw_basis.eval (hat_K, base::_hat_node[variant], base::_vdm[variant]);
  size_type loc_ndof = base::size(hat_K);
  base::_inv_vdm[variant].resize (loc_ndof,loc_ndof);
  arma::Mat<T> identity (loc_ndof,loc_ndof);
  identity.eye();
  bool inv_ok = solve (base::_inv_vdm[variant], base::_vdm[variant], identity);
  check_macro (inv_ok, "unisolvence failed for "
	<< base::name() <<"(" << hat_K.name() << ") basis");
}
// evaluation of all basis functions at hat_x:
template<class T>
void
basis_fem_Pk_bernstein<T>::eval (
    reference_element                 hat_K,
    const point_basic<T>&             hat_x,
    arma::Col<T>&                     value) const
{
  base::_initialize_guard (hat_K);
  _raw_basis.eval (hat_K, hat_x, value);
}
// evaluate the gradient:
template<class T>
void
basis_fem_Pk_bernstein<T>::grad_eval (
  reference_element           hat_K,
  const point_basic<T>&       hat_x,
  std::vector<point_basic<T> >& value) const 
{
  base::_initialize_guard (hat_K);
  _raw_basis.grad_eval (hat_K, hat_x, value);
}
// dofs for a scalar-valued function
template<class T>
void
basis_fem_Pk_bernstein<T>::_compute_dofs (
  reference_element     hat_K,
  const arma::Col<T>&   f_xnod, 
        arma::Col<T>&   dof) const
{
  base::_initialize_guard (hat_K);
  dof = base::_inv_vdm[hat_K.variant()]*f_xnod;
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
#define _RHEOLEF_instanciation(T)                                             	\
template class basis_fem_Pk_bernstein<T>;

_RHEOLEF_instanciation(Float)

}// namespace rheolef
