/* deurlname.c - Remove URL-encoded characters from file names
 *
 * Copyright (C) 2001-2004 Oskar Liljeblad
 *
 * This file is part of the file renaming utilities (renameutils).
 *
 * This software is copyrighted work licensed under the terms of the
 * GNU General Public License. Please consult the file `COPYING' for
 * details.
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif
/* gettext */
#include <gettext.h> /* will include <libintl.h> if ENABLE_NLS */
#define _(String) gettext(String)
/* C89 */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <locale.h>
/* gnulib */
#include <progname.h>
/* POSIX/gnulib */
#include <getopt.h>
#include <stdbool.h>
/* common */
#include "common/error.h"
#include "common/strbuf.h"

#define PROGRAM "deurlname"

enum {
  VERSION_OPT = 256,
  HELP_OPT
};
static struct option option_table[] = {
    { "verbose", no_argument, NULL, 'v' },
    { "version", no_argument, NULL, VERSION_OPT },
    { "help",    no_argument, NULL, HELP_OPT    },
    { 0 }
};


static inline int
hexvalue(char c)
{
    if (c >= '0' && c <= '9')
    	return c-'0';
    if (c >= 'A' && c <= 'F')
    	return c-'A'+10;
    return c-'a'+10;
}

int
main(int argc, char **argv)
{
    char *newname;
    bool verbose = false;
    int c;
    int rc = 0;

    set_program_name(argv[0]);

    if (setlocale (LC_ALL, "") == NULL)
    	die(_("invalid locale"));
    if (bindtextdomain (PACKAGE, LOCALEDIR) == NULL)
    	die_errno(NULL);
    if (textdomain (PACKAGE) == NULL)
    	die_errno(NULL);

    while (true) {
    	c = getopt_long (argc, argv, "v", option_table, NULL);
        if (c == -1)
            break;

        switch (c) {
        case '?':
            exit(1);
        case 'v': /* --verbose */
            verbose = true;
            break;
        case VERSION_OPT:
            printf(_("%s (%s) %s\n\
Written by Oskar Liljeblad.\n\
\n\
Copyright (C) 2001-2004 Oskar Liljeblad.\n\
This is free software; see the source for copying conditions.  There is NO\n\
warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n"),
                PROGRAM, PACKAGE, VERSION);
  	    exit(0);
        case HELP_OPT:
	    printf(_("Usage: %s [OPTION]... [FILE]...\n\
Remove URL-encoded characters from file names.\n\
\n\
Options:\n\
  -v, --verbose              display every rename done\n\
      --help                 display this help and exit\n\
      --version              output version information and exit\n\
\n\
The encoded slash character (%%2f) is left untouched in file names.\n\
\n\
Report bugs to <%s>.\n"), program_name, PACKAGE_BUGREPORT);
  	    exit(0);
        }
    }

    if (optind >= argc)
    	die(_("missing file argument"));

    newname = strbuf_new();
    for (c = optind; c < argc; c++) {
    	char *name;
	char *basename;
	int d;

    	name = argv[c];
    	strbuf_clear(&newname);
	basename = strrchr(name, '/');
	d = 0;
	if (basename != NULL) {
	    strbuf_append_substring(&newname, name, 0, basename-name+1);
	    d = basename-name+1;
	}
	for (; name[d] != '\0'; d++) {
    	    if (name[d] == '%' && isxdigit(name[d+1]) && isxdigit(name[d+2])) {
	    	char newch = (hexvalue(name[d+1]) << 4) | hexvalue(name[d+2]);
	    	if (newch != '\0' && newch != '/') {
		    strbuf_append_char(&newname, newch);
	    	    d += 2;
	    	    continue;
                }
	    }
	    strbuf_append_char(&newname, name[d]);
	}

	if (strcmp(name, newname) != 0) {
	    if (verbose)
	        printf("%s => %s\n", name, newname);
	    if (rename(name, newname) < 0) {
	        warn_errno(_("cannot rename `%s'"), name);
                rc = 1;
            }
	}
    }

    strbuf_free(&newname);
    exit(rc);
}
