/***************************************************************************
  grain.cpp
  -------------------
  A grain class
  -------------------
  Copyright 1999-2006, David Johnson
  Please see the header file for copyright and license information
 ***************************************************************************/

#include "resource.h"
#include "grain.h"

using namespace Resource;

//////////////////////////////////////////////////////////////////////////////
// Grain()
// -------
// Constructor

Grain::Grain()
    : name_("Generic"), weight_(), extract_(0.0), color_(0.0),
      use_(USE_OTHER), yield_(0.0), hcu_(0.0)
{ ; }

Grain::Grain(const QString &name, const Weight &weight, const double &extract,
             const double &color, const QString &use)
    : name_(name), weight_(weight), extract_(extract), color_(color),
      use_(useStringToEnum(use))
{
    recalc();
}

Grain::~Grain() { ; }

//////////////////////////////////////////////////////////////////////////////
// Grain(const Grain&)
// --------------------
// Copy Constructor

Grain::Grain(const Grain &g)
    : name_(g.name_), weight_(g.weight_), extract_(g.extract_),
      color_(g.color_), use_(g.use_), yield_(g.yield_), hcu_(g.hcu_)
{ ; }

//////////////////////////////////////////////////////////////////////////////
// operator=()
// -----------
// Assignment operator

Grain Grain::operator=(const Grain &g)
{
    if (this != &g) {
        name_ = g.name_;
        weight_ = g.weight_;
        extract_ = g.extract_;
        color_ = g.color_;
        use_ = g.use_;
        yield_ = g.yield_;
        hcu_ = g.hcu_;
    }
    return *this;
}

//////////////////////////////////////////////////////////////////////////////
// operator==()
// ------------
// Equivalence operator

bool Grain::operator==(const Grain &g) const
{
    return (name_ == g.name_)
        && (weight_ == g.weight_)
        && (extract_ == g.extract_)
        && (color_ == g.color_)
        && (use_ == g.use_);
}

///////////////////////////////////////////////////////////////////////////////
// useString()
// -----------
// Return usage as a string

const QString &Grain::useString() const
{
    switch (use_) {
        case USE_EXTRACT: return GRAIN_EXTRACT;
        case USE_MASHED:  return GRAIN_MASHED;
        case USE_STEEPED: return GRAIN_STEEPED;
    }
    return GRAIN_OTHER;
}

///////////////////////////////////////////////////////////////////////////////
// useStringList()
// -----------
// Return string list of usage types

QStringList Grain::useStringList()
{
    return (QStringList()
            << GRAIN_EXTRACT
            << GRAIN_MASHED
            << GRAIN_STEEPED
            << GRAIN_OTHER);
}

///////////////////////////////////////////////////////////////////////////////
// useStringToEnum()
// -----------------
// Convert use string to enumeration type

int Grain::useStringToEnum(const QString &name)
{
    if ((name.toLower() == GRAIN_MASHED) || (name.toLower() == "mash"))
        return USE_MASHED;
    if ((name.toLower() == GRAIN_STEEPED) || (name.toLower() == "steep"))
        return USE_STEEPED;
    if (name.toLower() == GRAIN_EXTRACT)
        return USE_EXTRACT;
    return USE_OTHER;
}

///////////////////////////////////////////////////////////////////////////////
// recalc()
// --------
// Recalculate internal values

void Grain::recalc()
{
    yield_ = weight_.amount(Weight::pound) * (extract_ - 1.000);
    hcu_ = weight_.amount(Weight::pound) * color_;
}
