/*
 * PulseAudio plugin driver for FAUmachine
 *
 * Derived from MPlayer (libao2 - ao_polyp).
 *
 * Copyright (c) FAUmachine Team.
 * Copyright (c) MPlayer Team.
 * Copyright (c) Lennart Poettering.
 *
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <stdio.h>
#include <dlfcn.h>
#include <pulse/pulseaudio.h>

#include "config.h"

#include "glue-audio_internal.h"

#define	PULSE_CLIENT_NAME "FAUmachine"

/** General driver info */
static ao_info_t info = {
	"Pulseaudio audio output",
	"pulse",
	"Lennart Poettering",
	""
};

LIBAO_EXTERN(pulse);

/* pointers to functions to load from libpulse.so */
int    		(*p_sample_spec_valid) (const pa_sample_spec *ss);
size_t		(*p_bytes_per_second) (const pa_sample_spec *spec);
int		(*p_context_is_pending) (pa_context *c);
pa_context * 	(*p_context_new) (pa_mainloop_api *mainloop, const char *name);
int		(*p_context_connect) (pa_context *c, const char *server,
				pa_context_flags_t flags, const pa_spawn_api *api);
pa_context_state_t (*p_context_get_state) (pa_context *c);
void		(*p_context_unref) (pa_context *c);
int		(*p_context_errno) (pa_context *c);
int		(*p_mainloop_iterate) (pa_mainloop *m, int block, int *retval);
pa_mainloop_api * (*p_mainloop_get_api) (pa_mainloop*m);
pa_mainloop *	(*p_mainloop_new) (void);
void		(*p_mainloop_free) (pa_mainloop* m);
pa_operation_state_t (*p_operation_get_state) (pa_operation *o);
void		(*p_operation_unref) (pa_operation *o);
void		(*p_operation_cancel) (pa_operation *o);
pa_stream *	(*p_stream_new) (pa_context *c, const char *name,
				const pa_sample_spec *ss, const pa_channel_map *map);
int		(*p_stream_connect_playback) (pa_stream *s, const char *dev,
				const pa_buffer_attr *attr, pa_stream_flags_t flags,
				pa_cvolume *volume, pa_stream *sync_stream);
pa_stream_state_t (*p_stream_get_state) (pa_stream *p);
pa_operation *	(*p_stream_drain) (pa_stream *s, pa_stream_success_cb_t cb, void *userdata);
void		(*p_stream_unref) (pa_stream *s);
pa_operation *	(*p_stream_trigger) (pa_stream *s, pa_stream_success_cb_t cb, void *userdata);
int		(*p_stream_write) (pa_stream *p, const void *data, size_t length,
				pa_free_cb_t free_cb, int64_t offset, pa_seek_mode_t seek);
size_t		(*p_stream_writable_size) (pa_stream *p);
int		(*p_stream_get_latency) (pa_stream *s, pa_usec_t *r_usec, int *negative);

/** List of symbols and destinations for their pointers to load */
static struct {
	const char * name;
	void * funcpp;
} pulse_array[] = {
	{ "pa_sample_spec_valid" , 	&p_sample_spec_valid },
	{ "pa_bytes_per_second", 	&p_bytes_per_second },
	{ "pa_context_is_pending", 	&p_context_is_pending },
	{ "pa_context_new", 		&p_context_new },
	{ "pa_context_connect", 	&p_context_connect },
	{ "pa_context_get_state", 	&p_context_get_state },
	{ "pa_context_unref", 		&p_context_unref },
	{ "pa_context_errno", 		&p_context_errno },
	{ "pa_mainloop_iterate", 	&p_mainloop_iterate },
	{ "pa_mainloop_new", 		&p_mainloop_new },
	{ "pa_mainloop_get_api", 	&p_mainloop_get_api },
	{ "pa_mainloop_free", 		&p_mainloop_free },
	{ "pa_operation_get_state", 	&p_operation_get_state },
	{ "pa_operation_unref", 	&p_operation_unref },
	{ "pa_operation_cancel", 	&p_operation_cancel },
	{ "pa_stream_new", 		&p_stream_new },
	{ "pa_stream_connect_playback", &p_stream_connect_playback },
	{ "pa_stream_get_state", 	&p_stream_get_state },
	{ "pa_stream_drain", 		&p_stream_drain },
	{ "pa_stream_unref", 		&p_stream_unref },
	{ "pa_stream_trigger", 		&p_stream_trigger },
	{ "pa_stream_write", 		&p_stream_write },
	{ "pa_stream_writable_size", 	&p_stream_writable_size },
	{ "pa_stream_get_latency", 	&p_stream_get_latency },
	{ NULL, NULL }
};

/** Handle to library obtained by dlopen */
static void * pulse_lib_handle = NULL;

/** Pulseaudio playback stream object */
static struct pa_stream *stream = NULL;

/** Pulseaudio connection context */
static struct pa_context *context = NULL;

/** Main event loop object */
static struct pa_mainloop *mainloop = NULL;

/** Check for errors on the connection context or stream */
static int check_error() {
	if(p_context_get_state(context) == PA_CONTEXT_FAILED)
		return 1;

	if(stream) {
		if(p_stream_get_state(stream) == PA_STREAM_FAILED)
			return 1;
	}
	return 0;
}

/** Wait until no further actions are pending on the connection context */
static void wait_for_completion() {
	while (p_context_is_pending(context)) {
		if(p_mainloop_iterate(mainloop, 1, NULL) < 0) {
			return;
		}
		if(check_error())
			return;
	}
}

/** Make sure that the connection context doesn't starve to death */
static void keep_alive(void) {
	while (p_mainloop_iterate(mainloop, 0, NULL) > 0);
}

/** Wait until the specified operation completes */
static void wait_for_operation(struct pa_operation *o) {
	while (p_operation_get_state(o) == PA_OPERATION_RUNNING) {
		if(p_mainloop_iterate(mainloop, 1, NULL) < 0) {
			goto fail;
		}
		if(check_error()) {
			goto fail;
		}
	}
	p_operation_unref(o);
	return;
fail:
	p_operation_cancel(o);
	p_operation_unref(o);
}

/** initialization function */
static int init() {
	struct pa_sample_spec ss;
	struct pa_buffer_attr a;
	int i;

	pulse_lib_handle = dlopen("libpulse.so.0", RTLD_NOW | RTLD_GLOBAL | RTLD_NODELETE);
	if (pulse_lib_handle == NULL) {
		return 0;
	}

	for (i=0; pulse_array[i].name; i++) {
		dlerror();
		*(void**) pulse_array[i].funcpp = dlsym(pulse_lib_handle,pulse_array[i].name);
		if (dlerror() != NULL) {
			return 0;
		}
	}

	ss.channels = GLUE_AUDIO_CHANNELS;
	ss.rate = GLUE_AUDIO_RATE;

#if GLUE_AUDIO_FORMAT == AFMT_S16_LE
	ss.format = PA_SAMPLE_S16LE;
#elif GLUE_AUDIO_FORMAT == AFMT_S16_BE
	ss.format = PA_SAMPLE_S16BE;
#else
#error unsupported format
#endif

	if (!p_sample_spec_valid(&ss)) {
		goto fail;
	}

	mainloop = p_mainloop_new();

	context = p_context_new(p_mainloop_get_api(mainloop), PULSE_CLIENT_NAME);

	p_context_connect(context, NULL, PA_CONTEXT_NOAUTOSPAWN, NULL);

	wait_for_completion();

	if (p_context_get_state(context) != PA_CONTEXT_READY) {
		goto fail;
	}

	stream = p_stream_new(context, "audio stream", &ss, NULL);

	a.maxlength = p_bytes_per_second(&ss)*1;
	a.tlength = a.maxlength*9/10;
	a.prebuf = a.tlength/2;
	a.minreq = a.tlength/10;

	p_stream_connect_playback(stream, NULL, &a, PA_STREAM_INTERPOLATE_TIMING, NULL, NULL);

	wait_for_completion();

	if (p_stream_get_state(stream) != PA_STREAM_READY) {
		goto fail;
	}

	return 1;

fail:
	uninit(1);
	return 0;
}

/** Destroy libao driver */
static void uninit(int immed) {
	if (stream) {
		if (!immed && p_stream_get_state(stream) == PA_STREAM_READY)
			wait_for_operation(p_stream_drain(stream, NULL, NULL));

		p_stream_unref(stream);
		stream = NULL;
	}

	if (context) {
		p_context_unref(context);
		context = NULL;
	}

	if (mainloop) {
		p_mainloop_free(mainloop);
		mainloop = NULL;
	}
}

/** Play the specified data to the polypaudio server */
static int play(void* data, int len) {

	if (p_stream_get_state(stream) != PA_STREAM_READY)
		return -1;

	if (!len)
		wait_for_operation(p_stream_trigger(stream, NULL, NULL));
	else
		p_stream_write(stream, data, len, NULL, 0, PA_SEEK_RELATIVE);

	wait_for_completion();

	if (p_stream_get_state(stream) != PA_STREAM_READY)
		return -1;

	return len;
}

/** Return number of bytes that may be written to the server without blocking */
static int get_space(void) {
	uint32_t l;

	keep_alive();

	l = p_stream_writable_size(stream);

	return l;
}

/** Return the current latency in seconds */
static float get_delay(void) {
	pa_usec_t latency;
	int negative;

	p_stream_get_latency(stream, &latency, &negative);

	return (float) latency/1000000;
}
