/*
*
* cobex_core.c - Core functions for OBEX-over-cable 
*
* Copyright (c) 2003,2004,2005,2006 Fredrik Srensson
*
* History:
* v0.2   - fsn - 03-08-24 - First version
* v0.3   - fsn - 04-07-07 - Fixes, clean up
* v0.4   - fsn - 04-07-08 - Added header parser
* v0.4.1 - fsn - 05-01-31 - Renamed everything
* v0.5   - fsn - 06-09-24 - National Character support
* Source:
*
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <langinfo.h>
#include <locale.h>
#include <iconv.h>

#include "cobex_core.h"

// Building packets, OPcodes

int obex_opcode_connect ( obex_packet *inPacket ) {
	inPacket->l=0;

	if ( inPacket->max < 7 ) { return COBEX_ERR_BUFFER_OVERFLOW; }

	inPacket->buffer[inPacket->l++] = OBEX_OPCODE_CONNECT;
	inPacket->buffer[inPacket->l++] = 0x00;
	inPacket->buffer[inPacket->l++] = 0x00;
	inPacket->buffer[inPacket->l++] = 0x10;
	inPacket->buffer[inPacket->l++] = 0x00;
	inPacket->buffer[inPacket->l++] = (u_char)((inPacket->max>>8) & 0xff);
	inPacket->buffer[inPacket->l++] = (u_char)(inPacket->max & 0xff);

	return COBEX_OK;
}

int obex_opcode_disconnect ( obex_packet *inPacket ) {
	inPacket->l=0;

	if ( inPacket->max < 3 ) { return COBEX_ERR_BUFFER_OVERFLOW; }

	inPacket->buffer[inPacket->l++] = OBEX_OPCODE_DISCONNECT;
	inPacket->buffer[inPacket->l++] = 0x00;
	inPacket->buffer[inPacket->l++] = 0x00;

	return COBEX_OK;
}

int obex_opcode_put (obex_packet *inPacket) {
	inPacket->l = 0;

	if (inPacket->max < 3) { return COBEX_ERR_BUFFER_OVERFLOW; }

	inPacket->buffer[inPacket->l++] = OBEX_OPCODE_PUT;
	inPacket->buffer[inPacket->l++] = 0x00;
	inPacket->buffer[inPacket->l++] = 0x00;

	return COBEX_OK;
}

int obex_opcode_abort (obex_packet *inPacket) {
	inPacket->l = 0;

	if (inPacket->max < 3) { return COBEX_ERR_BUFFER_OVERFLOW; }

	inPacket->buffer[inPacket->l++] = OBEX_OPCODE_ABORT;
	inPacket->buffer[inPacket->l++] = 0x00;
	inPacket->buffer[inPacket->l++] = 0x00;

	return COBEX_OK;
}

int obex_opcode_setpath (obex_packet *inPacket, char flags) {
	inPacket->l = 0;

	if (inPacket->max < 5) { return COBEX_ERR_BUFFER_OVERFLOW; }

	inPacket->buffer[inPacket->l++] = OBEX_OPCODE_SETPATH;
	inPacket->buffer[inPacket->l++] = 0x00;
	inPacket->buffer[inPacket->l++] = 0x00;
	inPacket->buffer[inPacket->l++] = flags;
	inPacket->buffer[inPacket->l++] = 0x00;

	return COBEX_OK;
}

int obex_opcode_get (obex_packet *inPacket) {
	inPacket->l = 0;

	if (inPacket->max < 3) { return COBEX_ERR_BUFFER_OVERFLOW; }

	inPacket->buffer[inPacket->l++] = OBEX_OPCODE_GET;
	inPacket->buffer[inPacket->l++] = 0x00;
	inPacket->buffer[inPacket->l++] = 0x00;

	return COBEX_OK;
}

//
// Building packets, Headers
//
// lgt is how long part of msg we should copy, add one for the null termination.

int obex_hi_name (obex_packet *inPacket, char *msg, int lgt) {
	int i;

	if (inPacket->max < (inPacket->l + (lgt*2) +3) ) { return COBEX_ERR_BUFFER_OVERFLOW; }
	inPacket->buffer[inPacket->l++] = OBEX_HI_NAME;
	inPacket->buffer[inPacket->l++] = (u_char)(((2*lgt+3)>>8) & 0xff);
	inPacket->buffer[inPacket->l++] = (u_char) ((2*lgt+3)     & 0xff);

	for (i=0; i < lgt; i++) {
		inPacket->buffer[inPacket->l++] = 0x00;
		inPacket->buffer[inPacket->l++] = msg[i];
	}
	return COBEX_OK;
}

// same as above, but you'll have to fix the msg-string yourself, ie. you make the
// unicode. You should use this one, above is for compatability with cobex < 0.2.2 

int obex_hi_name2 (obex_packet *inPacket, char *msg, int lgt) {
	int i;

	if (inPacket->max < (inPacket->l + lgt +3) ) { return COBEX_ERR_BUFFER_OVERFLOW; }
	inPacket->buffer[inPacket->l++] = OBEX_HI_NAME;
	inPacket->buffer[inPacket->l++] = (u_char)(((lgt+3)>>8) & 0xff);
	inPacket->buffer[inPacket->l++] = (u_char) ((lgt+3)     & 0xff);
	for (i=0; i < lgt; i++) {
		inPacket->buffer[inPacket->l++] = msg[i];
	}
	return COBEX_OK;
}

// same as above, but it will handle strings in whatever encoding you throw at it,
// as long as the information from nl_langinfo is correct
// You should really use this one, above is for compatability with cobex < 0.2.2 and cobex < 0.2.10 

int obex_hi_name3 (obex_packet *inPacket, char *msg, int lgt) {
	
	int i;
	int convLgt;
	u_int inBytes = lgt;
	u_int outBytes = MAXCHARS;
	char outOrigMsg[MAXCHARS];
		
	char *inMsgP = msg;
	char *outMsgP = outOrigMsg;

	iconv_t iconvHandle;
	size_t sizeHandle;
	
	setlocale(LC_CTYPE, "");
	
	iconvHandle = iconv_open("UTF-16BE" , nl_langinfo(CODESET) );
	sizeHandle = iconv(iconvHandle, &inMsgP, &inBytes, &outMsgP, &outBytes);
	convLgt = MAXCHARS-outBytes;
	iconv_close(iconvHandle);
		
	if (inPacket->max < (inPacket->l + convLgt +3) ) { return COBEX_ERR_BUFFER_OVERFLOW; }
	inPacket->buffer[inPacket->l++] = OBEX_HI_NAME;
	inPacket->buffer[inPacket->l++] = (u_char)(((convLgt+3)>>8) & 0xff);
	inPacket->buffer[inPacket->l++] = (u_char) ((convLgt+3)     & 0xff);
	for (i=0; i < convLgt; i++) {
		inPacket->buffer[inPacket->l++] = outOrigMsg[i];
	}
	
	return COBEX_OK;
}


int obex_hi_type (obex_packet *inPacket, char *msg, int lgt) {
	int i;

	if (inPacket->max < (inPacket->l+lgt+3) ) { return COBEX_ERR_BUFFER_OVERFLOW; }
	inPacket->buffer[inPacket->l++] = OBEX_HI_TYPE;
	inPacket->buffer[inPacket->l++] = (u_char)(((lgt+3)>>8) & 0xff);
	inPacket->buffer[inPacket->l++] = (u_char) ((lgt+3)     & 0xff);

	for (i=0; i < lgt; i++) {
		inPacket->buffer[inPacket->l++] = msg[i];
	}
	return COBEX_OK;
}

int obex_hi_target (obex_packet *inPacket, char *msg, int lgt) {
	int i;

	if (inPacket->max < (inPacket->l + (lgt*2) +3) ) { return COBEX_ERR_BUFFER_OVERFLOW; }
	inPacket->buffer[inPacket->l++] = OBEX_HI_TARGET;
	inPacket->buffer[inPacket->l++] = (u_char)(((lgt+3)>>8) & 0xff);
	inPacket->buffer[inPacket->l++] = (u_char) ((lgt+3)     & 0xff);

	for (i=0; i < lgt; i++) {
		inPacket->buffer[inPacket->l++] = msg[i];
	}
	return COBEX_OK;
}

// Length, however, is a 32 bit integer.

int obex_hi_length (obex_packet *inPacket, u_int length) {

	if (inPacket->max < (inPacket->l + 5) ) { return COBEX_ERR_BUFFER_OVERFLOW; }
	inPacket->buffer[inPacket->l++] = OBEX_HI_LENGTH;
	inPacket->buffer[inPacket->l++] = (char)((length >> 24) & 255);
	inPacket->buffer[inPacket->l++] = (char)((length >> 16) & 255);
	inPacket->buffer[inPacket->l++] = (char)((length >>  8) & 255);
	inPacket->buffer[inPacket->l++] = (char)( length        & 255);

	return COBEX_OK;
}

// ...and here, msg is a four byte array of chars.

int obex_hi_connection_id (obex_packet *inPacket, char *msg) {
	int i;

	if (inPacket->max < (inPacket->l + 5) ) { return COBEX_ERR_BUFFER_OVERFLOW; }
	inPacket->buffer[inPacket->l++] = OBEX_HI_CONNECTION_ID;

	for (i=0; i < 4; i++) {
		inPacket->buffer[inPacket->l++] = msg[i];
	}
	return COBEX_OK;
}

int obex_hi_body (obex_packet *inPacket, char *data, int dataLgt, int dataStart) {

	int freeBuffer = inPacket->max - inPacket->l;
	int consumed;
	int i;

	if (freeBuffer < 3) { return COBEX_ERR_BUFFER_OVERFLOW; }

	consumed = imin (freeBuffer-3,dataLgt-dataStart);

	inPacket->buffer[inPacket->l++] = OBEX_HI_BODY;
	inPacket->buffer[inPacket->l++] = (u_char)(((consumed+3)>>8) & 0xff);
	inPacket->buffer[inPacket->l++] = (u_char) ((consumed+3)     & 0xff);

	for (i=0; i<consumed; i++) {
		inPacket->buffer[inPacket->l++] = data[dataStart+i];
	}

	return consumed;
}

int obex_hi_endofbody (obex_packet *inPacket) {

	if (inPacket->max < (inPacket->l +3) ) { return COBEX_ERR_BUFFER_OVERFLOW; }

	inPacket->buffer[inPacket->l++] = OBEX_HI_END_OF_BODY;
	inPacket->buffer[inPacket->l++] = 0x00;
	inPacket->buffer[inPacket->l++] = 0x03;

	return COBEX_OK;
}

// When the packet is ready, run this to set the correct length

int cobex_packlgt (obex_packet *inPacket ) {

	if (inPacket->l < 3) { return COBEX_ERR; }
	inPacket->buffer[1] = (u_char)((inPacket->l >>8) & 0xff);
	inPacket->buffer[2] = (u_char) (inPacket->l      & 0xff);

	return COBEX_OK;
}

// Set "final bit" in a packet operation code.

int cobex_set_final_bit (obex_packet *inPacket) {
	if (inPacket->l < 3) { return COBEX_ERR; }
	inPacket->buffer[0] = inPacket->buffer[0] | OBEX_FINAL_BIT;
	return COBEX_OK;
}

//
// Functions related to parsing
//
// Return the responsecode

int cobex_response_code (obex_packet *inPacket) {
	return inPacket->l > 0 ? (int)(inPacket->buffer[0]) : COBEX_ERR ;
}
// Begin parsing, slightly more intelligent, and then braindead again.
//
// It is hard to determine where the headers start by looking at the responsecode.
// But by looking at the responsecode, and if the third byte is 0x10, which is not a
// legal HI value, we make an aducated guess.

int cobex_parseStart ( obex_packet *inPacket ) {

	if ( cobex_response_code(inPacket) == 0xA0 ) {
		if (inPacket->buffer[3] == OBEX_VERSION) {
			return 7;
		} else {
			return 3;
		}
	}

	switch (cobex_response_code(inPacket)) {
		case OBEX_OPCODE_CONNECT:
			return 7;
		case OBEX_OPCODE_SETPATH:
			return 5;
		default:
			return 3;
		}
	return 3;
}

// Parse next Header

parsedData cobex_parseNext ( obex_packet *inPacket, u_int p) {
	u_int hLen;
	
	if (p > inPacket->l) { return (parsedData){ 0x00, 0x00, NULL, p}; }
	switch (inPacket->buffer[p]&0xC0) {
		case 0x00:
		case 0x40:
			hLen = (inPacket->buffer[p+1]<<8) + inPacket->buffer[p+2];
			return (parsedData){ (inPacket->buffer[p]), hLen-3, 
				(inPacket->buffer)+p+3, p+hLen};
			break;
		case 0x80:
			return (parsedData){ (inPacket->buffer[p]), 1, 
				(inPacket->buffer)+p+1, p+2 };
			break;
		case 0xC0:
			return (parsedData){ (inPacket->buffer[p]), 4, 
				(inPacket->buffer)+p+1, p+5 };
			break;
	}
	return (parsedData){ 0x00, 0x00, NULL, p};
}


// 
// Error reporting functions
//
// Dump packet content on stdout

char *cobex_respstring ( int respno ) {
	
	static const char *respText[] = {
		"??? - Unknown error",		/* 0x00 */
		"100 - Continue",		/* 0x01 */
		"200 - Ok, Success",		/* 0x02 */
		"201 - Created",		/* 0x03 */
		"202 - Accepted",		/* 0x04 */
		"203 - Non-Authoritative Information", 	/* 0x05 */
		"204 - No Content",		/* 0x06 */
		"205 - Reset Content",		/* 0x07 */
		"206 - Partial Content",	/* 0x08 */
		"300 - Multiple Choices",	/* 0x09 */
		"301 - Moved Permanently",	/* 0x0A */
		"302 - Moved Temporarily",	/* 0x0B */
		"303 - See Other",		/* 0x0C */
		"304 - Not Modified",		/* 0x0D */
		"305 - Use Proxy",		/* 0x0E */
		"400 - Bad Request - server couldn't understand request",	/* 0x0F */
		"401 - Unauthorized",		/* 0x10 */
		"402 - Payment required",	/* 0x11 */
		"403 - Forbidden - operation is understood but refused",	/* 0x12 */
		"404 - Not Found",		/* 0x13 */
		"405 - Method not allowed",	/* 0x14 */
		"406 - Not Acceptable",		/* 0x15 */
		"407 - Proxy Authentication required",	/* 0x16 */
		"408 - Request Time Out",	/* 0x17 */
		"409 - Conflict",		/* 0x18 */
		"410 - Gone",			/* 0x19 */
		"411 - Length Required",	/* 0x1A */
		"412 - Precondition failed",	/* 0x1B */
		"413 - Requested entity too large",	/* 0x1C */
		"414 - Request URL too large",	/* 0x1D */
		"415 - Unsupported media type",	/* 0x1E */
		"500 - Internal server error",	/* 0x1F */
		"501 - Not Implemented",	/* 0x20 */
		"502 - Bad Gateway",		/* 0x21 */
		"503 - Service Unavailable",	/* 0x22 */
		"504 - Gateway Timeout",	/* 0x23 */
		"505 - HTTP version not supported",	/* 0x24 */
		"--- - Database Full",		/* 0x25 */
		"--- - Database Locked"		/* 0x26 */
	};
	
	static const int respLookup[] = {
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x0# */
		0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x1# */
		0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x00,  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x2# */
		0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x00, 0x00,  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x3# */
		0x0F, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16,  0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, /* 0x4# */
		0x1F, 0x20, 0x21, 0x22, 0x23, 0x24, 0x00, 0x00,  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x5# */
		0x05, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x6# */
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x7# */
				
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x8# */
		0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0x9# */
		0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x00,  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xA# */
		0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x00, 0x00,  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xB# */
		0x0F, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16,  0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, /* 0xC# */
		0x1F, 0x20, 0x21, 0x22, 0x23, 0x24, 0x00, 0x00,  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xD# */
		0x05, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xE# */
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 0xF# */
	};
	
	return respno > 0xFF ? NULL : (char*)respText[respLookup[respno]];
} 


// 
// Debug and MISC functions
//
// Dump packet content on stdout

void cobex_packet_dump ( obex_packet *inPacket ) {
	u_int i;

	for (i=0; i<inPacket->l; i++) {
		fprintf (stderr,"%2X ",inPacket->buffer[i]);
		if (i%16==15) { printf ("\n"); }
	}
}

// Don't mind me, I'm almost not here...

int imin ( int a, int b) {
	return (a<b) ? a : b;
}
