#!/bin/bash
# Creates several zones and charts from the ourairports.com data base.
# ILS are missing from that data base; some have been added using the
# tools/create-ils.exe program and then completed by hand for some selected
# major airport.

# Destination dir. of the creates sceneries:
ZONES_DIR=../../objects/zones

# Destination dir. of the created charts:
CHARTS_DIR=../../doc/charts

# WMM coefficients:
WMM_COF=../../objects/WMM.COF


# Creates a scenery for the given range of geographical coordinates.
function create_zone()
{
	local scenery title lat_min lat_max lon_min lon_max
	scenery=$1
	title=$2
	lat_min=$3
	lat_max=$4
	lon_min=$5
	lon_max=$6
	echo "Generating zone $dst ..."
	{
		echo "# $title."
		echo "# Created: " $(date +%Y-%m-%d)
		echo "# Latitude range: [$lat_min, $lat_max["
		echo "# Longitude range: [$lon_min, $lon_max["
		echo "# 2017-11-01 Source: http://ourairports.com"
		echo
		echo "GROUND_COLOR #305030"
		echo
		./ourairports_com.exe \
			--lat-min $lat_min \
			--lat-max $lat_max \
			--lon-min $lon_min \
			--lon-max $lon_max \
			--src-dir ../../../ourairports.com
	} > $scenery
}

# Generates PDF of a navigation chart.
# Parameters:
# $1: scenery file.
# $2: destination PDF file name without extension.
# $3: title.
# $4: central latitude.
# $5: central longitude.
# $6: scale.
# $7, $8, $9: optional further parameters for the chart.exe program.
function doChart
{
	local dst
	dst=$2
	echo "Generating chart $dst.pdf ..."
	../chart/chart.exe --wmm-cof $WMM_COF \
		--scene "$1" \
		--title "$3" \
		--olat $4 \
		--olon $5 \
		--scale $6 \
		$7 $8 $9 > $dst.ps || exit 1
	ps2pdf -sPAPERSIZE=a4 $dst.ps $dst.pdf || exit 1
	rm $dst.ps
}


function france()
{
	locale scenery dst
	scenery=$ZONES_DIR/europe/france.txt
	create_zone $scenery "France zone"  42N 50N 005W 006E
	{

		echo "# Paris, Charles De Gaulle (LFPG):"
		echo "ILS 08L ILS     GLE 108.70 48-59-56.47N 002-36-51.36E 48-59-41.31N 002-33-09.34E  387 5  85.3 3"
		echo "ILS 26R ILS     GAU 109.10 48-59-43.72N 002-32-55.15E 48-59-58.89N 002-36-37.18E  397 5 265.3 3"
		echo "ILS 08R ILS/DME DSE 108.55 48-59-42.44N 002-36-23.46E 48-59-31.23N 002-33-55.86E  387 5  85.3 3"
		echo "ILS 26L ILS/DME DSU 108.35 48-59-33.64N 002-33-41.67E 48-59-44.85N 002-36-09.28E  397 5 265.3 3"
		echo "ILS 09L ILS/DME PNE 109.35 49-01-36.92N 002-33-56.80E 49-01-25.71N 002-31-29.08E  387 5  85.3 3"
		echo "ILS 27R ILS/DME PNW 110.35 49-01-28.12N 002-31-14.88E 49-01-39.33N 002-33-42.61E  397 5 265.3 3"
		echo "ILS 09R ILS/DME CGE 110.10 49-01-26.11N 002-34-27.76E 49-01-10.95N 002-30-46.50E  387 5  85.3 3"
		echo "ILS 27L ILS/DME PNW 110.35 49-01-13.36N 002-30-32.31E 49-01-28.53N 002-34-13.57E  397 5 265.3 3"

		echo "# Paris, Le Bourget (LFPB):"
		echo "ILS 07 ILS/DME LBG 109.50 48-58-30.07N 002-27-43.36E 48-57-46.72N 002-25-11.45E  193 5  67.8 3"
		echo "ILS 25 ILS     LBW 111.10 48-57-46.01N 002-24-59.39E 48-58-29.35N 002-27-31.30E  243 5 247.8 3"
		echo "ILS 27 ILS/DME RGE 110.55 48-57-48.50N 002-24-58.74E 48-57-57.57N 002-26-44.79E  224 5 265.2 3"

		echo "# Paris, Orly (LFPO):"
		echo "ILS 02 ILS     OLN 110.30 48-44-26.02N 002-23-17.71E 48-43-02.08N 002-22-32.57E  228 5  18.3 3"
		echo "ILS 06 ILS/DME ORE 108.50 48-44-12.38N 002-21-51.40E 48-43-09.19N 002-18-59.00E  260 5  61.9 3"
		echo "ILS 24 ILS/DME OLO 110.90 48-43-07.41N 002-18-47.97E 48-44-10.60N 002-21-40.36E  322 5 241.8 3"
		echo "ILS 08 LOCALIZER OLE 108.15 48-43-41.24N 002-24-21.59E 48-43-06.75N 002-21-29.74E  273 5  74.5 3"
		echo "ILS 26 ILS/DME OLW 111.75 48-43-07.24N 002-21-16.78E 48-43-41.73N 002-24-08.63E  309 5 254.4 3"

	} >> $scenery
	local src dst
	src=$ZONES_DIR/europe/france.txt
	dst=$CHARTS_DIR/europe/france
	# Navigation charts:
	doChart $src $dst/france-nw "France North-West area" 49N 002-30E 500000 --no-ils
	doChart $src $dst/france-ne "France North-East area" 48N 004-30E 1500000 --no-ils
	doChart $src $dst/france-sw "France South-West area" 44N 004E 1500000 --no-ils
	doChart $src $dst/france-se "France South-East area" 44N 005E 1000000 --no-ils
	doChart $src $dst/paris Paris/France 48-55N 002-20E 250000
	doChart $src $dst/degaulle "Paris, De Gaulle" 49-01N 002-34E 100000
	doChart $src $dst/lebourget "Paris, Le Bourget" 49-00N 002-28E 100000
}


function germany()
{
	create_zone $ZONES_DIR/europe/germany.txt "Germany zone"  47N 55N 006E 015E
	{
		echo "# Dusseldorf (EDDL):"
		echo "ILS 05L ILS/DME IDNE 110.95 51-18-00.10N 006-46-59.08E 51-16-58.94N 006-44-52.49E  109 5  52.9 3"
		echo "ILS 23R ILS/DME IDNW 109.30 51-16-55.45N 006-44-43.06E 51-17-56.62N 006-46-49.64E  185 5 232.8 3"
		echo "ILS 05R ILS/DME IDSE 111.50 51-17-51.11N 006-47-22.72E 51-16-44.18N 006-45-04.26E  109 5  52.8 3"
		echo "ILS 23L ILS/DME IDSW 109.90 51-16-40.69N 006-44-54.84E 51-17-47.62N 006-47-13.29E  185 5 232.8 3"

		echo "# Frankfurt (EDDF):"
		echo "ILS 07C ILS/DME IFCE 110.55 50-02-45.73N 008-35-27.27E 50-01-54.49N 008-32-02.65E 342 5  69.7 3"
		echo "ILS 25C ILS/DME IFCW 111.55 50-01-53.98N 008-31-50.54E 50-02-45.23N 008-35-15.16E 386 5 249.6 3"
		echo "ILS 07L ILS/DME IFNE 111.75 50-02-48.24N 008-32-15.54E 50-02-10.68N 008-29-47.48E 342 5  69.8 3"
		echo "ILS 25R ILS/DME IFNW 111.35 50-02-10.20N 008-29-35.35E 50-02-47.76N 008-32-03.41E 386 5 249.7 3"
		echo "ILS 07R ILS/DME IFSE 110.95 50-02-27.75N 008-35-25.63E 50-01-36.13N 008-32-00.98E 342 5  69.5 3"
		echo "ILS 25L ILS/DME IFSW 111.15 50-01-35.60N 008-31-48.89E 50-02-27.23N 008-35-13.55E 387 5 249.5 3"

		echo "# Munich (EDDM):"
		echo "ILS 08L ILS/DME IMNE 109.50 48-22-01.94N 011-49-30.80E 48-21-42.99N 011-46-01.97E 1479 5  83.5 3"
		echo "ILS 26R ILS/DME IMNW 108.70 48-21-44.97N 011-45-48.88E 48-22-03.93N 011-49-17.71E 1495 5 263.4 3"
		echo "ILS 08R ILS/DME IMSE 109.30 48-20-42.38N 011-48-31.03E 48-20-23.43N 011-45-02.21E 1479 5  83.5 3"
		echo "ILS 26L ILS/DME IMSW 108.30 48-20-25.41N 011-44-49.13E 48-20-44.37N 011-48-17.95E 1495 5 263.4 3"

	} >> $ZONES_DIR/europe/germany.txt

	local src dst
	src=$ZONES_DIR/europe/germany.txt
	dst=$CHARTS_DIR/europe/germany
	# Navigation charts:
	doChart $src $dst/frankfurt "Frankfurt area" 50-02N 008-31E 250000
	doChart $src $dst/munich "Munich area" 48-20N 011-50E 250000
	doChart $src $dst/dusseldorf "Dusseldorf area" 51-17N 006-44E 250000
}


function spain()
{
	locale zone
	zone=$ZONES_DIR/europe/spain.txt
	create_zone $zone "Spain zone"  35N 42N 010W 006E

	{
		# LEMD:
		echo "ILS 32R ILS/DME MBB 109.10 40-29-49.32N 003-33-36.14W 40-28-13.88N 003-31-59.62W 1938 5 322.2 3"
		echo "ILS 32L ILS/DME MAA 109.90 40-29-13.32N 003-34-41.44W 40-27-22.40N 003-32-49.30W 1938 5 322.2 3"
		echo "ILS 18L ILS/DME IML 111.50 40-29-54.24N 003-33-33.11W 40-31-57.48N 003-33-31.01W 2074 5 179.8 3"
		echo "ILS 18R ILS/DME IMR 110.70 40-29-23.64N 003-34-28.62W 40-31-54.60N 003-34-26.71W 2074 5 179.8 3"

		# LEBL:
		echo "ILS 02  ILS/DME BLT 108.75 41-18-43.03N 002-05-45.16E 41-17-15.10N 002-05-02.69E   59 5  19.0 3"
		echo "ILS 07L ILS/DME QAA 110.30 41-18-24.53N 002-06-25.18E 41-17-30.08N 002-03-53.09E   19 5  65.6 3"
		echo "ILS 25R ILS/DME BCA 109.50 41-17-28.98N 002-03-42.66E 41-18-23.44N 002-06-14.74E   43 5 245.6 3"
		echo "ILS 07R ILS/DME BLE 110.75 41-17-35.94N 002-06-23.55E 41-16-53.36N 002-04-26.35E   19 5  65.6 3"
		echo "ILS 25L ILS/DME BLW 111.50 41-16-52.26N 002-04-15.92E 41-17-34.84N 002-06-13.12E   43 5 245.6 3"

		# LEPA:
		echo "ILS 06L ILS/DME PLM 110.90 39-33-50.08N 002-44-46.20E 39-32-46.85N 002-42-37.07E   13 5  58.5 3"
		echo "ILS 24R ILS     PAA 109.90 39-32-44.48N 002-42-27.92E 39-33-47.71N 002-44-37.04E   67 5 238.5 3"
		echo "ILS 24L ILS/DME IPAL 109.30 39-32-23.63N 002-43-43.52E 39-33-21.80N 002-45-43.06E   67 5 238.7 3"

		# LEMG:
		echo "ILS 13 ILS/DME GMM 109.50 36-39-48.99N 004-28-59.91W 36-41-01.88N 004-30-43.49W  106 5 131.5 3"
		echo "ILS 31 ILS/DME GAA 109.90 36-41-10.65N 004-30-54.37W 36-39-57.76N 004-29-10.77W    0 5 311.5 3"

	} >> $zone

	# Navigation charts:
	local dst
	dst=$CHARTS_DIR/europe/spain
	doChart $zone $dst/madrid-barajas "Madrid, Barajas (LEMD)" 40-29N 003-33W 250000
	doChart $zone $dst/barcelona-elprat "Barcelona, El Prat (LEBL)" 41-17N 002-05E 250000
	doChart $zone $dst/palmademallorca "Palma de Mallorca (LEPA)" 39-33N 002-44E 250000
	doChart $zone $dst/malaga "Malaga (LEMG)" 36-40N 004-30W 250000
}


function united_kingdom()
{
	create_zone $ZONES_DIR/europe/united-kingdom.txt "United Kingdom zone"  50N 60N 010W 002E

	{
		echo "# Heathrow (EGLL):"
		echo "ILS 09L ILS/DME IAA 110.30 51-28-39.77N 000-25-44.21W 51-28-35.76N 000-29-21.92W   83 5  89.7 3"
		echo "ILS 27R ILS/DME IRR 110.30 51-28-38.94N 000-29-37.48W 51-28-42.96N 000-25-59.76W   83 5 269.7 3"
		echo "ILS 09R ILS/DME IBB 109.50 51-27-54.03N 000-25-47.13W 51-27-50.40N 000-29-12.35W   83 5  89.8 3"
		echo "ILS 27L ILS/DME ILL 109.50 51-27-53.61N 000-29-27.92W 51-27-57.24N 000-26-02.70W   83 5 269.8 3"

		echo "# Gatwick (EGKK):"
		echo "ILS 08R ILS IGG 110.90 51-09-06.40N 000-09-50.41W 51-08-39.20N 000-12-45.17W  189 5  77.6 3"
		echo "ILS 26L ILS IWW 110.90 51-08-40.27N 000-12-59.40W 51-09-07.48N 000-10-04.64W  215 5 257.6 3"

		echo "# Manchester (EGCC):"
		echo "ILS 05L ILS/DME IMM 108.50 53-21-50.76N 002-15-13.11W 53-20-39.78N 002-17-36.42W  219 5  50.9 3"
		echo "ILS 23R ILS/DME INN 109.50 53-20-36.24N 002-17-46.44W 53-21-47.21N 002-15-23.13W  295 5 230.9 3"
		echo "ILS 05R ILS/DME IMC 111.55 53-21-02.82N 002-16-17.30W 53-19-52.61N 002-18-40.92W  220 5  51.3 3"

		echo "# Stansted (EGSS):"
		echo "ILS 04 ILS ISED 110.50 51-53-49.84N 000-15-10.83E 51-52-28.17N 000-13-08.70E  303 5  42.9 3"
		echo "ILS 22 ILS ISX  110.50 51-52-23.24N 000-13-01.07E 51-53-44.91N 000-15-03.19E  393 5 222.8 3"

	} >> $ZONES_DIR/europe/united-kingdom.txt

	local src dst
	src=$ZONES_DIR/europe/united-kingdom.txt
	dst=$CHARTS_DIR/europe/united-kingdom
	# Navigation charts:
	doChart $src $dst/Heathrow London/Heathrow 51-28N 000-28W 250000
	doChart $src $dst/Gatwick  London/Gatwick  51-08N 000-12W 250000
	doChart $src $dst/Stansted London/Stansted 51-53N 000-15E 250000
	doChart $src $dst/Manchester London/Manchester 53-20N 002-20W 250000
}


france
germany
spain
united_kingdom
